













#![doc(html_root_url = "https://docs.rs/either/1/")]
#![no_std]

#[cfg(any(test, feature = "std"))]
extern crate std;

#[cfg(feature = "serde")]
pub mod serde_untagged;

#[cfg(feature = "serde")]
pub mod serde_untagged_optional;

use core::convert::{AsMut, AsRef};
use core::fmt;
use core::future::Future;
use core::ops::Deref;
use core::ops::DerefMut;
use core::pin::Pin;

#[cfg(any(test, feature = "std"))]
use std::error::Error;
#[cfg(any(test, feature = "std"))]
use std::io::{self, BufRead, Read, Seek, SeekFrom, Write};

pub use crate::Either::{Left, Right};







#[cfg_attr(feature = "serde", derive(serde::Serialize, serde::Deserialize))]
#[derive(Copy, PartialEq, Eq, PartialOrd, Ord, Hash, Debug)]
pub enum Either<L, R> {

    Left(L),

    Right(R),
}

























#[macro_export]
macro_rules! for_both {
    ($value:expr, $pattern:pat => $result:expr) => {
        match $value {
            $crate::Either::Left($pattern) => $result,
            $crate::Either::Right($pattern) => $result,
        }
    };
}























#[macro_export]
macro_rules! try_left {
    ($expr:expr) => {
        match $expr {
            $crate::Left(val) => val,
            $crate::Right(err) => return $crate::Right(::core::convert::From::from(err)),
        }
    };
}


#[macro_export]
macro_rules! try_right {
    ($expr:expr) => {
        match $expr {
            $crate::Left(err) => return $crate::Left(::core::convert::From::from(err)),
            $crate::Right(val) => val,
        }
    };
}

macro_rules! map_either {
    ($value:expr, $pattern:pat => $result:expr) => {
        match $value {
            Left($pattern) => Left($result),
            Right($pattern) => Right($result),
        }
    };
}

mod iterator;
pub use self::iterator::IterEither;

mod into_either;
pub use self::into_either::IntoEither;

impl<L: Clone, R: Clone> Clone for Either<L, R> {
    fn clone(&self) -> Self {
        match self {
            Left(inner) => Left(inner.clone()),
            Right(inner) => Right(inner.clone()),
        }
    }

    fn clone_from(&mut self, source: &Self) {
        match (self, source) {
            (Left(dest), Left(source)) => dest.clone_from(source),
            (Right(dest), Right(source)) => dest.clone_from(source),
            (dest, source) => *dest = source.clone(),
        }
    }
}

impl<L, R> Either<L, R> {









    pub fn is_left(&self) -> bool {
        match self {
            Left(_) => true,
            Right(_) => false,
        }
    }










    pub fn is_right(&self) -> bool {
        !self.is_left()
    }












    pub fn left(self) -> Option<L> {
        match self {
            Left(l) => Some(l),
            Right(_) => None,
        }
    }












    pub fn right(self) -> Option<R> {
        match self {
            Left(_) => None,
            Right(r) => Some(r),
        }
    }












    pub fn as_ref(&self) -> Either<&L, &R> {
        map_either!(self, inner => inner)
    }



















    pub fn as_mut(&mut self) -> Either<&mut L, &mut R> {
        map_either!(self, inner => inner)
    }



    pub fn as_pin_ref(self: Pin<&Self>) -> Either<Pin<&L>, Pin<&R>> {


        unsafe { map_either!(Pin::get_ref(self), inner => Pin::new_unchecked(inner)) }
    }



    pub fn as_pin_mut(self: Pin<&mut Self>) -> Either<Pin<&mut L>, Pin<&mut R>> {





        unsafe { map_either!(Pin::get_unchecked_mut(self), inner => Pin::new_unchecked(inner)) }
    }












    pub fn flip(self) -> Either<R, L> {
        match self {
            Left(l) => Right(l),
            Right(r) => Left(r),
        }
    }













    pub fn map_left<F, M>(self, f: F) -> Either<M, R>
    where
        F: FnOnce(L) -> M,
    {
        match self {
            Left(l) => Left(f(l)),
            Right(r) => Right(r),
        }
    }













    pub fn map_right<F, S>(self, f: F) -> Either<L, S>
    where
        F: FnOnce(R) -> S,
    {
        match self {
            Left(l) => Left(l),
            Right(r) => Right(f(r)),
        }
    }


















    pub fn map_either<F, G, M, S>(self, f: F, g: G) -> Either<M, S>
    where
        F: FnOnce(L) -> M,
        G: FnOnce(R) -> S,
    {
        match self {
            Left(l) => Left(f(l)),
            Right(r) => Right(g(r)),
        }
    }





















    pub fn map_either_with<Ctx, F, G, M, S>(self, ctx: Ctx, f: F, g: G) -> Either<M, S>
    where
        F: FnOnce(Ctx, L) -> M,
        G: FnOnce(Ctx, R) -> S,
    {
        match self {
            Left(l) => Left(f(ctx, l)),
            Right(r) => Right(g(ctx, r)),
        }
    }

















    pub fn either<F, G, T>(self, f: F, g: G) -> T
    where
        F: FnOnce(L) -> T,
        G: FnOnce(R) -> T,
    {
        match self {
            Left(l) => f(l),
            Right(r) => g(r),
        }
    }




















    pub fn either_with<Ctx, F, G, T>(self, ctx: Ctx, f: F, g: G) -> T
    where
        F: FnOnce(Ctx, L) -> T,
        G: FnOnce(Ctx, R) -> T,
    {
        match self {
            Left(l) => f(ctx, l),
            Right(r) => g(ctx, r),
        }
    }












    pub fn left_and_then<F, S>(self, f: F) -> Either<S, R>
    where
        F: FnOnce(L) -> Either<S, R>,
    {
        match self {
            Left(l) => f(l),
            Right(r) => Right(r),
        }
    }












    pub fn right_and_then<F, S>(self, f: F) -> Either<L, S>
    where
        F: FnOnce(R) -> Either<L, S>,
    {
        match self {
            Left(l) => Left(l),
            Right(r) => f(r),
        }
    }














    #[allow(clippy::should_implement_trait)]
    pub fn into_iter(self) -> Either<L::IntoIter, R::IntoIter>
    where
        L: IntoIterator,
        R: IntoIterator<Item = L::Item>,
    {
        map_either!(self, inner => inner.into_iter())
    }
















    pub fn iter(&self) -> Either<<&L as IntoIterator>::IntoIter, <&R as IntoIterator>::IntoIter>
    where
        for<'a> &'a L: IntoIterator,
        for<'a> &'a R: IntoIterator<Item = <&'a L as IntoIterator>::Item>,
    {
        map_either!(self, inner => inner.into_iter())
    }






















    pub fn iter_mut(
        &mut self,
    ) -> Either<<&mut L as IntoIterator>::IntoIter, <&mut R as IntoIterator>::IntoIter>
    where
        for<'a> &'a mut L: IntoIterator,
        for<'a> &'a mut R: IntoIterator<Item = <&'a mut L as IntoIterator>::Item>,
    {
        map_either!(self, inner => inner.into_iter())
    }

















    pub fn factor_into_iter(self) -> IterEither<L::IntoIter, R::IntoIter>
    where
        L: IntoIterator,
        R: IntoIterator,
    {
        IterEither::new(map_either!(self, inner => inner.into_iter()))
    }















    pub fn factor_iter(
        &self,
    ) -> IterEither<<&L as IntoIterator>::IntoIter, <&R as IntoIterator>::IntoIter>
    where
        for<'a> &'a L: IntoIterator,
        for<'a> &'a R: IntoIterator,
    {
        IterEither::new(map_either!(self, inner => inner.into_iter()))
    }

















    pub fn factor_iter_mut(
        &mut self,
    ) -> IterEither<<&mut L as IntoIterator>::IntoIter, <&mut R as IntoIterator>::IntoIter>
    where
        for<'a> &'a mut L: IntoIterator,
        for<'a> &'a mut R: IntoIterator,
    {
        IterEither::new(map_either!(self, inner => inner.into_iter()))
    }

















    pub fn left_or(self, other: L) -> L {
        match self {
            Either::Left(l) => l,
            Either::Right(_) => other,
        }
    }













    pub fn left_or_default(self) -> L
    where
        L: Default,
    {
        match self {
            Either::Left(l) => l,
            Either::Right(_) => L::default(),
        }
    }













    pub fn left_or_else<F>(self, f: F) -> L
    where
        F: FnOnce(R) -> L,
    {
        match self {
            Either::Left(l) => l,
            Either::Right(r) => f(r),
        }
    }

















    pub fn right_or(self, other: R) -> R {
        match self {
            Either::Left(_) => other,
            Either::Right(r) => r,
        }
    }













    pub fn right_or_default(self) -> R
    where
        R: Default,
    {
        match self {
            Either::Left(_) => R::default(),
            Either::Right(r) => r,
        }
    }













    pub fn right_or_else<F>(self, f: F) -> R
    where
        F: FnOnce(L) -> R,
    {
        match self {
            Either::Left(l) => f(l),
            Either::Right(r) => r,
        }
    }




















    pub fn unwrap_left(self) -> L
    where
        R: core::fmt::Debug,
    {
        match self {
            Either::Left(l) => l,
            Either::Right(r) => {
                panic!("called `Either::unwrap_left()` on a `Right` value: {:?}", r)
            }
        }
    }




















    pub fn unwrap_right(self) -> R
    where
        L: core::fmt::Debug,
    {
        match self {
            Either::Right(r) => r,
            Either::Left(l) => panic!("called `Either::unwrap_right()` on a `Left` value: {:?}", l),
        }
    }




















    pub fn expect_left(self, msg: &str) -> L
    where
        R: core::fmt::Debug,
    {
        match self {
            Either::Left(l) => l,
            Either::Right(r) => panic!("{}: {:?}", msg, r),
        }
    }




















    pub fn expect_right(self, msg: &str) -> R
    where
        L: core::fmt::Debug,
    {
        match self {
            Either::Right(r) => r,
            Either::Left(l) => panic!("{}: {:?}", msg, l),
        }
    }













    pub fn either_into<T>(self) -> T
    where
        L: Into<T>,
        R: Into<T>,
    {
        for_both!(self, inner => inner.into())
    }
}

impl<L, R> Either<Option<L>, Option<R>> {












    pub fn factor_none(self) -> Option<Either<L, R>> {
        match self {
            Left(l) => l.map(Either::Left),
            Right(r) => r.map(Either::Right),
        }
    }
}

impl<L, R, E> Either<Result<L, E>, Result<R, E>> {














    pub fn factor_err(self) -> Result<Either<L, R>, E> {
        match self {
            Left(l) => l.map(Either::Left),
            Right(r) => r.map(Either::Right),
        }
    }
}

impl<T, L, R> Either<Result<T, L>, Result<T, R>> {














    pub fn factor_ok(self) -> Result<T, Either<L, R>> {
        match self {
            Left(l) => l.map_err(Either::Left),
            Right(r) => r.map_err(Either::Right),
        }
    }
}

impl<T, L, R> Either<(T, L), (T, R)> {












    pub fn factor_first(self) -> (T, Either<L, R>) {
        match self {
            Left((t, l)) => (t, Left(l)),
            Right((t, r)) => (t, Right(r)),
        }
    }
}

impl<T, L, R> Either<(L, T), (R, T)> {












    pub fn factor_second(self) -> (Either<L, R>, T) {
        match self {
            Left((l, t)) => (Left(l), t),
            Right((r, t)) => (Right(r), t),
        }
    }
}

impl<T> Either<T, T> {











    pub fn into_inner(self) -> T {
        for_both!(self, inner => inner)
    }












    pub fn map<F, M>(self, f: F) -> Either<M, M>
    where
        F: FnOnce(T) -> M,
    {
        match self {
            Left(l) => Left(f(l)),
            Right(r) => Right(f(r)),
        }
    }
}

impl<L, R> Either<&L, &R> {


    pub fn cloned(self) -> Either<L, R>
    where
        L: Clone,
        R: Clone,
    {
        map_either!(self, inner => inner.clone())
    }



    pub fn copied(self) -> Either<L, R>
    where
        L: Copy,
        R: Copy,
    {
        map_either!(self, inner => *inner)
    }
}

impl<L, R> Either<&mut L, &mut R> {


    pub fn cloned(self) -> Either<L, R>
    where
        L: Clone,
        R: Clone,
    {
        map_either!(self, inner => inner.clone())
    }



    pub fn copied(self) -> Either<L, R>
    where
        L: Copy,
        R: Copy,
    {
        map_either!(self, inner => *inner)
    }
}


impl<L, R> From<Result<R, L>> for Either<L, R> {
    fn from(r: Result<R, L>) -> Self {
        match r {
            Err(e) => Left(e),
            Ok(o) => Right(o),
        }
    }
}


impl<L, R> From<Either<L, R>> for Result<R, L> {
    fn from(val: Either<L, R>) -> Self {
        match val {
            Left(l) => Err(l),
            Right(r) => Ok(r),
        }
    }
}


impl<L, R> Future for Either<L, R>
where
    L: Future,
    R: Future<Output = L::Output>,
{
    type Output = L::Output;

    fn poll(
        self: Pin<&mut Self>,
        cx: &mut core::task::Context<'_>,
    ) -> core::task::Poll<Self::Output> {
        for_both!(self.as_pin_mut(), inner => inner.poll(cx))
    }
}

#[cfg(any(test, feature = "std"))]



impl<L, R> Read for Either<L, R>
where
    L: Read,
    R: Read,
{
    fn read(&mut self, buf: &mut [u8]) -> io::Result<usize> {
        for_both!(self, inner => inner.read(buf))
    }

    fn read_exact(&mut self, buf: &mut [u8]) -> io::Result<()> {
        for_both!(self, inner => inner.read_exact(buf))
    }

    fn read_to_end(&mut self, buf: &mut std::vec::Vec<u8>) -> io::Result<usize> {
        for_both!(self, inner => inner.read_to_end(buf))
    }

    fn read_to_string(&mut self, buf: &mut std::string::String) -> io::Result<usize> {
        for_both!(self, inner => inner.read_to_string(buf))
    }
}

#[cfg(any(test, feature = "std"))]



impl<L, R> Seek for Either<L, R>
where
    L: Seek,
    R: Seek,
{
    fn seek(&mut self, pos: SeekFrom) -> io::Result<u64> {
        for_both!(self, inner => inner.seek(pos))
    }
}

#[cfg(any(test, feature = "std"))]

impl<L, R> BufRead for Either<L, R>
where
    L: BufRead,
    R: BufRead,
{
    fn fill_buf(&mut self) -> io::Result<&[u8]> {
        for_both!(self, inner => inner.fill_buf())
    }

    fn consume(&mut self, amt: usize) {
        for_both!(self, inner => inner.consume(amt))
    }

    fn read_until(&mut self, byte: u8, buf: &mut std::vec::Vec<u8>) -> io::Result<usize> {
        for_both!(self, inner => inner.read_until(byte, buf))
    }

    fn read_line(&mut self, buf: &mut std::string::String) -> io::Result<usize> {
        for_both!(self, inner => inner.read_line(buf))
    }
}

#[cfg(any(test, feature = "std"))]



impl<L, R> Write for Either<L, R>
where
    L: Write,
    R: Write,
{
    fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
        for_both!(self, inner => inner.write(buf))
    }

    fn write_all(&mut self, buf: &[u8]) -> io::Result<()> {
        for_both!(self, inner => inner.write_all(buf))
    }

    fn write_fmt(&mut self, fmt: fmt::Arguments<'_>) -> io::Result<()> {
        for_both!(self, inner => inner.write_fmt(fmt))
    }

    fn flush(&mut self) -> io::Result<()> {
        for_both!(self, inner => inner.flush())
    }
}

impl<L, R, Target> AsRef<Target> for Either<L, R>
where
    L: AsRef<Target>,
    R: AsRef<Target>,
{
    fn as_ref(&self) -> &Target {
        for_both!(self, inner => inner.as_ref())
    }
}

macro_rules! impl_specific_ref_and_mut {
    ($t:ty, $($attr:meta),* ) => {
        $(#[$attr])*
        impl<L, R> AsRef<$t> for Either<L, R>
            where L: AsRef<$t>, R: AsRef<$t>
        {
            fn as_ref(&self) -> &$t {
                for_both!(self, inner => inner.as_ref())
            }
        }

        $(#[$attr])*
        impl<L, R> AsMut<$t> for Either<L, R>
            where L: AsMut<$t>, R: AsMut<$t>
        {
            fn as_mut(&mut self) -> &mut $t {
                for_both!(self, inner => inner.as_mut())
            }
        }
    };
}

impl_specific_ref_and_mut!(str,);
impl_specific_ref_and_mut!(
    ::std::path::Path,
    cfg(feature = "std"),
    doc = "Requires crate feature `std`."
);
impl_specific_ref_and_mut!(
    ::std::ffi::OsStr,
    cfg(feature = "std"),
    doc = "Requires crate feature `std`."
);
impl_specific_ref_and_mut!(
    ::std::ffi::CStr,
    cfg(feature = "std"),
    doc = "Requires crate feature `std`."
);

impl<L, R, Target> AsRef<[Target]> for Either<L, R>
where
    L: AsRef<[Target]>,
    R: AsRef<[Target]>,
{
    fn as_ref(&self) -> &[Target] {
        for_both!(self, inner => inner.as_ref())
    }
}

impl<L, R, Target> AsMut<Target> for Either<L, R>
where
    L: AsMut<Target>,
    R: AsMut<Target>,
{
    fn as_mut(&mut self) -> &mut Target {
        for_both!(self, inner => inner.as_mut())
    }
}

impl<L, R, Target> AsMut<[Target]> for Either<L, R>
where
    L: AsMut<[Target]>,
    R: AsMut<[Target]>,
{
    fn as_mut(&mut self) -> &mut [Target] {
        for_both!(self, inner => inner.as_mut())
    }
}

impl<L, R> Deref for Either<L, R>
where
    L: Deref,
    R: Deref<Target = L::Target>,
{
    type Target = L::Target;

    fn deref(&self) -> &Self::Target {
        for_both!(self, inner => &**inner)
    }
}

impl<L, R> DerefMut for Either<L, R>
where
    L: DerefMut,
    R: DerefMut<Target = L::Target>,
{
    fn deref_mut(&mut self) -> &mut Self::Target {
        for_both!(self, inner => &mut *inner)
    }
}

#[cfg(any(test, feature = "std"))]



impl<L, R> Error for Either<L, R>
where
    L: Error,
    R: Error,
{
    fn source(&self) -> Option<&(dyn Error + 'static)> {
        for_both!(self, inner => inner.source())
    }

    #[allow(deprecated)]
    fn description(&self) -> &str {
        for_both!(self, inner => inner.description())
    }

    #[allow(deprecated)]
    fn cause(&self) -> Option<&dyn Error> {
        for_both!(self, inner => inner.cause())
    }
}

impl<L, R> fmt::Display for Either<L, R>
where
    L: fmt::Display,
    R: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        for_both!(self, inner => inner.fmt(f))
    }
}

impl<L, R> fmt::Write for Either<L, R>
where
    L: fmt::Write,
    R: fmt::Write,
{
    fn write_str(&mut self, s: &str) -> fmt::Result {
        for_both!(self, inner => inner.write_str(s))
    }

    fn write_char(&mut self, c: char) -> fmt::Result {
        for_both!(self, inner => inner.write_char(c))
    }

    fn write_fmt(&mut self, args: fmt::Arguments<'_>) -> fmt::Result {
        for_both!(self, inner => inner.write_fmt(args))
    }
}

#[test]
fn basic() {
    let mut e = Left(2);
    let r = Right(2);
    assert_eq!(e, Left(2));
    e = r;
    assert_eq!(e, Right(2));
    assert_eq!(e.left(), None);
    assert_eq!(e.right(), Some(2));
    assert_eq!(e.as_ref().right(), Some(&2));
    assert_eq!(e.as_mut().right(), Some(&mut 2));
}

#[test]
fn macros() {
    use std::string::String;

    fn a() -> Either<u32, u32> {
        let x: u32 = try_left!(Right(1337u32));
        Left(x * 2)
    }
    assert_eq!(a(), Right(1337));

    fn b() -> Either<String, &'static str> {
        Right(try_right!(Left("foo bar")))
    }
    assert_eq!(b(), Left(String::from("foo bar")));
}

#[test]
fn deref() {
    use std::string::String;

    fn is_str(_: &str) {}
    let value: Either<String, &str> = Left(String::from("test"));
    is_str(&value);
}

#[test]
fn iter() {
    let x = 3;
    let mut iter = match x {
        3 => Left(0..10),
        _ => Right(17..),
    };

    assert_eq!(iter.next(), Some(0));
    assert_eq!(iter.count(), 9);
}

#[test]
fn seek() {
    use std::io;

    let use_empty = false;
    let mut mockdata = [0x00; 256];
    for (i, data) in mockdata.iter_mut().enumerate() {
        *data = i as u8;
    }

    let mut reader = if use_empty {

        Left(io::Cursor::new([]))
    } else {
        Right(io::Cursor::new(&mockdata[..]))
    };

    let mut buf = [0u8; 16];
    assert_eq!(reader.read(&mut buf).unwrap(), buf.len());
    assert_eq!(buf, mockdata[..buf.len()]);


    assert_eq!(reader.read(&mut buf).unwrap(), buf.len());
    assert_ne!(buf, mockdata[..buf.len()]);


    reader.seek(io::SeekFrom::Start(0)).unwrap();
    assert_eq!(reader.read(&mut buf).unwrap(), buf.len());
    assert_eq!(buf, mockdata[..buf.len()]);
}

#[test]
fn read_write() {
    use std::io;

    let use_stdio = false;
    let mockdata = [0xff; 256];

    let mut reader = if use_stdio {
        Left(io::stdin())
    } else {
        Right(&mockdata[..])
    };

    let mut buf = [0u8; 16];
    assert_eq!(reader.read(&mut buf).unwrap(), buf.len());
    assert_eq!(&buf, &mockdata[..buf.len()]);

    let mut mockbuf = [0u8; 256];
    let mut writer = if use_stdio {
        Left(io::stdout())
    } else {
        Right(&mut mockbuf[..])
    };

    let buf = [1u8; 16];
    assert_eq!(writer.write(&buf).unwrap(), buf.len());
}

#[test]
fn error() {
    let invalid_utf8 = b"\xff";
    #[allow(invalid_from_utf8)]
    let res = if let Err(error) = ::std::str::from_utf8(invalid_utf8) {
        Err(Left(error))
    } else if let Err(error) = "x".parse::<i32>() {
        Err(Right(error))
    } else {
        Ok(())
    };
    assert!(res.is_err());
    #[allow(deprecated)]
    res.unwrap_err().description(); // make sure this can be called
}


macro_rules! check_t {
    ($t:ty) => {{
        fn check_ref<T: AsRef<$t>>() {}
        fn propagate_ref<T1: AsRef<$t>, T2: AsRef<$t>>() {
            check_ref::<Either<T1, T2>>()
        }
        fn check_mut<T: AsMut<$t>>() {}
        fn propagate_mut<T1: AsMut<$t>, T2: AsMut<$t>>() {
            check_mut::<Either<T1, T2>>()
        }
    }};
}


fn _unsized_ref_propagation() {
    check_t!(str);

    fn check_array_ref<T: AsRef<[Item]>, Item>() {}
    fn check_array_mut<T: AsMut<[Item]>, Item>() {}

    fn propagate_array_ref<T1: AsRef<[Item]>, T2: AsRef<[Item]>, Item>() {
        check_array_ref::<Either<T1, T2>, _>()
    }

    fn propagate_array_mut<T1: AsMut<[Item]>, T2: AsMut<[Item]>, Item>() {
        check_array_mut::<Either<T1, T2>, _>()
    }
}


#[cfg(feature = "std")]
fn _unsized_std_propagation() {
    check_t!(::std::path::Path);
    check_t!(::std::ffi::OsStr);
    check_t!(::std::ffi::CStr);
}
