#' Print method for BayesPET model fitting objects
#'
#' Displays a concise overview of an object of class \code{"BayesPET_fit"},
#' including whether the analysis is blinded, the number of posterior draws,
#' and the model components.
#'
#' @param x
#' An object of class \code{"BayesPET_fit"} returned by \code{\link{fit_models}}.
#'
#' @param \dots
#' Additional arguments passed to methods. Not used.
#'
#' @seealso
#' Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_enroll}},
#' \code{\link{fit_event_blind}}, \code{\link{fit_models}}, \cr \code{\link{fit_event_unblind}}
#'
#' @return The object \code{x}, invisibly.
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' example_enroll <- data_example$example_enroll
#' example_eventcensor <- data_example$example_eventcensor
#'
#' # Blinded analysis
#' example_eventcensor.blind <- example_eventcensor
#' example_eventcensor.blind$trt <- NA
#' ## Use 2 chains and iter = 2000 here to reduce runtime for the example;
#' ## use more chains in real analyses.
#' fit.blind <- fit_models(
#'   data.enroll = example_enroll,
#'   data.eventcensor = example_eventcensor.blind,
#'   blinded = TRUE, p_trt = 0.5,
#'   chains = 2, iter = 2000, seed = list(123),
#'   return_fit = TRUE, mc.cores = 1, quiet = FALSE
#' )
#' print(fit.blind)
#' }
#'
#' @export
print.BayesPET_fit <- function(x, ...) {

  cat("BayesPET fitted model\n")
  cat("---------------------\n")

  ## ---- posterior draws ----
  draw_keys <- c("mu", "rho_e", "lambda_0e", "rho_c", "lambda_0c", "eta")
  draw_key  <- draw_keys[draw_keys %in% names(x)][1]

  if (!is.na(draw_key)) {
    S <- length(x[[draw_key]])
    cat("Posterior draws:", S, "\n")
  }

  if ("blinded" %in% names(x)) {
    cat("Blinded analysis:", if (isTRUE(x$blinded)) "yes" else "no", "\n")
  }

  print_block <- function(title, keys) {
    cat("\n", title, "\n", sep = "")
    for (nm in keys) {
      if (!nm %in% names(x)) next
      if (is.null(x[[nm]])) {
        cat("  - ", nm, ": NULL\n", sep = "")
      } else {
        cat("  - ", nm, "\n", sep = "")
      }
    }
    invisible(NULL)
  }

  ## ---- model components ----
  print_block("Enrollment model:", c("mu"))

  print_block("Censoring model:", c("rho_c", "lambda_0c", "beta_c"))

  event_keys <- c("rho_e", "lambda_0e", "eta", "beta_e")
  if (isTRUE(x$blinded)) {
    event_keys <- c(event_keys, "x")
  } else {
    event_keys <- c(event_keys, "treatment_ind")
  }
  print_block("Event model:", event_keys)

  ## ---- Stan fits ----
  if ("fit" %in% names(x)) {
    cat("\n'rstan' fit objects:\n")
    for (nm in names(x$fit)) cat("  - ", nm, "\n", sep = "")
  }

  invisible(x)
}


#' Print method for BayesPET prediction objects
#'
#' Displays a brief overview of an object of class \code{"BayesPET_predtime"}
#' and lists the components available in the result. For numerical summaries,
#' use \code{\link{summary}}; for visualization, use \code{\link{plot}}.
#'
#' @param x
#' An object of class \code{"BayesPET_predtime"} returned by
#' \code{\link{predict_eventtime}}.
#'
#' @param \dots
#' Additional arguments passed to methods. Not used.
#'
#' @seealso
#' Other BayesPET prediction: \code{\link{plot.BayesPET_predtime}},
#' \code{\link{predict_eventtime}}, \cr
#' \code{\link{summary.BayesPET_predtime}}
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' ## Reduced number of chains and iterations compared to defaults
#' ## to keep the example computationally manageable.
#' pred <- predict_eventtime(
#'   N = 200,
#'   E_target = 150,
#'   data.enroll = data_example$example_enroll,
#'   data.eventcensor = data_example$example_eventcensor,
#'   blinded = TRUE,
#'   p_trt = 0.5,
#'   chains = 2,
#'   iter = 2000,
#'   assess_window = 2,
#'   seed.fit = 1,
#'   seed.pred = 2,
#'   return_fit = TRUE,
#'   return_draws = TRUE,
#'   quiet = TRUE
#' )
#'
#' print(pred)
#' summary(pred)
#' plot(pred)
#' }
#'
#' @return The object \code{x}, invisibly.
#' @export
print.BayesPET_predtime <- function(x, ...) {
  cat("BayesPET predicted target event time\n")
  cat("-----------------------------------\n")

  comps <- names(x)
  if (length(comps) > 0L) {
    cat("Components:\n")
    for (nm in comps) {
      cat("  - ", nm, "\n", sep = "")
    }
  }

  cat("\nUse summary() for quantiles and plot() for visualization.\n")
  invisible(x)
}


#' Summary method for BayesPET prediction objects
#'
#' Summarizes an object of class \code{"BayesPET_predtime"} by reporting
#' summary statistics of the posterior predictive distribution of the
#' calendar time at which the target number of events is reached.
#'
#' @param object
#' An object of class \code{"BayesPET_predtime"} returned by
#' \code{\link{predict_eventtime}}.
#'
#' @param \dots
#' Additional arguments passed to methods. Not used.
#'
#' @return
#' \code{\link{summary.BayesPET_predtime}} returns an object of class \code{"summary.BayesPET_predtime"}, which is a named list
#' containing summary information for the posterior predictive distribution of
#' the target event time. Components include:
#' \itemize{
#'   \item \code{S}: Total number of posterior predictive draws.
#'   \item \code{n_infinite}: Number of draws in which the target number of events
#'     is not reached.
#'   \item \code{prob_not_reached}: Proportion of draws in which the target is not reached.
#'   \item \code{q25}: Posterior 25th percent quantile of the target event time.
#'   \item \code{median}: Posterior median of the target event time.
#'   \item \code{q75}: Posterior 75th percent quantile of the target event time.
#'   \item \code{call}: The function call used to generate the prediction.
#' }
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' ## Reduced number of chains and iterations compared to defaults
#' ## to keep the example computationally manageable.
#' pred <- predict_eventtime(
#'   N = 200,
#'   E_target = 150,
#'   data.enroll = data_example$example_enroll,
#'   data.eventcensor = data_example$example_eventcensor,
#'   blinded = TRUE,
#'   p_trt = 0.5,
#'   chains = 2,
#'   iter = 2000,
#'   assess_window = 2,
#'   seed.fit = 1,
#'   seed.pred = 2,
#'   return_fit = TRUE,
#'   return_draws = TRUE,
#'   quiet = TRUE
#' )
#'
#' print(pred)
#' summary(pred)
#' plot(pred)
#' }
#'
#' @seealso
#' Other BayesPET prediction: \code{\link{plot.BayesPET_predtime}},
#' \code{\link{predict_eventtime}}, \cr
#' \code{\link{print.BayesPET_predtime}}
#'
#' @export
summary.BayesPET_predtime <- function(object, ...) {
  if (!is.list(object) || is.null(object$prediction)) {
    stop("Invalid 'BayesPET_predtime' object: missing component 'prediction'.",
         call. = FALSE)
  }

  pred <- object$prediction
  S <- length(pred)
  n_inf <- sum(!is.finite(pred))

  qs <- if (length(pred) > 0L) {
    stats::quantile(pred,
                    probs = c(0.25, 0.5, 0.75),
                    na.rm = FALSE,
                    names = TRUE,
                    type = 7)
  } else {
    stats::setNames(rep(NA_real_, 3), c("25%", "50%", "75%"))
  }

  out <- list(
    S = S,
    n_infinite = n_inf,
    prob_not_reached = if (S > 0) n_inf / S else NA_real_,
    q25 = unname(qs[[1]]),
    median = unname(qs[[2]]),
    q75 = unname(qs[[3]]),
    call = object$call
  )

  class(out) <- "summary.BayesPET_predtime"
  out
}




#' @param x
#' An object of class \code{"summary.BayesPET_predtime"} returned by
#' \code{\link{summary}} applied to a \code{"BayesPET_predtime"} object.
#'
#' @param digits
#' Integer. Number of significant digits to use when printing numerical
#' summaries. Defaults to \code{4}.
#'
#' @param \dots
#' Additional arguments passed to methods. Not used.
#'
#' @rdname summary.BayesPET_predtime
#'
#' @return \code{\link{print.summary.BayesPET_predtime}} returns the object \code{x}, invisibly.
#' @export
print.summary.BayesPET_predtime <- function(x, digits = 4, ...) {
  fmt <- function(z) formatC(z, digits = 4, format = "fg")

  cat("Summary: BayesPET target event time\n")
  cat("----------------------------------\n")

  if (!is.null(x$call)) {
    cat("Call:\n")
    print(x$call)
  }

  cat("\nPosterior draws:", x$S, "\n")

  if (x$n_infinite > 0L) {
    cat("Inf draws:", x$n_infinite,
        "(P =", fmt(x$prob_not_reached), ")\n")
  }

  cat("\nPosterior quantiles (including Inf):\n")
  cat("  25%:    ", fmt(x$q25), "\n", sep = "")
  cat("  Median: ", fmt(x$median), "\n", sep = "")
  cat("  75%:    ", fmt(x$q75), "\n", sep = "")

  invisible(x)
}


#' Plot method for BayesPET prediction objects
#'
#' Plots an object of class \code{"BayesPET_predtime"} by displaying a histogram
#' of posterior predictive draws of the calendar time at which the target number
#' of events is reached. Only finite draws are included. A vertical line indicates
#' the posterior median when it is finite.
#'
#' @param x
#' An object of class \code{"BayesPET_predtime"} returned by
#' \code{\link{predict_eventtime}}.
#'
#' @param breaks
#' Passed to \code{\link[graphics]{hist}}. Default is \code{"Sturges"}.
#'
#' @param xlab
#' X-axis label. Default is \code{"Calendar time to reach E_target"}.
#'
#' @param \dots
#' Additional arguments passed to methods. Not used.
#'
#' @seealso
#' Other BayesPET prediction: \code{\link{predict_eventtime}}, \code{\link{print.BayesPET_predtime}}, \cr
#' \code{\link{summary.BayesPET_predtime}}
#'
#' @return
#' Invisibly returns \code{NULL}.
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' data(data_example)
#' ## Reduced number of chains and iterations compared to defaults
#' ## to keep the example computationally manageable.
#' pred <- predict_eventtime(
#'   N = 200,
#'   E_target = 150,
#'   data.enroll = data_example$example_enroll,
#'   data.eventcensor = data_example$example_eventcensor,
#'   blinded = TRUE,
#'   p_trt = 0.5,
#'   chains = 2,
#'   iter = 2000,
#'   assess_window = 2,
#'   seed.fit = 1,
#'   seed.pred = 2,
#'   return_fit = TRUE,
#'   return_draws = TRUE,
#'   quiet = TRUE
#' )
#'
#' print(pred)
#' summary(pred)
#' plot(pred)
#' }
#'
#' @export
plot.BayesPET_predtime <- function(x,
                                   breaks = "Sturges",
                                   xlab = "Predicted calendar time to reach target number of events",
                                   ...) {
  if (!is.list(x) || is.null(x$prediction)) {
    stop("Invalid 'BayesPET_predtime' object: missing component 'prediction'.",
         call. = FALSE)
  }

  pred <- x$prediction
  pred_fin <- pred[is.finite(pred)]

  if (length(pred_fin) == 0L) {
    graphics::plot.new()
    graphics::text(0.5, 0.5, "No finite predictions to plot.", cex = 1)
    return(invisible(NULL))
  }

  # Draw histogram (no title)
  h <- graphics::hist(pred_fin, breaks = breaks, main = "", xlab = xlab)

  # Median from full prediction (including Inf)
  med <- stats::quantile(pred, probs = 0.5,
                         na.rm = FALSE, names = FALSE, type = 7)

  if (is.finite(med)) {
    graphics::abline(v = med, lwd = 2, lty = 2)

    # Allow text to extend slightly beyond plot region
    old <- graphics::par(xpd = NA)
    on.exit(graphics::par(old), add = TRUE)

    # Place label just above tallest bar, aligned with vline
    y_pos <- max(h$counts) * 1.02

    graphics::text(
      x = med,
      y = y_pos,
      labels = paste0("Median = ", formatC(med, digits = 5, format = "fg")),
      pos = 3,            # above the point
      cex = 0.9
    )
  }

  invisible(NULL)
}

#' Summary method for BayesPET operating characteristics object
#'
#' @description
#' Computes summary measures of prediction accuracy from a
#' \code{"BayesPET_oc"} object. The summaries are based on the
#' differences between the predicted median and true
#' calendar times to reach the target number of events.
#'
#' @param object An object of class \code{"BayesPET_oc"} returned by \code{\link{get_oc}}.
#'
#' @param thresholds Numeric vector of non-negative thresholds used to compute
#'   the proportion of replicates for which the absolute prediction error is
#'   less than each threshold. Defaults to
#'   \code{c(0.25, 0.5, 1, 1.5, 2)}.
#' @param ... Not used.
#'
#' @return
#' \code{\link{summary.BayesPET_oc}} returns an object of class \code{"summary.BayesPET_oc"}, a list containing:
#' \itemize{
#'   \item \code{n_valid}: Number of valid simulation replicates (where the target
#'     event count can be reached).
#'   \item \code{n_attempt}: Number of datasets generated to obtain the valid
#'     replicates.
#'   \item \code{success_rate}: Proportion of attempted datasets that produced
#'     valid replicates.
#'   \item \code{thresholds}: Threshold values used to evaluate prediction accuracy.
#'   \item \code{pr_lt}: Proportion of replicates with prediction error less than
#'     each threshold.
#'   \item \code{mae}: Mean absolute prediction error.
#'   \item \code{median_ae}: Median absolute prediction error.
#'   \item \code{rmse}: Root mean squared prediction error.
#'   \item \code{call}: The matched function call.
#' }
#'
#' @family BayesPET operating characteristics
#'
#' @examplesIf requireNamespace("rstan", quietly = TRUE)
#' \donttest{
#' ## Using nsim = 2, chains = 2, and iter = 2000 to reduce runtime.
#' ## Use larger nsim, chains and iter in real analyses.
#' oc <- get_oc(
#'   N = 200, E_target = 150,
#'   E_cutoff = 75, p_trt = 0.5,
#'   cov_type = c("binary", "continuous"),
#'   cov_dist = c(0.5, 2),
#'   beta.event = c(-0.2, -0.2),
#'   beta.censor = c(0, 0),
#'   logHR.trt = log(0.65),
#'   enroll_rate = 16,
#'   dist.event = "Weibull", dist.censor = "Weibull",
#'   event.scale = 1/5^3, event.shape = 3,
#'   censor.scale = 1/10^6, censor.shape = 6,
#'   blinded = TRUE,
#'   assess_window = 2,
#'   seed = 1,
#'   chains = 2, iter = 2000,
#'   nsim = 2,
#'   n_workers = 1
#' )
#'
#' summary(oc)
#' }
#' @export
summary.BayesPET_oc <- function(object, thresholds = c(0.25,0.5,1,1.5,2), ...) {
  df <- object$replicate
  if (!is.data.frame(df) || !"difference" %in% names(df)) {
    stop("Invalid 'BayesPET_oc_predtime' object: missing 'replicate$difference'.", call. = FALSE)
  }

  d <- df$difference
  d <- d[is.finite(d)]

  pr_lt <- vapply(thresholds, function(t) mean(d < t), numeric(1))

  out <- list(
    n_valid = object$n_valid,
    n_attempt = object$n_attempt,
    success_rate = if (!is.null(object$n_attempt) && object$n_attempt > 0)
      object$n_valid / object$n_attempt else NA_real_,
    thresholds = thresholds,
    pr_lt = pr_lt,
    mae = mean(d),
    median_ae = stats::median(d),
    rmse = sqrt(mean(d^2)),
    call = object$call
  )

  class(out) <- "summary.BayesPET_oc"
  out
}


#' @param x
#' An object of class \code{"summary.BayesPET_oc"} returned by
#' \code{\link{summary.BayesPET_oc}}.
#'
#' @param digits
#' Integer specifying the number of decimal places to use when printing
#' numerical summaries. Defaults to \code{3}.
#'
#' @return \code{\link{print.summary.BayesPET_oc}} returns the object \code{x}, invisibly.
#' @rdname summary.BayesPET_oc
#' @export
print.summary.BayesPET_oc <- function(x, digits = 3, ...) {
  fmt <- paste0("%.", digits, "f")

  cat("BayesPET operating characteristics summary\n\n")

  cat("Valid replicates:", x$n_valid, "\n")
  if (!is.null(x$n_attempt)) {
    cat("Attempted datasets:", x$n_attempt, "\n")
  }
  cat("\n")

  cat("Prediction accuracy metrics:\n")
  cat("  Mean absolute error:", sprintf(fmt, x$mae), "\n")
  cat("  Median absolute error:", sprintf(fmt, x$median_ae), "\n")
  cat("  RMSE:", sprintf(fmt, x$rmse), "\n\n")

  cat("Proportion of replicates with event time prediction absolute error < threshold:\n")
  out <- data.frame(
    threshold = x$thresholds,
    proportion = x$pr_lt
  )
  print(out, row.names = FALSE, digits = digits)

  invisible(x)
}



