\name{plot.feasible}
\alias{plot.feasible}
\title{Plot Method for \code{feasible} objects}
\description{
Generate diagnostic plots for objects of class \code{"feasible"}, returned by \code{\link{feasible}}.
One can display either (i) mean feasible vs. target interventions, or (ii) the non-overlap ratio.
}
\usage{
\method{plot}{feasible}(x, x.axis = c("strategy", "time"),
                        which  = c("feasible", "nonoverlap"),
                        facet  = c("none", "time", "strategy"), ...)
}
\arguments{
  \item{x}{An object of class \code{"feasible"} with a \code{"summary"} attribute
           (typically returned by \code{\link{feasible}}).}
  
  \item{x.axis}{A string specifying the x-axis: 
    \itemize{
      \item If \code{"strategy"}, and each strategy corresponds to the same target value
            at every time-point (i.e., this relationship is consistent across time),
            the method uses \code{abar} for the x-axis, otherwise the strategy index is used.
            \item If \code{"time"}, the x-axis shows discrete time-points and colors represent
            targets or strategies, depending on the context.
    }
  }
  
  \item{which}{Which plot to show:
    \itemize{
      \item \code{"feasible"}: mean feasible intervention values compared to original target intervention.
      \item \code{"nonoverlap"}: non-overlap ratio (proportion of mass below the density threshold).
    }
 }
  
  \item{facet}{Optional faceting to reduce overplotting:
    \itemize{
      \item \code{"none"} (default): no faceting, all series in a single panel.
      \item \code{"time"}: one panel per time-point.
      \item \code{"strategy"}: one panel per intervention strategy.
    }
    Facet strips are labelled with variable name and value (via \code{\link[ggplot2]{label_both}}).}
  
  \item{\dots}{Additional arguments (currently unused). Included for method consistency.}
}
\details{
Both plot types are drawn with \pkg{ggplot2}. To reduce overplotting,
lines and points use transparency (\code{alpha}) and slightly smaller
widths/sizes by default. Faceting by \code{time} or \code{strategy}
can further improve readability when many series are present.

The \code{"summary"} attribute of a \code{"feasible"} object is expected to contain
(at least) the following columns:
\itemize{
  \item \code{time}: discrete time index.
  \item \code{Strategy}: strategy index (row index of the intervention design).
  \item \code{Abar}: target value (intervention level) for that strategy at that time.
  \item \code{Feasible}: mean feasible value under the estimated feasible intervention.
  \item \code{Low}: non-overlap ratio (proportion of mass below the density threshold).
}

\strong{Interpretation of \code{abar}:}
\itemize{
  \item In \code{\link{feasible}}, the \code{abar} argument may be either
        a numeric vector (static grid of targets) or a numeric matrix (dynamic interventions).
  \item If \code{abar} is a vector, each distinct value defines a strategy that is
        constant over time; in this case each strategy represents the same target value
        at every time-point.
  \item If \code{abar} is a matrix, rows index intervention strategies and
        columns index time-points. In the summary, \code{Strategy} identifies the row,
        and \code{Abar} is the entry of that row at the corresponding time-point.
}

\strong{Plot types:}
\enumerate{
  \item \strong{Feasible vs Target (\code{which = "feasible"})}:
  \itemize{
    \item \emph{Y-axis:} mean feasible intervention (\code{Feasible}).
    \item \emph{X-axis:} controlled by \code{x.axis}:
      \itemize{
        \item \code{x.axis = "time"}: x-axis shows time; colors represent Targets (\code{Abar})
              when each strategy has the same target at all time-points, or represent strategies
              when targets vary over time within a strategy.
        \item \code{x.axis = "strategy"}: x-axis shows strategy index; if each strategy corresponds
              to a single target value at all time-points and this relationship is consistent,
              the x-axis is relabelled to show \code{Abar} (Targets) instead.
      }
    \item \emph{Reference line and ticks:}
      \item When the x-axis is on the Target scale (strategies are constant over time with respect to \code{Abar}),
            the plot includes a dashed 1:1 reference line \eqn{\text{Feasible} = \text{Target}} and aligns
            the x- and y-axis limits to the range of \code{Abar}, when plausible (i.e., when the feasible values
            lie within the range of \code{Abar}).
      \item When \code{x.axis = "time"}, short horizontal ticks at each time-point indicate the
            \code{Abar} values for each strategy (or Target when strategies are constant over time),
            using the same color mapping as the series.
      \item When \code{x.axis = "strategy"} and strategies do not correspond to a single target over all time-points,
            ticks are drawn at each strategy to indicate the \code{Abar} values across time.
  }
  
  \item \strong{Non-overlap Ratio (\code{which = "nonoverlap"})}:
  \itemize{
    \item \emph{Y-axis:} non-overlap ratio \code{Low} (bounded between 0 and 1), plotted with fixed limits
          \code{c(0, 1)}.
    \item \emph{X-axis:} same choice of \code{x.axis} as for the feasible plot.
  }
}

\emph{Terminology:} Throughout the plots, \dQuote{Target} refers to the intervention values passed as the
\code{abar} argument to \code{\link{feasible}} (stored as column \code{Abar} in the object's summary).
When strategies are constant over time with respect to \code{Abar} and this structure is consistent across time,
each Target corresponds to an identical intervention pattern at all time-points. This is reflected in both the
x-axis labelling and the legend.
}
\value{
Invisibly returns the \pkg{ggplot2} object that is drawn (either the feasible plot or the non-overlap plot).
}
\seealso{
\code{\link{feasible}}, \code{\link{summary.feasible}}
}
\examples{
data(EFV)

\donttest{
Lnodes <- c("adherence.1","weight.1",
            "adherence.2","weight.2",
            "adherence.3","weight.3",
            "adherence.4","weight.4")
Ynodes <- c("VL.0","VL.1","VL.2","VL.3","VL.4")
Anodes <- c("efv.0","efv.1","efv.2","efv.3","efv.4")

## ------------------------------------------------------------------
## Example 1: Static grid of Targets (vector abar)
## Each strategy uses the same target value at every time-point
## ------------------------------------------------------------------

abar_static <- seq(0, 10, by = 2)

m_static <- feasible(X = EFV,
                     Anodes = Anodes,
                     Lnodes = Lnodes,
                     Ynodes = Ynodes,
                     d.method = "parametric",
                     abar = abar_static,
                     grid.size = NULL,
                     left.boundary = 0,
                     right.boundary = 10)

## Feasible vs Target with time on x-axis (default).
## Colors indicate Targets (Abar), and short ticks show Abar at each time.
plot(m_static, which = "feasible")

## Feasible vs Target with time on x-axis.
plot(m_static, x.axis = "time", which = "feasible")

## Non-overlap ratio 
plot(m_static, which = "nonoverlap")

## Facet by time to reduce overplotting
plot(m_static, which = "feasible", facet = "time")


## ------------------------------------------------------------------
## Example 2: Non-constant intervention strategies (matrix abar)
## Strategies can have different target values at different time-points
## ------------------------------------------------------------------

## Here rows define strategies and columns define time-points.
abar_matrix <- rbind(
  c(0, 2, 4, 6, 8),  # strategy 1
  c(9, 6, 2, 1, 0),  # strategy 2
  c(1, 3, 5, 7, 9)   # strategy 3
)

set.seed(456)
m_matrix <- feasible(X = EFV,
                     Anodes = Anodes,
                     Lnodes = Lnodes,
                     Ynodes = Ynodes,
                     d.method = "parametric",
                     abar = abar_matrix,
                     grid.size = 1,
                     left.boundary = 0,
                     right.boundary = 10)

## Feasible vs Target with time on the x-axis.
## Colors represent strategies; short ticks at each time show
## the corresponding Abar for each strategy.
plot(m_matrix,
     x.axis = "time",
     which  = "feasible",
     facet  = "none")

## Feasible vs Target with strategy on the x-axis.
## Strategies no longer use the same target at all time-points,
## so the x-axis stays on the strategy index, and ticks at each
## strategy indicate the Abar values across time.
plot(m_matrix,
     x.axis = "strategy",
     which  = "feasible",
     facet  = "none")

## Non-overlap ratio for these non-constant strategies,
## shown over time and faceted by strategy for clarity.
plot(m_matrix,
     x.axis = "time",
     which  = "nonoverlap",
     facet  = "strategy")
}
}
\author{Han Bao, Michael Schomaker}