% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CKTmatrix.kernel.R
\name{CKTmatrix.kernel}
\alias{CKTmatrix.kernel}
\title{Estimate the conditional Kendall's tau matrix
at different conditioning points}
\usage{
CKTmatrix.kernel(
  dataMatrix,
  observedZ,
  gridZ,
  averaging = "no",
  blockStructure = NULL,
  h,
  kernel.name = "Epa",
  typeEstCKT = "wdm"
)
}
\arguments{
\item{dataMatrix}{a matrix of size \code{(n,d)} containing \code{n} observations of a
\code{d}-dimensional random vector \eqn{X}.}

\item{observedZ}{vector of observed points of a conditioning variable \eqn{Z}.
It must have the same length as the number of rows of \code{dataMatrix}.}

\item{gridZ}{points at which the conditional Kendall's tau is computed.}

\item{averaging}{type of averaging used for fast estimation.
Possible choices are \itemize{
  \item \code{no}: no averaging;
  \item \code{all}: averaging all Kendall's taus in each block;
  \item \code{diag}: averaging along diagonal blocks elements.
}}

\item{blockStructure}{list of vectors.
Each vector corresponds to one group of variables
and contains the indexes of the variables that belongs to this group.
\code{blockStructure} must be a partition of \code{1:d},
where \code{d} is the number of columns in \code{dataMatrix}.}

\item{h}{bandwidth. It can be a real, in this case the same \code{h}
will be used for every element of \code{gridZ}.
If \code{h} is a vector then its elements are recycled to match the length of
\code{gridZ}.}

\item{kernel.name}{name of the kernel used for smoothing.
Possible choices are: \code{"Gaussian"} (Gaussian kernel)
and \code{"Epa"} (Epanechnikov kernel).}

\item{typeEstCKT}{type of estimation of the conditional Kendall's tau.}
}
\value{
array with dimensions depending on \code{averaging}:
\itemize{
  \item If \code{averaging = "no"}:
  it returns an array of dimensions \code{(n, n, length(gridZ))},
  containing the estimated conditional Kendall's tau matrix given \eqn{Z = z}.
  Here, \code{n} is the number of rows in \code{dataMatrix}.

  \item If \code{averaging = "all"} or \code{"diag"}:
  it returns an array of dimensions
  \code{(length(blockStructure), length(blockStructure), length(gridZ))},
  containing the block estimates of the conditional Kendall's tau given \eqn{Z = z}
  with ones on the diagonal.
}
}
\description{
Assume that we are interested in a random vector \eqn{(X, Z)},
where \eqn{X} is of dimension \eqn{d > 2} and \eqn{Z} is of dimension \eqn{1}.
We want to estimate the dependence across the elements of the conditioned vector \eqn{X}
given \eqn{Z=z}.
This function takes in parameter observations of \eqn{(X,Z)}
and returns kernel-based estimators of \deqn{\tau_{i,j | Z=zk}}
which is the conditional Kendall's tau between \eqn{X_i} and \eqn{X_j}
given to \eqn{Z=zk}, for every conditioning point \eqn{zk} in \code{gridZ}.
If the conditional Kendall's tau matrix has a block structure,
then improved estimation is possible by averaging over the kernel-based estimators of
pairwise conditional Kendall's taus.
Groups of variables composing the same blocks can be defined
using the parameter \code{blockStructure}, and the averaging can be set on using
the parameter \code{averaging=all}, or \code{averaging=diag}
for faster estimation by averaging only over diagonal elements of each block.
}
\examples{

# Data simulation
n = 100
Z = runif(n)
d = 5
CKT_11 = 0.8
CKT_22 = 0.9
CKT_12 = 0.1 + 0.5 * cos(pi * Z)
data_X = matrix(nrow = n, ncol = d)
for (i in 1:n){
  CKT_matrix = matrix(data =
    c(  1      , CKT_11   , CKT_11   , CKT_12[i], CKT_12[i] ,
      CKT_11   ,   1      , CKT_11   , CKT_12[i], CKT_12[i] ,
      CKT_11   , CKT_11   ,    1     , CKT_12[i], CKT_12[i] ,
      CKT_12[i], CKT_12[i], CKT_12[i],   1      , CKT_22    ,
      CKT_12[i], CKT_12[i], CKT_12[i], CKT_22   ,   1
      ) ,
     nrow = 5, ncol = 5)
  sigma = sin(pi * CKT_matrix/2)
  data_X[i, ] = mvtnorm::rmvnorm(n = 1, sigma = sigma)
}
plot(as.data.frame.matrix(data_X))

# Estimation of CKT matrix
h = 1.06 * sd(Z) * n^{-1/5}
gridZ = c(0.2, 0.8)
estMatrixAll <- CKTmatrix.kernel(
  dataMatrix = data_X, observedZ = Z, gridZ = gridZ, h = h)
# Averaging estimator
estMatrixAve <- CKTmatrix.kernel(
  dataMatrix = data_X, observedZ = Z, gridZ = gridZ,
  averaging = "diag", blockStructure = list(1:3,4:5), h = h)

# The estimated CKT matrix conditionally to Z=0.2 is:
estMatrixAll[ , , 1]
# Using the averaging estimator,
# the estimated CKT between the first group (variables 1 to 3)
# and the second group (variables 4 and 5) is
estMatrixAve[1, 2, 1]

# True value (of CKT between variables in block 1 and 2 given Z = 0.2):
0.1 + 0.5 * cos(pi * 0.2)


}
\references{
van der Spek, R., & Derumigny, A. (2025).
Fast estimation of Kendall's Tau and conditional Kendall's Tau matrices
under structural assumptions.
\doi{10.1515/demo-2025-0012}
}
\seealso{
\code{\link{CKT.kernel}} for kernel-based estimation of conditional Kendall's tau
between two variables (i.e. the equivalent of this function
when \eqn{X} is bivariate and \code{d=2}).
}
\author{
Rutger van der Spek, Alexis Derumigny
}
