#' @export
#' @title Overlapping Generations Models with Private Firm
#' @aliases gemOLG_PrivateFirm
#' @description Some examples of overlapping generations models with private firm.
#' A public (i.e. publicly held) firm exists permanently and operates independently.
#' If a public firm ownership transfers between generations, this transfer will be done through
#' the exchange of shares.
#' In contrast, here a private firm is established by a consumer and only runs before she retires.
#'
#' In the first example, there are some two-period-lived consumers and a private firm.
#' Suppose age1 has a unit of labor and age2 has not.
#' In each period age1 establishes a private firm and the firm gets some labor as investment from age1
#' and will sell it in the market to buy some inputs for production.
#' In the next period, age2 (i.e. age1 in the previous period) gets the output of the firm.
#' Age2 consumes product and labor (i.e. service). Hence age1 and the firm can sell labor to age2
#' and buy product from age2.
#'
#' In the second example with three-period-lived consumers, there are two private firms (i.e. firm1 and firm2).
#' In each period age1 establishes a new firm1 and age2 establishes a new firm2.
#' Firm1 gets some labor as investment from age1 and firm2 gets some product and labor as investment from age2.
#' The output of firm1 belongs to age2 in the next period and the output of firm2 belongs to age3 in the next period.
#' In each period age2 (i.e. age1 in the previous period) takes away the output of firm1 and
#' age3 (i.e. age2 in the previous period) takes away the output of firm2.
#' @param ... arguments to be passed to the function sdm2.
#' @references Acemoglu, D. (2009, ISBN: 9780691132921) Introduction to Modern Economic Growth. Princeton University Press.
#' @seealso {
#' \code{\link{gemOLGF_PureExchange}}
#' }
#' @examples
#' \donttest{
#' #### an example with a private firm and two-period-lived consumers
#' saving.rate <- 0.5
#' beta.consumer <-  c(1 / 2, 1 / 2) # c(9 / 10, 1 / 10)
#'
#' dst.firm <- node_new(
#'   "prod",
#'   type = "CD", alpha = 5,
#'   beta = c(2 / 3, 1 / 3),
#'   "prod", "lab"
#' )
#'
#' dst.age1 <- node_new(
#'   "util",
#'   type = "CD", alpha = 1,
#'   beta = beta.consumer,
#'   "prod", "lab"
#' )
#'
#' dst.age2 <- Clone(dst.age1)
#'
#' ge <- sdm2(
#'   A = list(
#'     dst.firm, dst.age1, dst.age2
#'   ),
#'   B = matrix(c(
#'     1, 0, 0,
#'     0, 0, 0
#'   ), 2, 3, TRUE),
#'   S0Exg = matrix(c(
#'     NA, NA, NA,
#'     NA, 1, NA
#'   ), 2, 3, TRUE),
#'   names.commodity = c("prod", "lab"),
#'   names.agent = c("firm", "age1", "age2"),
#'   numeraire = "lab",
#'   policy = function(time, state) {
#'     if (time > 1) {
#'       supply.prod <- state$S[1, 1]
#'       supply.lab <- state$S[2, 2]
#'       state$S <- 0 * state$S
#'       state$S[1, 3] <- supply.prod # age2 supplies prod.
#'       state$S[2, 1] <- saving.rate * supply.lab # The firm gets investment from age1.
#'       state$S[2, 2] <- (1 - saving.rate) * supply.lab
#'     }
#'     state
#'   }
#' )
#'
#' ge$p
#' addmargins(ge$D, 2)
#' addmargins(ge$S, 2)
#' addmargins(ge$DV)
#' addmargins(ge$SV)
#'
#' #### an example with two private firm and three-period-lived consumers
#' saving.rate.age1 <- 1 / 3
#' saving.rate.age2 <- 0.95
#'
#' dst.firm1 <- dst.firm2 <- node_new(
#'   "prod",
#'   type = "CD", alpha = 5,
#'   beta = c(2 / 3, 1 / 3),
#'   "prod", "lab"
#' )
#'
#' dst.age1 <- dst.age2 <- dst.age3 <- node_new(
#'   "util",
#'   type = "Leontief", a = 1,
#'   "prod"
#' )
#'
#' ge <- sdm2(
#'   A = list(
#'     dst.firm1, dst.firm2, dst.age1, dst.age2, dst.age3
#'   ),
#'   B = matrix(c(
#'     1, 1, 0, 0, 0,
#'     0, 0, 0, 0, 0
#'   ), 2, 5, TRUE),
#'   S0Exg = matrix(c(
#'     NA, NA, NA, NA, NA,
#'     NA, NA, 1, 1, NA
#'   ), 2, 5, TRUE),
#'   names.commodity = c("prod", "lab"),
#'   names.agent = c("firm1", "firm2", "age1", "age2", "age3"),
#'   numeraire = "lab",
#'   policy = function(time, state) {
#'     if (time > 1) {
#'       state$S[1, 5] <- state$S[1, 2] # Age3 takes away the output of firm2.
#'       state$S[1, 2] <- 0
#'
#'       # Age2 takes away the output of firm1.
#'       prod.age2 <- state$S[1, 1]
#'       state$S[1, 1] <- 0
#'       # Age2 establishes a new firm2.
#'       lab.age2 <- state$S[2, 4]
#'       state$S[2, 2] <- lab.age2 * saving.rate.age2 # Firm2 sells labor.
#'       state$S[1, 2] <- prod.age2 * saving.rate.age2 # Firm2 sells product.
#'
#'       state$S[2, 4] <- lab.age2 * (1 - saving.rate.age2) # Age2 sells labor.
#'       state$S[1, 4] <- prod.age2 * (1 - saving.rate.age2) # Age2 sells product.
#'
#'       # Age1 establishes a new firm1.
#'       state$S[2, 1] <- state$S[2, 3] * saving.rate.age1 # Firm1 sells labor.
#'       state$S[2, 3] <- state$S[2, 3] * (1 - saving.rate.age1) # Age1 sells labor.
#'     }
#'     state
#'   }
#' )
#'
#' ge$p
#' ge$z
#' addmargins(ge$D, 2)
#' addmargins(ge$S, 2)
#' addmargins(ge$DV)
#' addmargins(ge$SV)
#' }



gemOLG_PrivateFirm <- function(...) sdm2(...)
