% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/save_har.R
\name{save_har}
\alias{save_har}
\title{Save Data to GEMPACK HAR Format}
\usage{
save_har(
  data_list,
  file_path,
  dimensions,
  value_cols = NULL,
  long_desc = NULL,
  coefficients = NULL,
  export_sets = TRUE,
  lowercase = TRUE,
  dim_order = NULL,
  dim_rename = NULL
)
}
\arguments{
\item{data_list}{A named list of data frames or arrays (names up to 4 characters).}

\item{file_path}{Path to the output HAR file.}

\item{dimensions}{A named list specifying dimension columns for each header.
\itemize{
\item HAR dimension names follow these column names; rename columns before saving to change them.
}}

\item{value_cols}{A named vector of value column names. Defaults to "Value".}

\item{long_desc}{A named vector of long descriptions (optional).}

\item{coefficients}{A named vector of coefficient names (optional).}

\item{export_sets}{Logical; if TRUE, exports dimension sets as 1C headers. Default is TRUE.}

\item{lowercase}{Logical; if TRUE, converts elements to lowercase. Default is TRUE.}

\item{dim_order}{Optional dimension ordering specification. Can be:
\itemize{
\item \code{NULL} (default): alphabetical A–Z ordering
\item a data frame with columns matching dimension names
\item a named list specifying order for each dimension
\item a character string giving the path to an Excel or CSV file with order definitions
}}

\item{dim_rename}{Optional named list to rename dimensions in HAR output.
\itemize{
\item Names are original column names, values are desired HAR dimension names
\item Allows duplicate dimension names in HAR (e.g., COMMxREGxREG from COMMxSREGxDREG)
\item Example: list(RTMS = c(COMM = "COMM", SREG = "REG", DREG = "REG"))
}}
}
\value{
Invisibly returns a list containing export metadata, including file path and header summary.
}
\description{
Writes one or more data matrices or arrays into a GEMPACK-compatible
HAR file format, automatically generating associated 1C set headers.
}
\details{
\itemize{
\item Supports \verb{1CFULL} (string sets) and \code{REFULL} (real arrays) headers
\item \verb{2IFULL} (integer) and \code{RESPSE} (sparse) types are not implemented
\item Up to seven dimensions and ~2 million elements per chunk
\item No practical file-size limit or header count restriction
}
}
\examples{
# Example 1: Save a single matrix
har_path <- system.file("extdata", "TAR10-WEL.har", package = "HARplus")
har_data <- load_harx(har_path)
welfare_data <- get_data_by_var("A", har_data)
welfare_data_A <- welfare_data[["har_data"]][["A"]]

save_har(
  data_list   = list(WELF = welfare_data),
  file_path   = file.path(tempdir(), "output_single.har"),
  dimensions  = list(WELF = c("REG", "COLUMN")),
  value_cols  = c(WELF = "Value"),
  long_desc   = c(WELF = "Welfare Decomposition"),
  export_sets = TRUE,
  lowercase   = FALSE
)

# Example 2: Save multiple matrices
welfare_data <- get_data_by_var(c("A", "A1"), har_data)
welfare_data <- welfare_data[["har_data"]]

save_har(
  data_list   = list(WELF = welfare_data[["A"]],
                     DECOM = welfare_data[["A1"]]),
  file_path   = file.path(tempdir(), "output_multi.har"),
  dimensions  = list(WELF = c("REG", "COLUMN"),
                     DECOM = c("REG", "ALLOCEFF")),
  value_cols  = list(WELF = "Value",
                     DECOM = "Value"),
  long_desc   = list(WELF = "Welfare Decomposition",
                     DECOM = "Allocative efficiency effect"),
  export_sets = TRUE,
  lowercase   = FALSE
)

# Example 3: Apply mapping file for sorting
mapping <- list(
  REG    = c("RestofWorld", "MENA", "EU_28"),
  COLUMN = c("alloc_A1", "tot_E1")
)

save_har(
  data_list   = list(
    WELF  = welfare_data[["A"]],
    DECOM = welfare_data[["A1"]]
  ),
  file_path   = file.path(tempdir(), "output_sorted.har"),
  dimensions  = list(
    WELF  = c("REG", "COLUMN"),
    DECOM = c("REG", "ALLOCEFF")
  ),
  value_cols  = list(
    WELF  = "Value",
    DECOM = "Value"
  ),
  long_desc   = list(
    WELF  = "Welfare Decomposition",
    DECOM = "Allocative efficiency effect"
  ),
  export_sets = TRUE,
  dim_order   = mapping,
  lowercase   = FALSE
)

# Example 4: Rename duplicated dimension names  
# to export as a single structure (e.g., COMMxREGxREG)
har_path <- system.file("extdata", "TAR10-WEL.har", package = "HARplus")
har_data <- load_harx(har_path)
welfare_data <- get_data_by_var("C17", har_data)
welfare_data <- welfare_data[["har_data"]][["C17"]] 

mapping <- list(
  COMM = c("TransComm", "MeatLstk"),
  REG  = c("SSA", "RestofWorld", "SEAsia")  
)

# Export HAR
save_har(
  data_list   = list(TEST = welfare_data),
  file_path   = file.path(tempdir(), "output_single.har"),
  dimensions  = list(TEST = c("COMM", "REG", "REG.1")),
  value_cols  = list(TEST = "Value"),
  long_desc   = list(TEST = "Shock RTMS"),
  dim_rename  = list(TEST = c(COMM = "COMM", SREG = "REG", DREG = "REG.1")),
  export_sets = TRUE,
  dim_order   = mapping,
  lowercase   = FALSE
)
}
\seealso{
\code{\link{load_harx}}, \code{\link{load_sl4x}}
}
\author{
Pattawee Puangchit
}
