#' Approximate Variance of the Median Absolute Deviation (MAD) Estimator.
#' @description
#' This function calculates the approximate (large-sample) variance of the MAD estimator.
#' @param x a numeric vector of data values.
#' @param constant a scale factor. Default choice ensures population mad is equal to the SD for normal distributions.
#' @param use.gld a logical indicating whether to use the flexible Generalised Lambda Distribution (GLD) to estimate the density and distribution function.
#' @param gld.est a character string indicating which estimator of the GLD parameters is to be used if `use.gld=TRUE`.
#' @param ... additional arguments, if desired, to be passed to function density for density estimation (see help file for [density()] for more details).
#'
#' @return A number which is the estimated approximate variance of the mad estimator
#' @details
#' This function calculates an approximate variance for the traditional Median of Absolute Deviations (MAD) estimator.
#' The variance requires estimation of the probability density function and the distribution function which, by default, is carried out
#' using the `density()` and `ecdf()` functions respectively.  An alternative is to use the flexible four-parameter Generalised Lambda Distribution (GLD)
#' to estimate these functions.  This is done by using `use.gld = TRUE` and the default estimator is `gld.est = TM` for
#' Titterington's Method (Titterington, 1985).  Other estimators are available and further details can be found in the [gld::fit.fkml] documentation from the `gld` package
#' (King et al., 2022).  For further details on the approximate variance see Arachchige and Prendergast (2024).
#'
#' @references
#' Arachchige, C.N.P.G., & Prendergast, L.A. (2026). Confidence intervals for median absolute deviations. Communications in Statistics-Simulation and Computation, 55(1), 13-22.
#'
#' King R, Dean B, Klinke S, & van Staden, P (2022). gld: Estimation and Use of the Generalised (Tukey) Lambda Distribution. R package version 2.6.6, <https://CRAN.R-project.org/package=gld>.
#'
#' Titterington, D. M. (1985), Comment on ‘Estimating Parameters in Continuous Univariate Distributions', Journal of the Royal Statistical Society, Series B, 47, 115–116.
#' @import gld
#' @export
#'
#' @examples
#' # Create some data
#' set.seed(1234)
#' x <- rlnorm(100)
#'
#' # Calculate the mad and its variance
#' mad.est <- mad(x)
#' mad.est
#' v.est <- varmad(x)
#' v.est
#'
#' # Calculate an approximate 95% confidence interval for the true mad
#' mad.est + c(-1, 1) * qnorm(0.975) * sqrt(v.est)
varmad <- function(x, constant = 1.4826, use.gld = FALSE, gld.est = "TM", ...){
  if (!is.numeric(x))
    stop("Argument 'x' must be numeric.")

  if (anyNA(x)) {
    count.x.na <- sum(is.na(x))
    warning(paste0(count.x.na), " missing values removed in ",
            deparse(substitute(x)), ".\n")
    x <- stats::na.omit(x)
  }

  n <- length(x)
  med.est <- stats::median(x)
  mad.est <- stats::mad(x, constant = constant)

  if(!use.gld){
    dens <- stats::density(x, ...)
    fFinv <- stats::approx(dens$x, dens$y,
                    xout = med.est + c(-1, 1, 0)*mad.est/constant)$y

    F.est <- stats::ecdf(x)
    FFinv <- F.est(med.est + c(-1, 1)*mad.est/constant)

  } else{
    lambda <- gld::fit.fkml(x, method = gld.est)$lambda
    fFinv <- gld::dgl(med.est + c(-1, 1, 0)*mad.est/constant, lambda1 = lambda)
    FFinv <- gld::pgl(med.est + c(-1, 1)*mad.est/constant, lambda1 = lambda)
  }

  A <- fFinv[1] + fFinv[2]
  C <- fFinv[1] - fFinv[2]
  B <- C^2 + 4*C*fFinv[3]*(1 - FFinv[2] - FFinv[1])

  constant^2*(1/(4 * A^2))*(1 + B/fFinv[3]^2)/n
}
