#' @title Geometric Mean Productivity (GMP)
#' @description
#' This function computes Geometric Mean Productivity (GMP) using any traits (like yield) under stress and non-stress conditions. The lower values of GMP indicates greater tolerance. For more details see Fernandez (1992).
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A data frame with \code{Gen}, \code{GMP}.
#'
#' \deqn{GMP = \sqrt{YN \times YS}}
#'
#' @references Fernandez, G.C.J. (1992). Effective selection criteria for assessing plant stress tolerance. In: Proceedings of the international symposium on adaptation of vegetables and other food crops in temperature and water stress. AVRDC Publication: Tainan, Taiwan: Shanhua: Chapter (25), 257–270.
#' @examples
#' out=GMP(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
GMP <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  prod <- YN * YS
  prod[prod < 0] <- NA_real_
  out <- sqrt(prod)
  return(data.frame(Gen = as.character(Gen), GMP = out, row.names = NULL))
}
