# Tests for detect_regimes main function and evaluation metrics
# Test file: tests/testthat/test-detect.R

# Helper function
generate_test_data <- function(n = 200, changepoints = 100, means = c(0, 3), sd = 1) {
  if (length(changepoints) == 1 && changepoints < n) {
    c(rnorm(changepoints, means[1], sd), 
      rnorm(n - changepoints, means[2], sd))
  } else {
    rnorm(n, means[1], sd)
  }
}

# =============================================================================
# Main Function Tests
# =============================================================================

test_that("detect_regimes returns regime_result class", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- detect_regimes(data)
  
  expect_s3_class(result, "regime_result")
})

test_that("detect_regimes works with different methods", {
  set.seed(123)
  data <- generate_test_data()
  
  methods <- c("pelt", "bocpd", "cusum")
  
  for (method in methods) {
    result <- detect_regimes(data, method = method)
    expect_s3_class(result, "regime_result")
  }
})

test_that("detect_regimes handles ts objects", {
  set.seed(123)
  data <- ts(generate_test_data(), frequency = 12)
  
  result <- detect_regimes(data)
  
  expect_s3_class(result, "regime_result")
})

test_that("detect_regimes handles NA values", {
  set.seed(123)
  data <- generate_test_data()
  data[c(50, 150)] <- NA
  
  expect_warning(result <- detect_regimes(data))
  expect_s3_class(result, "regime_result")
})

test_that("detect_regimes supports different types", {
  set.seed(123)
  data <- generate_test_data()
  
  result_mean <- detect_regimes(data, type = "mean")
  result_var <- detect_regimes(data, type = "variance")
  result_both <- detect_regimes(data, type = "both")
  
  expect_s3_class(result_mean, "regime_result")
  expect_s3_class(result_var, "regime_result")
  expect_s3_class(result_both, "regime_result")
})

test_that("detect_regimes computes segments", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- detect_regimes(data, method = "pelt")
  
  expect_true("segments" %in% names(result))
})

# =============================================================================
# Result Structure Tests
# =============================================================================

test_that("regime_result has required fields", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- detect_regimes(data)
  
  required_fields <- c("changepoints", "n_changepoints", "method")
  expect_true(all(required_fields %in% names(result)))
})

test_that("print method works for regime_result", {
  set.seed(123)
  data <- generate_test_data()
  
  result <- detect_regimes(data)
  
  expect_output(print(result), "Regime")
})

# =============================================================================
# Evaluation Metrics Tests
# =============================================================================

test_that("hausdorff_distance computes correctly", {
  estimated <- c(50, 100, 150)
  true_cp <- c(48, 102, 148)
  
  dist <- hausdorff_distance(estimated, true_cp)
  
  expect_type(dist, "double")
  expect_true(dist >= 0)
  expect_true(dist <= 4)
})

test_that("hausdorff_distance handles edge cases", {
  # Empty sets
  expect_equal(hausdorff_distance(numeric(0), numeric(0)), 0)
  
  # Perfect match
  expect_equal(hausdorff_distance(c(50, 100), c(50, 100)), 0)
})

test_that("precision_score computes correctly", {
  estimated <- c(50, 100, 200)
  true_cp <- c(48, 102)
  
  prec <- precision_score(estimated, true_cp, tolerance = 5)
  
  expect_type(prec, "double")
  expect_true(prec >= 0 && prec <= 1)
})

test_that("recall_score computes correctly", {
  estimated <- c(50)
  true_cp <- c(48, 102)
  
  rec <- recall_score(estimated, true_cp, tolerance = 5)
  
  expect_type(rec, "double")
  expect_true(rec >= 0 && rec <= 1)
})

test_that("f1_score computes correctly", {
  estimated <- c(50, 100)
  true_cp <- c(48, 102)
  
  f1 <- f1_score(estimated, true_cp, tolerance = 5)
  
  expect_type(f1, "double")
  expect_true(f1 >= 0 && f1 <= 1)
})

test_that("rand_index computes correctly", {
  n <- 100
  est_cp <- c(50)
  true_cp <- c(50)
  
  ri <- rand_index(est_cp, true_cp, n)
  
  expect_equal(ri, 1)
})

test_that("adjusted_rand_index computes correctly", {
  n <- 100
  est_cp <- c(50)
  true_cp <- c(50)
  
  ari <- adjusted_rand_index(est_cp, true_cp, n)
  
  expect_true(ari >= 0)
  expect_true(ari <= 1)
})

test_that("covering_metric computes correctly", {
  n <- 100
  est_cp <- c(50)
  true_cp <- c(50)
  
  cov <- covering_metric(est_cp, true_cp, n)
  
  expect_equal(cov, 1)
})

# =============================================================================
# Evaluate Function Tests
# =============================================================================

test_that("evaluate function works", {
  set.seed(123)
  data <- generate_test_data(n = 200, changepoints = 100)
  
  result <- detect_regimes(data)
  eval_result <- evaluate(result, true_changepoints = 100)
  
  expect_s3_class(eval_result, "regime_evaluation")
})

test_that("evaluate handles no changepoints", {
  set.seed(123)
  data <- rnorm(100)
  
  result <- detect_regimes(data, method = "pelt")
  eval_result <- evaluate(result, true_changepoints = numeric(0))
  
  expect_s3_class(eval_result, "regime_evaluation")
})

# =============================================================================
# Method Comparison Tests
# =============================================================================

test_that("compare_methods works", {
  set.seed(123)
  data <- generate_test_data()
  true_cp <- 100
  
  comparison <- compare_methods(
    data = data,
    methods = c("pelt", "cusum"),
    true_changepoints = true_cp
  )
  
  expect_s3_class(comparison, "regime_comparison")
  expect_true("results" %in% names(comparison))
})

test_that("print works for regime_comparison", {
  set.seed(123)
  data <- generate_test_data()
  
  comparison <- compare_methods(
    data = data,
    methods = c("pelt", "cusum"),
    true_changepoints = 100
  )
  
  expect_output(print(comparison), "Method")
})

