#' Find image overlaps
#'
#' \code{findOverlap} - This function searches for the best overlap between
#' each pair of consecutive images (see parameters for more details on the
#' method).
#'
#' @param img_paths (Optional, default = NULL) Character vector specifying all
#' of the individual image paths of interest. This is only used if
#' \code{imgs} is set to NULL. For \code{ImageCorr()} it must have length 2.
#' @param imgs List of images (e.g., provided by the RootDetector). Each image
#' can be a PNG, i.e., an array with 3 dimensions (3 layers each containing a
#' 2-dim. numeric matrix with values between 0 and 1), or a 2-dim. matrix.
#' For \code{ImageCorr()} it must have length 2.
#' @param overlap_px Numeric vector (default NULL) specifying the (likely)
#' widths of the overlaps between two consecutive images. The vector must have
#' one element less than there are images and must not contain any negative
#' values. If NULL, it is set to 1's.
#' @param max_shift_px Numeric vector (default NULL) specifying the maximal
#' deviation in pixels from the \code{overlap_px}, i.e., all possible
#' overlaps in that range from the likely overlap are compared and the best
#' is chosen if it is better than the \code{overlap_px} (see also
#' \code{perc_better_per_px}). If \code{overlap_px} is already exact, then set
#' \code{max_shift_px} to zero(s). If at NULL, it is set to 5 percent of the
#' image widths.
#' @param perc_better_per_px Numeric value (percentage, default 0.01, i.e., 1
#' percent) specifying how much better the correlation of an overlap must be
#' than the \code{overlap_px} per pixel difference, to be selected instead.
#' If set to 0, the overall best correlation determines the overlap.
#' Example: If set to 0.01 = 1 percent, an overlap being 4 pixels away from
#' the specified \code{overlap_px} must have a correlation better/higher than
#' 1.01^4 times the correlation value of \code{overlap_px} to be chosen as the
#' better overlap.
#' @param corr_formula Character value specifying the formula to be used
#' (by default 1) for calculating how good an overlap of two images is, i.e.,
#' how similar the two overlapping images are. Available are the following
#' formulas: \cr
#' - "frac_matches_rgb_intensities": Fraction of matching intensities over all
#' three color channels. Only suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "frac_matches_colors": Fraction of matching colors in the images. Only
#' suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "weighted_matches_b_w": Counts the matches of black and white pixels and
#' weighs them anti-proportional to the black and white pixel frequencies.
#' Both black and white make up half of the correlation score.
#' Only suitable for images with mostly pure black and white pixels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "weighted_matches_colors": Counts the matches per unique color and weighs
#' them anti-proportional to the frequencies of the colors.
#' Each unique color makes up the same fraction of the correlation score.
#' Only suitable for images with few unique colors.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_sqrd_diff_rgb_intensities": One minus the relative squared
#' difference of intensities across all color channels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_abs_diff_rgb_intensities": One minus the relative absolute
#' difference of intensities across all color channels.
#' Ranges from 0 (no matches) to 1 (full match).\cr
#' - "1-rel_eucl_diff_colors" (default): One minus the relative Euclidean
#' differences of colors.
#' Ranges from 0 (no matches) to 1 (full match).
#' @param stitch_direction Character specifying in which order the images
#' should be stitched. Available are: 'left_to_right' (default),
#' 'right_to_left', 'top_to_bottom', and 'bottom_to_top'.
#' @param return_all_results Specify if all checked overlaps with their
#' respective correlation score should be returned (default FALSE).
#' @param show_messages Specify if messages should be depicted (default TRUE).
#'
#' @return \code{findOverlap} Numeric vector containing the best (according
#' to the parameters) widths of the overlaps between the consecutive images.
#' The vector has one element less than there are images. Its attribute "corr"
#' holds the respective correlation value.
#' If \code{return_all_results} is set to true, a list containing a
#' 3-column-matrix for each element in the above mentioned vector, is returned,
#' i.e, for each transition of two consecutive images we have the first
#' column containing the overlaps in pixel, the second column holding the
#' respective correlation values, and the third holding the threshold
#' correlation values accoriding to \code{overlap_px} and
#' \code{perc_better_per_px}.
#' @export
#' @rdname findOverlap
#'
#' @examples
#' # Example of finding the best overlap of two matrices.
#' overlap <- findOverlap(imgs = list(matrix(c(1,0,0,0,
#'                                             0,1,0,0,
#'                                             0,0,1,1), ncol = 4, nrow = 3,
#'                                       byrow = TRUE),
#'                                    matrix(c(0,0,0,0,
#'                                             1,0,0,1,
#'                                             0,1,1,0), ncol = 4, nrow = 3,
#'                                       byrow = TRUE)),
#'                        overlap_px = 1, max_shift_px = 2,
#'                        return_all_results = TRUE)
findOverlap <- function(img_paths = NULL, imgs = NULL,
                        overlap_px = NULL, max_shift_px = NULL,
                        perc_better_per_px = 0.01,
                        corr_formula = "1-rel_eucl_diff_colors",
                        stitch_direction = "left_to_right",
                        return_all_results = FALSE,
                        show_messages = TRUE) {
  # Check input PART I. --------------------------------------------------------
  if(!corr_formula %in% c("frac_matches_rgb_intensities",
                          "frac_matches_colors",
                          "weighted_matches_b_w",
                          "weighted_matches_colors",
                          "1-rel_sqrd_diff_rgb_intensities",
                          "1-rel_abs_diff_rgb_intensities",
                          "1-rel_eucl_diff_colors")){
    stop("Unknown 'corr_formula'.")
  }
  if(!stitch_direction %in% c("bottom_to_top", "top_to_bottom",
                              "right_to_left", "left_to_right")){
    stop("Unknown 'stitch_direction'.")
  }

  # Load the images. -----------------------------------------------------------
  # If only paths to images are provided, load the images.
  if(is.null(imgs)){
    imgs <- lapply(img_paths, png::readPNG)
  } else { # If images are already loaded.
    # If the images are only matrices, give them three layers.
    imgs <- lapply(imgs, function(curr_img){
      if(length(dim(curr_img))==2){
        curr_img <- array(c(curr_img,curr_img,curr_img),
                          dim = c(dim(curr_img),3))
      }
      return(curr_img)
    })
  }

  # Rotate the images according to the stitch direction, to always have
  # left to right. -------------------------------------------------------------
  if(stitch_direction == "bottom_to_top"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 90))
                   })
  } else if(stitch_direction == "top_to_bottom"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 270))
                   })
  } else if(stitch_direction == "right_to_left"){
    imgs <- lapply(1:length(imgs),
                   function(X){
                     return(rotate_clockwise(imgs[[X]], degrees = 180))
                   })
  }
  # Check input PART II. -------------------------------------------------------
  if(is.null(overlap_px)) {
    overlap_px <- rep(1,length(imgs)-1)
  } else if (sum(overlap_px<=0 | overlap_px > sapply(imgs, dim)[2,][-1])>0 ||
             length(overlap_px) != (length(imgs) - 1)){
    stop("No usable 'overlap_px' specified.")
  }
  if(length(imgs)<2){
    stop("At least two images needed.")
  }
  if(is.null(max_shift_px)) {
    max_shift_px <- ceiling(sapply(imgs, dim)[2,][-1]*0.05)
  } else if (sum(max_shift_px<0)>0 ||
             length(max_shift_px) != (length(imgs) - 1)){
    stop("No usable 'max_shift_px' specified.")
  }

  # Start searching for the best overlap. --------------------------------------
  # Vector with best computed shift for each transition.
  if(return_all_results){
    all_results <- vector("list", length(imgs)-1)
  } else {
    best_shift_all <- c()
    best_corr_all <- c()
  }

  for(i in 1:(length(imgs)-1)){ # Look at each transition.
    img1 <- imgs[[i]]
    img2 <- imgs[[i+1]]
    img_width1 <- dim(img1)[2]
    img_width2 <- dim(img2)[2]
    min_img_width <- min(img_width1,img_width2)
    # Compute all correlation values. ------------------------------------------
    range_shifts <- max(overlap_px[i]-max_shift_px[i], 1):
      min(overlap_px[i]+max_shift_px[i], min_img_width)
    corr_shifts <- rep(NA, length(range_shifts))
    for (j in 1:length(range_shifts)){
      # Define the overlap region (stitch left to right).
      # Columns of overlap in current stitched and next image:
      ovrlp_cols1 <- (img_width1 - range_shifts[j] + 1):img_width1
      ovrlp_cols2 <- 1:range_shifts[j]
      # Extract overlapping regions
      overlap1 <- img1[,ovrlp_cols1, , drop = FALSE]
      overlap2 <- img2[,ovrlp_cols2, , drop = FALSE]
      corr_shifts[j] <- imageCorr(imgs = list(overlap1, overlap2),
                                  corr_formula = corr_formula)
    }
    pos_ovrlp_i <- which(range_shifts == overlap_px[i])
    # Take into account how much better other shifts are with respect to their
    # distance from the specified overlap.
    threshold_corr <- corr_shifts[pos_ovrlp_i]*
      (1+perc_better_per_px)^(c(seq((pos_ovrlp_i-1),0),
                                seq(0,(length(range_shifts)-pos_ovrlp_i))[-1]))
    if(return_all_results){
      all_results[[i]] <- cbind(range_shifts, corr_shifts, threshold_corr)
      colnames(all_results[[i]]) <- c("overlap_px", "corr_value",
                                      "threshold_corr")
    } else {
      corr_with_pbpp <- corr_shifts - threshold_corr
      ovrlp_candidates <- which(corr_with_pbpp == max(corr_with_pbpp))
      # If there is more than one candidate choose the one closest to the
      # specified overlap.
      if(length(ovrlp_candidates)>1){
        ovrlp_candidates <- ovrlp_candidates[which.min(abs(ovrlp_candidates-
                                                             pos_ovrlp_i))]
      }
      best_shift_all[i] <- range_shifts[ovrlp_candidates]
      best_corr_all[i] <- corr_shifts[ovrlp_candidates]
    }
  }
  if(return_all_results){
    return(all_results)
  } else {
    attr(best_shift_all, "corr") <- best_corr_all
    if(show_messages && sum(best_shift_all != overlap_px)>0){
      message(paste0("Choosing overlaps (", paste(best_shift_all,
                                                  collapse = ","),
                     ") instead of likely overlaps (",
                     paste(overlap_px, collapse = ","),")."))
    }
    return(best_shift_all)
  }
}
#' Find image overlaps
#'
#' \code{imageCorr} - This function computes the similarity/correlation of two
#' images.
#'
#' @return \code{imageCorr} Numeric value (correlations score).
#' @export
#' @rdname findOverlap
#'
#' @examples
#' # Example of computing the similarity of two images/matrices.
#' imageCorr(imgs = list(matrix(c(1,0,0,1,
#'                                1,1,1,1,
#'                                0,1,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'                       matrix(c(1,0,0,0,
#'                                1,1,1,1,
#'                                0,1,1,1), ncol = 4, nrow = 3, byrow = TRUE)),
#'                       corr_formula = "weighted_matches_colors")
imageCorr <- function(img_paths = NULL, imgs = NULL,
                      corr_formula = "1-rel_eucl_diff_colors"){
  # Check input PART I. --------------------------------------------------------
  if(!corr_formula %in% c("frac_matches_rgb_intensities",
                          "frac_matches_colors",
                          "weighted_matches_b_w",
                          "weighted_matches_colors",
                          "1-rel_sqrd_diff_rgb_intensities",
                          "1-rel_abs_diff_rgb_intensities",
                          "1-rel_eucl_diff_colors")){
    stop("Unknown 'corr_formula'.")
  }

  # Load the images. -----------------------------------------------------------
  if(is.null(imgs)){
    imgs <- lapply(img_paths, png::readPNG)
  } else { # If images are already loaded.
    # If the images are only matrices, give them three layers.
    imgs <- lapply(imgs, function(curr_img){
      if(length(dim(curr_img))==2){
        curr_img <- array(c(curr_img,curr_img,curr_img),
                          dim = c(dim(curr_img),3))
      }
      return(curr_img)
    })
  }
  # Check input PART II. -------------------------------------------------------
  if(length(imgs)!=2){
    stop("Please provide exactly two images.")
  }
  if(length(dim(imgs[[1]]))!=3 || length(dim(imgs[[2]]))!=3 ||
     dim(imgs[[1]])[3]!=3 || dim(imgs[[2]])[3]!=3){
    stop("The images must have 2 or 3 dimensions (third dimension must be 3).")
  }
  if(sum(dim(imgs[[1]]) != dim(imgs[[2]]))>0){
    stop("Dimensions of images do not match.")
  }
  img1 <- imgs[[1]]
  img2 <- imgs[[2]]
  height <- dim(img1)[1]
  width <- dim(img1)[2]

  # Compute correlation. -------------------------------------------------------
  if (corr_formula == "frac_matches_rgb_intensities") {
    # Check for intensity matches in each color channel.
    corr <- sum(img1 == img2) / (3*height*width)
  } else if (corr_formula == "frac_matches_colors") {
    # Check for full RGB match.
    corr <- sum(img1[,,1] == img2[,,1] &
                img1[,,2] == img2[,,2] &
                img1[,,3] == img2[,,3]) / (height*width)
  } else if (corr_formula == "weighted_matches_b_w") {
    # Get all unique colors and their frequencies.
    all_colors1 <- matrix(img1, ncol = 3, byrow = FALSE)
    all_colors2 <- matrix(img2, ncol = 3, byrow = FALSE)
    all_colors <- rbind(all_colors1, all_colors2)
    unique_colors <- rbind(c(1,1,1), c(0,0,0)) # Only black and white relevant.
    unique_counts1 <- apply(unique_colors, 1, function(color) {
      sum(colSums(t(all_colors1) == color) == 3)  # Check for full RGB match.
    })
    unique_counts2 <- apply(unique_colors, 1, function(color) {
      sum(colSums(t(all_colors2) == color) == 3)  # Check for full RGB match.
    })
    unique_counts <- unique_counts1 + unique_counts2
    unique_min_counts <- apply(rbind(unique_counts1, unique_counts2),2,min)
    # Only give weights to colors that could match.
    unique_weights <- rep(0, length(unique_min_counts))
    # Weight is inverse of possible number of matches (=half the count).
    unique_weights[unique_min_counts>0] <- 1/
      (unique_counts[unique_min_counts>0]/2)
    unique_matches <- apply(unique_colors, 1, function(color) {
      # Check for full RGB match on both images.
        sum(colSums(t(all_colors1) == color) == 3 &
            colSums(t(all_colors2) == color) == 3)
      })
    # Ensure that every color can make up an equal fraction of the correlation
    # value.
    corr <- sum(unique_weights * unique_matches)/nrow(unique_colors)
  }  else if (corr_formula == "weighted_matches_colors") {
    # Get all unique colors and their frequencies.
    all_colors1 <- matrix(img1, ncol = 3, byrow = FALSE)
    all_colors2 <- matrix(img2, ncol = 3, byrow = FALSE)
    all_colors <- rbind(all_colors1, all_colors2)
    unique_colors <- unique(all_colors)
    unique_counts1 <- apply(unique_colors, 1, function(color) {
      sum(colSums(t(all_colors1) == color) == 3)  # Check for full RGB match.
    })
    unique_counts2 <- apply(unique_colors, 1, function(color) {
      sum(colSums(t(all_colors2) == color) == 3)  # Check for full RGB match.
    })
    unique_counts <- unique_counts1 + unique_counts2
    unique_min_counts <- apply(rbind(unique_counts1, unique_counts2),2,min)
    # Only give weights to colors that could match.
    unique_weights <- rep(0, length(unique_min_counts))
    # Weight is inverse of possible number of matches (=half the count).
    unique_weights[unique_min_counts>0] <- 1/
      (unique_counts[unique_min_counts>0]/2)
    unique_matches <- sapply(1:nrow(unique_colors), function(X) {
      # Check for full RGB match on both images.
      sum(colSums(t(all_colors1) == unique_colors[X,]) == 3 &
            colSums(t(all_colors2) == unique_colors[X,]) == 3)
    })
    # Ensure that every color can make up an equal fraction of the correlation
    # value.
    corr <- sum(unique_weights * unique_matches)/nrow(unique_colors)
  } else if (corr_formula == "1-rel_sqrd_diff_rgb_intensities") {
    # Check for intensity differences in each color channel.
    corr <- 1- (sum((img1 - img2)^2) / (3*height*width))
  } else if (corr_formula == "1-rel_abs_diff_rgb_intensities") {
    # Check for intensity differences in each color channel.
    corr <- 1- (sum(abs(img1 - img2)) / (3*height*width))
  } else if (corr_formula == "1-rel_eucl_diff_colors") {
    # Check for full RGB match.
    corr <- 0
    for(i in 1:dim(img1)[1]){
      for(j in 1:dim(img1)[2]){
        corr <- corr + sqrt(sum((img1[i,j,] - img2[i,j,])^2))
      }
    }
    corr <- 1- (corr / (sqrt(3)*height*width))
  }
  # Optional formulas to add:
  # 1.) to do case denominator == 0
  # corr <- sum(img1 * img2) / sqrt(sum(img1^2) * sum(img2^2))
  # 2.)
  # corr <- sum(ifelse(img1 - img2 == 0, 1, 0))/shift
  return(corr)
}
