#' R² for Cluster Solutions after Buttler & Fickel (1995)
#'
#' Computes the proportion of explained distance variation (R²) for a given
#' clustering solution using a distance matrix derived from the
#' Buttler-Fickel distance. The statistic reflects how well the clustering
#' partitions the total pairwise distance structure.
#'
#' The R² is defined as:
#' \deqn{R^2 = 1 - \frac{D_{\text{within}}}{D_{\text{total}}}}
#' where \eqn{D_{\text{total}}} is the sum of all pairwise distances and
#' \eqn{D_{\text{within}}} is the sum of distances within clusters.
#'
#' @param D A distance object of class \code{dist}, usually computed via
#'   \code{buttler_fickel_dist()}.
#' @param cluster An integer or factor vector of cluster assignments,
#'   typically obtained from \code{cutree()} or another clustering method.
#'
#' @return A numeric value between 0 and 1 indicating the proportion of
#'   explained distance variation. Higher values represent better cluster fit.
#'
#' @examples
#' df <- data.frame(
#'   sex    = factor(c("m","f","m","f")),
#'   height = c(180, 165, 170, 159),
#'   age    = c(25, 32, 29, 28)
#' )
#'
#' types <- c("nominal", "metric", "metric")
#'
#' D <- buttler_fickel_dist(df, types)
#' hc <- hclust(D)
#' cl <- cutree(hc, k = 2)
#'
#' bf_R2(D, cl)
#'
#' @export
bf_R2 <- function(D, cluster) {
  if (!inherits(D, "dist")) {
    stop("D must be a dist object.")
  }

  Dmat <- as.matrix(D)

  # total pairwise distance sum
  total <- sum(Dmat[upper.tri(Dmat)])

  # within-cluster distance sum
  clusters <- unique(cluster)
  within <- 0

  for (cl in clusters) {
    idx <- which(cluster == cl)
    if (length(idx) > 1) {
      within <- within + sum(Dmat[idx, idx][upper.tri(Dmat[idx, idx])])
    }
  }

  R2 <- 1 - within / total
  return(R2)
}
