#pragma once
#include <RcppArmadillo.h>

/**
 *  Rank-completion of Sigma matrices via the max-exp rule
 */
arma::cube rank_complete_s(const arma::cube& S_cube,     // p×p×n
                           const arma::mat& X,           // n×q
                           const arma::mat& Gamma_prev,  // p×(k−1)
                           const arma::mat& B)           // q×(k−1)
{
  const arma::uword p = S_cube.n_rows;
  const arma::uword n = S_cube.n_slices;
  const arma::uword k_minus1 = Gamma_prev.n_cols;  // k−1

  // ---- trivial case : no previous directions -----------------------------
  if (k_minus1 == 0) return S_cube;

  // ---- dimension checks ---------------------------------------------------
  if (X.n_rows != n) Rcpp::stop("rank_complete_s_maxexp: X must have n rows.");
  if (B.n_rows != X.n_cols || B.n_cols != k_minus1)
    Rcpp::stop("rank_complete_s_maxexp: B must be q×(k-1) and match X.");

  // ---- compute   E = exp(X %*% B)   and  a = max(E) -----------------------
  arma::mat E = arma::exp(X * B);  // n × (k−1)
  const double a = 2 * E.max();    // scalar

  // ---- pre-allocate result cube ------------------------------------------
  arma::cube S_tilde(p, p, n, arma::fill::zeros);

  // ---- main loop over subjects -------------------------------------------
  for (arma::uword i = 0; i < n; ++i) {
    arma::rowvec Ei = E.row(i);   // 1 × (k−1)
    arma::vec di = (a - Ei).t();  // (k−1) × 1

    arma::mat correction =
        Gamma_prev * arma::diagmat(di) * Gamma_prev.t();  // p × p

    S_tilde.slice(i) = S_cube.slice(i) + correction;
  }

  return S_tilde;
}

arma::cube deflate_s_projection(const arma::cube& S_cube,
                                const arma::mat& Gamma_prev) {
  const arma::uword p = S_cube.n_rows;
  const arma::uword n = S_cube.n_slices;

  if (Gamma_prev.n_cols == 0) return S_cube;

  arma::mat Q, R;
  arma::qr_econ(Q, R, Gamma_prev);  // Q: p×(k−1), orthonormal columns
  arma::mat P = Q * Q.t();          // true orthogonal projector
  arma::mat I = arma::eye(p, p);
  arma::mat M = I - P;

  arma::cube S_tilde(p, p, n, arma::fill::zeros);
  for (arma::uword i = 0; i < n; ++i) {
    arma::mat Si = S_cube.slice(i);
    S_tilde.slice(i) = M * Si * M;  // (I-P) S (I-P)
    S_tilde.slice(i) = 0.5 * (S_tilde.slice(i) + S_tilde.slice(i).t());  // sym
  }
  return S_tilde;
}

arma::mat inv_sqrt_matrix(const arma::mat& A, const int k) {
  // Force symmetry before the eigendecomposition to avoid warnings/failures
  arma::mat A_sym = 0.5 * (A + A.t());

  arma::vec eigval;
  arma::mat eigvec;
  const bool ok = arma::eig_sym(eigval, eigvec, A_sym, "dc");
  if (!ok) Rcpp::stop("inv_sqrt_matrix: eigen-decomposition failed.");

  // Guard against non-positive eigenvalues (rank deficiency / numerical noise)
  const double lambda_max = eigval.max();
  const double floor =
      std::max(lambda_max * 1e-6, 1e-12);  // cap condition number, avoid zeros
  eigval.transform([floor](double v) { return (v < floor) ? floor : v; });

  arma::vec inv_sqrt_eigval = 1.0 / arma::sqrt(eigval.head(k));
  arma::mat inv_sqrt_D = arma::diagmat(inv_sqrt_eigval);
  arma::mat eigvec_k = eigvec.cols(0, k - 1);

  return eigvec_k * inv_sqrt_D * eigvec_k.t();
}

arma::cube rank_complete_multiply(const arma::cube& S_cube,     // p×p×n
                                  const arma::mat& X,           // n×q
                                  const arma::mat& Gamma_prev,  // p×(k−1)
                                  const arma::mat& B)           // q×(k−1)
{
  const arma::uword n = S_cube.n_slices;
  const arma::uword p = S_cube.n_rows;

  if (Gamma_prev.n_cols == 0) return S_cube;

  if (X.n_rows != n) Rcpp::stop("rank_complete_multiply: X must have n rows.");
  if (B.n_rows != X.n_cols || B.n_cols != Gamma_prev.n_cols)
    Rcpp::stop(
        "rank_complete_multiply: B must be q×(k-1) and match Gamma_prev.");

  arma::mat E = arma::exp(X * B);  // n × (k−1)

  // ---- pre-allocate result cube ------------------------------------------
  arma::cube S_tilde(p, p, n, arma::fill::zeros);

  // ---- main loop over subjects -------------------------------------------
  for (arma::uword i = 0; i < n; ++i) {
    arma::rowvec Ei = E.row(i);  // 1 × (k−1)

    arma::mat Shat = Gamma_prev * arma::diagmat(Ei) * Gamma_prev.t();  // p × p
    // Shat = 0.5 * (Shat + Shat.t());  // enforce symmetry for eig_sym
    arma::mat Ssqrtinv = inv_sqrt_matrix(Shat, Gamma_prev.n_cols);

    S_tilde.slice(i) = Ssqrtinv * S_cube.slice(i) * Ssqrtinv;
  }

  return S_tilde;
}

/*──────────────────────────────────────────────────────────────────────────────
  Rank–completion step used in the CAP flip-flop:

      Ŷᵢ   …  p×p data matrices   (cube  p×p×n)
      Γ     …  p×(k-1)  previously-estimated directions
      B     …  q×(k-1)  coefficient matrix,   row 0 = intercepts β_{j0}

  For each i = 1..n
  ────────────────────────────────────────────────────────────────────────────
    1.   Ŷᵢ  = Yᵢ  −  Yᵢ Γ Γᵀ                       (residual)
    2.   U D Vᵀ = svd( Ŷᵢ )
    3.   D̃  = diag(  D₁ … D_{p-(k-1)},  √e^{β_{10}}, …, √e^{β_{(k-1)0}} )
    4.   Ỹᵢ = U D̃ Vᵀ
  ────────────────────────────────────────────────────────────────────────────
  Returns a cube of the same shape as Y with slices Ỹᵢ.
────────────────────────────────────────────────────────────────────────────*/

arma::cube rank_complete_y(const arma::cube& Y,          // p × p × n
                           const arma::mat& Gamma_prev,  // p × (k-1)
                           const arma::mat& B)           // q × (k-1)
{
  const arma::uword p = Y.n_rows;
  const arma::uword n = Y.n_slices;
  const arma::uword k_minus = Gamma_prev.n_cols;  // = k−1

  if (k_minus != 0 && B.n_cols != k_minus)
    Rcpp::stop("B must have k-1 columns, matching Gamma_prev.");

  arma::cube Y_tilde(p, p, n);

  // intercepts β_{j0}  (first row of B)
  arma::rowvec beta0 = (k_minus == 0) ? arma::rowvec() : B.row(0);

  for (arma::uword i = 0; i < n; ++i) {
    arma::mat Yi = Y.slice(i);

    // 1. residual   Ŷᵢ
    arma::mat Yhat =
        (k_minus == 0) ? Yi : Yi - Yi * Gamma_prev * Gamma_prev.t();

    // 2. thin SVD   Ŷᵢ = U D Vᵀ
    arma::mat U, V;
    arma::vec D;
    arma::svd(U, D, V, Yhat, "std");

    // 3. build D̃
    arma::vec Dtilde(p, arma::fill::zeros);
    arma::uword keep = p - k_minus;  // #original sing.vals

    if (keep > 0) Dtilde.head(keep) = D.head(keep);

    for (arma::uword j = 0; j < k_minus; ++j)
      Dtilde(keep + j) = std::sqrt(std::exp(beta0(j)));

    // 4. reconstruct
    Y_tilde.slice(i) = U * arma::diagmat(Dtilde) * V.t();
  }

  return Y_tilde;
}
