% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddbs_predicates.R
\name{ddbs_predicate}
\alias{ddbs_predicate}
\title{Spatial predicate operations}
\usage{
ddbs_predicate(
  x,
  y,
  predicate = "intersects",
  conn = NULL,
  id_x = NULL,
  id_y = NULL,
  sparse = TRUE,
  distance = NULL,
  quiet = FALSE
)
}
\arguments{
\item{x}{An \code{sf} spatial object. Alternatively, it can be a string with the
name of a table with geometry column within the DuckDB database \code{conn}.
Data is returned from this object.}

\item{y}{An \code{sf} spatial object. Alternatively, it can be a string with the
name of a table with geometry column within the DuckDB database \code{conn}.
Data is returned from this object.}

\item{predicate}{A geometry predicate function. Defaults to \code{intersects},
a wrapper of \code{ST_Intersects}. See details for other options.}

\item{conn}{A connection object to a DuckDB database. If \code{NULL}, the function
runs on a temporary DuckDB database.}

\item{id_x}{Character; optional name of the column in \code{x} whose values will
be used to name the list elements. If \code{NULL}, integer row numbers of \code{x} are used.}

\item{id_y}{Character; optional name of the column in \code{y} whose values will
replace the integer indices returned in each element of the list.}

\item{sparse}{A logical value. If \code{TRUE}, it returns a sparse index list. If \code{FALSE},
it returns a dense logical matrix.}

\item{distance}{a numeric value specifying the distance for ST_DWithin. Units correspond to
the coordinate system of the geometry (e.g. degrees or meters)}

\item{quiet}{A logical value. If \code{TRUE}, suppresses any informational messages.
Defaults to \code{FALSE}.}
}
\value{
A \strong{list} of length equal to the number of rows in \code{x}.
\itemize{
\item Each element contains:
\itemize{
\item \strong{integer vector} of row indices of \code{y} that satisfy the predicate with
the corresponding geometry of \code{x}, or
\item \strong{character vector} if \code{id_y} is supplied.
}
\item The names of the list elements:
\itemize{
\item are integer row numbers of \code{x}, or
\item the values of \code{id_x} if provided.
}
}

If there's no match between \code{x} and \code{y} it returns \code{NULL}
}
\description{
Computes spatial relationships between two geometry datasets using DuckDB's
spatial extension. Returns a list where each element corresponds to a row of
\code{x}, containing the indices (or IDs) of rows in \code{y} that satisfy the specified
spatial predicate.
}
\details{
This function provides a unified interface to all spatial predicate operations
in DuckDB's spatial extension. It performs pairwise comparisons between all
geometries in \code{x} and \code{y} using the specified predicate.
\subsection{Available Predicates}{
\itemize{
\item \strong{intersects}: Geometries share at least one point
\item \strong{covers}: Geometry \code{x} completely covers geometry \code{y}
\item \strong{touches}: Geometries share a boundary but interiors do not intersect
\item \strong{disjoint}: Geometries have no points in common
\item \strong{within}: Geometry \code{x} is completely inside geometry \code{y}
\item \strong{dwithin}: Geometry \code{x} is completely within a distance of geometry \code{y}
\item \strong{contains}: Geometry \code{x} completely contains geometry \code{y}
\item \strong{overlaps}: Geometries share some but not all points
\item \strong{crosses}: Geometries have some interior points in common
\item \strong{equals}: Geometries are spatially equal
\item \strong{covered_by}: Geometry \code{x} is completely covered by geometry \code{y}
\item \strong{intersects_extent}: Bounding boxes of geometries intersect (faster but less precise)
\item \strong{contains_properly}: Geometry \code{x} contains geometry \code{y} without boundary contact
\item \strong{within_properly}: Geometry \code{x} is within geometry \code{y} without boundary contact
}

If \code{x} or \code{y} are not DuckDB tables, they are automatically copied into a
temporary in-memory DuckDB database (unless a connection is supplied via \code{conn}).

\code{id_x} or \code{id_y} may be used to replace the default integer indices with the
values of an identifier column in \code{x} or \code{y}, respectively.
}
}
\examples{
\dontrun{
## Load packages
library(duckspatial)
library(dplyr)
library(sf)

## create in-memory DuckDB database
conn <- ddbs_create_conn(dbdir = "memory")

## read countries data, and rivers
countries_sf <- read_sf(system.file("spatial/countries.geojson", package = "duckspatial")) |>
  filter(CNTR_ID \%in\% c("PT", "ES", "FR", "IT"))
rivers_sf <- st_read(system.file("spatial/rivers.geojson", package = "duckspatial")) |>
  st_transform(st_crs(countries_sf))

## Store in DuckDB
ddbs_write_vector(conn, countries_sf, "countries")
ddbs_write_vector(conn, rivers_sf, "rivers")

## Example 1: Check which rivers intersect each country
ddbs_predicate(countries_sf, rivers_sf, predicate = "intersects", conn)

## Example 2: Find neighboring countries
ddbs_predicate(countries_sf, countries_sf, predicate = "touches",
               id_x = "NAME_ENGL", id_y = "NAME_ENGL")

## Example 3: Find rivers that don't intersect countries
ddbs_predicate(countries_sf, rivers_sf, predicate = "disjoint",
               id_x = "NAME_ENGL", id_y = "RIVER_NAME")

## Example 4: Use table names inside duckdb
ddbs_predicate("countries", "rivers", predicate = "within", conn, "NAME_ENGL")
}
}
