#' Cronbach's Coefficient Alpha
#'
#' @description
#' Compute Cronbach's coefficient alpha and the associated standard error of
#' measurement (SEM) for a set of items.
#'
#' @param x A data frame or matrix containing item responses, with rows as
#'   respondents (subjects) and columns as items.
#'
#' @details
#' Cronbach's alpha is an estimate of the internal consistency reliability of a
#' test. This implementation:
#' \itemize{
#'   \item removes rows with any missing values using \code{stats::na.exclude()},
#'   \item computes the sample covariance matrix of the items,
#'   \item uses the classical formula
#'   \deqn{\alpha = \frac{k}{k-1} \left(1 - \frac{\sum \sigma_i^2}{\sigma_X^2}\right),}
#'   where \eqn{k} is the number of items, \eqn{\sigma_i^2} are item variances,
#'   and \eqn{\sigma_X^2} is the variance of the total score,
#'   \item computes SEM as \eqn{\text{SD}(X) \sqrt{1 - \alpha}}.
#' }
#'
#' @return A named list with the following elements:
#' \describe{
#'   \item{alpha}{Cronbach's coefficient alpha.}
#'   \item{sem}{Standard error of measurement (SEM) based on alpha.}
#' }
#'
#' @examples
#' data(data.u)
#' alpha(data.u)
#'
#' @export
alpha <- function(x) {
  # basic checks ---------------------------------------------------------------
  if (is.null(x)) {
    stop("`x` must not be NULL.")
  }

  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  # drop rows with missing values (same behavior as original code) ------------
  x <- stats::na.exclude(x)

  # ensure matrix numeric
  x <- as.matrix(x)
  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  # number of items -----------------------------------------------------------
  ni <- ncol(x)
  if (ni < 2L) {
    stop("`x` must have at least 2 items (columns) to compute Cronbach's alpha.")
  }

  # covariance matrix ---------------------------------------------------------
  cov_x <- stats::cov(x)

  # sum of variance of each item (sum of diagonals)
  vi <- sum(diag(cov_x))

  # total score variance (sum of all covariances)
  vt <- sum(cov_x)

  if (isTRUE(all.equal(vt, 0))) {
    stop("Total score variance is zero; Cronbach's alpha is undefined.")
  }

  # alpha ---------------------------------------------------------------------
  alpha <- ni / (ni - 1) * (vt - vi) / vt

  # SEM (based on total score SD and alpha) -----------------------------------
  total_scores <- rowSums(x)
  sem <- stats::sd(total_scores) * sqrt(1 - alpha)

  # return --------------------------------------------------------------------
  return(list(alpha = alpha, sem = sem))
}
