#' KR-21
#'
#' @description
#' Compute the KR-21 reliability coefficient for
#' dichotomously scored items (0/1), assuming equal item difficulty.
#'
#' @param x A data frame or matrix of item responses, with rows as persons
#'   and columns as items. Items are assumed to be dichotomous (0/1).
#'
#' @details
#' KR-21 is a simplified alternative to KR-20, assuming equal item difficulty.
#' Rows containing missing values are removed using \code{stats::na.exclude()}.
#'
#' @return A single numeric value: the KR-21 reliability coefficient.
#'
#' @examples
#' data(data.u)
#' kr21(data.u)
#'
#' @export
kr21 <- function(x) {

  # basic checks ---------------------------------------------------------------
  if (is.null(x)) {
    stop("`x` must not be NULL.")
  }
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  x <- stats::na.exclude(x)
  x <- as.matrix(x)

  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  ni   <- ncol(x)
  np <- nrow(x)

  if (ni < 2L) {
    stop("`x` must have at least 2 items (columns) to compute KR-21.")
  }
  if (np < 2L) {
    stop("`x` must have at least 2 persons (rows) to compute KR-21.")
  }

  # total-score variance -------------------------------------------------------
  total_scores <- rowSums(x)
  varsum <- stats::var(total_scores)

  if (isTRUE(all.equal(varsum, 0))) {
    stop("Total score variance is zero; KR-21 is undefined.")
  }

  # mean item score across all responses --------------------------------------
  mean_score <- sum(x) / (ni * np)

  pqsum <- ni * mean_score * (1 - mean_score)

  # KR-21 ----------------------------------------------------------------------
  kr21 <- ni / (ni - 1) * (1 - pqsum / varsum)

  return(kr21)
}
