#' Spearman–Brown Prophecy Formula
#'
#' @description
#' Compute the predicted test reliability after changing test length, or compute
#' the required test-length ratio to achieve a desired reliability, using the
#' Spearman–Brown prophecy formula.
#'
#' @param rxx A numeric value indicating the original reliability
#'   (must be between 0 and 1, exclusive).
#' @param input A numeric value indicating either:
#' \itemize{
#'   \item the ratio of new test length to original test length (if type = "r"), or
#'   \item the desired reliability of the new test (if type = "l").
#' }
#' @param type Character string specifying the calculation type:
#' \describe{
#'   \item{"r"}{Compute new reliability given the length ratio.}
#'   \item{"l"}{Compute the length ratio required to achieve a desired reliability.}
#' }
#'
#' @details
#' The Spearman–Brown prophecy formula is:
#' \deqn{r_{yy} = \frac{k r_{xx}}{1 + (k - 1) r_{xx}},}
#' where \eqn{r_{xx}} is the original reliability and \eqn{k} is the ratio of the
#' new test length to the original test length.
#'
#' Solving for \eqn{k} gives:
#' \deqn{k = \frac{r_{yy}(1 - r_{xx})}{r_{xx}(1 - r_{yy})}.}
#'
#' @return
#' A named list depending on \code{type}:
#' \describe{
#'   \item{reliability}{Predicted reliability of the new test (if type = "r").}
#'   \item{ratio}{Required ratio of new test length to original test length
#'   (if type = "l").}
#' }
#'
#' @examples
#' spearman_brown(0.7, 3.86, "r")
#' spearman_brown(0.7, 0.90, "l")
#'
#' @export
spearman_brown <- function(rxx, input, type = c("r", "l")) {

  type <- match.arg(type)

  # input checks ---------------------------------------------------------------
  if (!is.numeric(rxx) || length(rxx) != 1L) {
    stop("`rxx` must be a single numeric value.")
  }
  if (rxx <= 0 || rxx >= 1) {
    stop("`rxx` must be strictly between 0 and 1.")
  }

  if (!is.numeric(input) || length(input) != 1L) {
    stop("`input` must be a single numeric value.")
  }

  # compute new test reliability ----------------------------------------------
  if (type == "r") {

    if (input <= 0) {
      stop("Length ratio must be positive when type = 'r'.")
    }

    ryy <- input * rxx / (1 + (input - 1) * rxx)

    return(list(reliability = ryy))
  }

  # compute required length ratio ---------------------------------------------
  if (type == "l") {

    if (input <= 0 || input >= 1) {
      stop("Desired reliability must be strictly between 0 and 1 when type = 'l'.")
    }

    ratio <- input * (1 - rxx) / (rxx * (1 - input))

    return(list(ratio = ratio))
  }
}
