% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ekw.R
\name{dekw}
\alias{dekw}
\title{Density of the Exponentiated Kumaraswamy (EKw) Distribution}
\usage{
dekw(x, alpha = 1, beta = 1, lambda = 1, log = FALSE)
}
\arguments{
\item{x}{Vector of quantiles (values between 0 and 1).}

\item{alpha}{Shape parameter \code{alpha} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{beta}{Shape parameter \code{beta} > 0. Can be a scalar or a vector.
Default: 1.0.}

\item{lambda}{Shape parameter \code{lambda} > 0 (exponent parameter).
Can be a scalar or a vector. Default: 1.0.}

\item{log}{Logical; if \code{TRUE}, the logarithm of the density is
returned (\eqn{\log(f(x))}). Default: \code{FALSE}.}
}
\value{
A vector of density values (\eqn{f(x)}) or log-density values
(\eqn{\log(f(x))}). The length of the result is determined by the recycling
rule applied to the arguments (\code{x}, \code{alpha}, \code{beta},
\code{lambda}). Returns \code{0} (or \code{-Inf} if
\code{log = TRUE}) for \code{x} outside the interval (0, 1), or
\code{NaN} if parameters are invalid (e.g., \code{alpha <= 0},
\code{beta <= 0}, \code{lambda <= 0}).
}
\description{
Computes the probability density function (PDF) for the Exponentiated
Kumaraswamy (EKw) distribution with parameters \code{alpha} (\eqn{\alpha}),
\code{beta} (\eqn{\beta}), and \code{lambda} (\eqn{\lambda}).
This distribution is defined on the interval (0, 1).
}
\details{
The probability density function (PDF) of the Exponentiated Kumaraswamy (EKw)
distribution is given by:
\deqn{
f(x; \alpha, \beta, \lambda) = \lambda \alpha \beta x^{\alpha-1} (1 - x^\alpha)^{\beta-1} \bigl[1 - (1 - x^\alpha)^\beta \bigr]^{\lambda - 1}
}
for \eqn{0 < x < 1}.

The EKw distribution is a special case of the five-parameter
Generalized Kumaraswamy (GKw) distribution (\code{\link{dgkw}}) obtained
by setting the parameters \eqn{\gamma = 1} and \eqn{\delta = 0}.
When \eqn{\lambda = 1}, the EKw distribution reduces to the standard
Kumaraswamy distribution.
}
\examples{
\donttest{
# Example values
x_vals <- c(0.2, 0.5, 0.8)
alpha_par <- 2.0
beta_par <- 3.0
lambda_par <- 1.5 # Exponent parameter

# Calculate density
densities <- dekw(x_vals, alpha_par, beta_par, lambda_par)
print(densities)

# Calculate log-density
log_densities <- dekw(x_vals, alpha_par, beta_par, lambda_par, log = TRUE)
print(log_densities)
# Check: should match log(densities)
print(log(densities))

# Compare with dgkw setting gamma = 1, delta = 0
densities_gkw <- dgkw(x_vals, alpha_par, beta_par, gamma = 1.0, delta = 0.0,
                      lambda = lambda_par)
print(paste("Max difference:", max(abs(densities - densities_gkw)))) # Should be near zero

# Plot the density for different lambda values
curve_x <- seq(0.01, 0.99, length.out = 200)
curve_y1 <- dekw(curve_x, alpha = 2, beta = 3, lambda = 0.5) # less peaked
curve_y2 <- dekw(curve_x, alpha = 2, beta = 3, lambda = 1.0) # standard Kw
curve_y3 <- dekw(curve_x, alpha = 2, beta = 3, lambda = 2.0) # more peaked

plot(curve_x, curve_y2, type = "l", main = "EKw Density Examples (alpha=2, beta=3)",
     xlab = "x", ylab = "f(x)", col = "red", ylim = range(0, curve_y1, curve_y2, curve_y3))
lines(curve_x, curve_y1, col = "blue")
lines(curve_x, curve_y3, col = "green")
legend("topright", legend = c("lambda=0.5", "lambda=1.0 (Kw)", "lambda=2.0"),
       col = c("blue", "red", "green"), lty = 1, bty = "n")
}

}
\references{
Nadarajah, S., Cordeiro, G. M., & Ortega, E. M. (2012). The exponentiated
Kumaraswamy distribution. \emph{Journal of the Franklin Institute}, \emph{349}(3),

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{dgkw}} (parent distribution density),
\code{\link{pekw}}, \code{\link{qekw}}, \code{\link{rekw}} (other EKw functions),
}
\author{
Lopes, J. E.
}
\keyword{density}
\keyword{distribution}
