/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 
#define H5F_FRIEND     

#include "H5private.h"   
#include "H5Cpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5MFprivate.h" 
#include "H5SLprivate.h" 

static herr_t H5C__autoadjust__ageout(H5F_t *f, double hit_rate, enum H5C_resize_status *status_ptr,
                                      size_t *new_max_cache_size_ptr, bool write_permitted);
static herr_t H5C__autoadjust__ageout__cycle_epoch_marker(H5C_t *cache_ptr);
static herr_t H5C__autoadjust__ageout__evict_aged_out_entries(H5F_t *f, bool write_permitted);
static herr_t H5C__autoadjust__ageout__insert_new_marker(H5C_t *cache_ptr);
static herr_t H5C__flush_invalidate_ring(H5F_t *f, H5C_ring_t ring, unsigned flags);
static herr_t H5C__serialize_ring(H5F_t *f, H5C_ring_t ring);

herr_t
H5C__auto_adjust_cache_size(H5F_t *f, bool write_permitted)
{
    H5C_t                 *cache_ptr             = f->shared->cache;
    bool                   reentrant_call        = false;
    bool                   inserted_epoch_marker = false;
    size_t                 new_max_cache_size    = 0;
    size_t                 old_max_cache_size    = 0;
    size_t                 new_min_clean_size    = 0;
    size_t                 old_min_clean_size    = 0;
    double                 hit_rate;
    enum H5C_resize_status status    = in_spec; 
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(cache_ptr);
    assert(cache_ptr->cache_accesses >= cache_ptr->resize_ctl.epoch_length);
    assert(0.0 <= cache_ptr->resize_ctl.min_clean_fraction);
    assert(cache_ptr->resize_ctl.min_clean_fraction <= 100.0);

    
    if (cache_ptr->resize_in_progress) {
        reentrant_call = true;
        HGOTO_DONE(SUCCEED);
    } 

    cache_ptr->resize_in_progress = true;

    if (!cache_ptr->resize_enabled)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Auto cache resize disabled");

    assert((cache_ptr->resize_ctl.incr_mode != H5C_incr__off) ||
           (cache_ptr->resize_ctl.decr_mode != H5C_decr__off));

    if (H5C_get_cache_hit_rate(cache_ptr, &hit_rate) != SUCCEED)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Can't get hit rate");

    assert((0.0 <= hit_rate) && (hit_rate <= 1.0));

    switch (cache_ptr->resize_ctl.incr_mode) {
        case H5C_incr__off:
            if (cache_ptr->size_increase_possible)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "size_increase_possible but H5C_incr__off?!?!?");
            break;

        case H5C_incr__threshold:
            if (hit_rate < cache_ptr->resize_ctl.lower_hr_threshold) {
                if (!cache_ptr->size_increase_possible)
                    status = increase_disabled;
                else if (cache_ptr->max_cache_size >= cache_ptr->resize_ctl.max_size) {
                    assert(cache_ptr->max_cache_size == cache_ptr->resize_ctl.max_size);
                    status = at_max_size;
                }
                else if (!cache_ptr->cache_full)
                    status = not_full;
                else {
                    new_max_cache_size =
                        (size_t)(((double)(cache_ptr->max_cache_size)) * cache_ptr->resize_ctl.increment);

                    
                    if (new_max_cache_size > cache_ptr->resize_ctl.max_size)
                        new_max_cache_size = cache_ptr->resize_ctl.max_size;

                    
                    if (cache_ptr->resize_ctl.apply_max_increment &&
                        ((cache_ptr->max_cache_size + cache_ptr->resize_ctl.max_increment) <
                         new_max_cache_size))
                        new_max_cache_size = cache_ptr->max_cache_size + cache_ptr->resize_ctl.max_increment;

                    status = increase;
                }
            }
            break;

        default:
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unknown incr_mode");
    }

    

    if (((cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out) ||
         (cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out_with_threshold)) &&
        (cache_ptr->epoch_markers_active < cache_ptr->resize_ctl.epochs_before_eviction)) {

        if (H5C__autoadjust__ageout__insert_new_marker(cache_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't insert new epoch marker");

        inserted_epoch_marker = true;
    }

    

    if (status == in_spec) {
        switch (cache_ptr->resize_ctl.decr_mode) {
            case H5C_decr__off:
                break;

            case H5C_decr__threshold:
                if (hit_rate > cache_ptr->resize_ctl.upper_hr_threshold) {
                    if (!cache_ptr->size_decrease_possible)
                        status = decrease_disabled;
                    else if (cache_ptr->max_cache_size <= cache_ptr->resize_ctl.min_size) {
                        assert(cache_ptr->max_cache_size == cache_ptr->resize_ctl.min_size);
                        status = at_min_size;
                    }
                    else {
                        new_max_cache_size =
                            (size_t)(((double)(cache_ptr->max_cache_size)) * cache_ptr->resize_ctl.decrement);

                        
                        if (new_max_cache_size < cache_ptr->resize_ctl.min_size)
                            new_max_cache_size = cache_ptr->resize_ctl.min_size;

                        
                        if (cache_ptr->resize_ctl.apply_max_decrement &&
                            ((cache_ptr->resize_ctl.max_decrement + new_max_cache_size) <
                             cache_ptr->max_cache_size))
                            new_max_cache_size =
                                cache_ptr->max_cache_size - cache_ptr->resize_ctl.max_decrement;

                        status = decrease;
                    }
                }
                break;

            case H5C_decr__age_out_with_threshold:
            case H5C_decr__age_out:
                if (!inserted_epoch_marker) {
                    if (!cache_ptr->size_decrease_possible)
                        status = decrease_disabled;
                    else {
                        if (H5C__autoadjust__ageout(f, hit_rate, &status, &new_max_cache_size,
                                                    write_permitted) < 0)
                            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ageout code failed");
                    } 
                }     
                break;

            default:
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unknown incr_mode");
        }
    }

    
    if (((cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out) ||
         (cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out_with_threshold)) &&
        !inserted_epoch_marker)
        
        if (H5C__autoadjust__ageout__cycle_epoch_marker(cache_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "error cycling epoch marker");

    if ((status == increase) || (status == decrease)) {
        old_max_cache_size = cache_ptr->max_cache_size;
        old_min_clean_size = cache_ptr->min_clean_size;

        new_min_clean_size =
            (size_t)((double)new_max_cache_size * (cache_ptr->resize_ctl.min_clean_fraction));

        
        assert(new_min_clean_size <= new_max_cache_size);
        assert(cache_ptr->resize_ctl.min_size <= new_max_cache_size);
        assert(new_max_cache_size <= cache_ptr->resize_ctl.max_size);

        cache_ptr->max_cache_size = new_max_cache_size;
        cache_ptr->min_clean_size = new_min_clean_size;

        if (status == increase)
            cache_ptr->cache_full = false;
        else if (status == decrease)
            cache_ptr->size_decreased = true;

        
        if (cache_ptr->flash_size_increase_possible) {
            switch (cache_ptr->resize_ctl.flash_incr_mode) {
                case H5C_flash_incr__off:
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL,
                                "flash_size_increase_possible but H5C_flash_incr__off?!");
                    break;

                case H5C_flash_incr__add_space:
                    cache_ptr->flash_size_increase_threshold =
                        (size_t)(((double)(cache_ptr->max_cache_size)) *
                                 (cache_ptr->resize_ctl.flash_threshold));
                    break;

                default: 
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Unknown flash_incr_mode?!?!?");
                    break;
            }
        }
    }

    if (cache_ptr->resize_ctl.rpt_fcn != NULL)
        (cache_ptr->resize_ctl.rpt_fcn)(cache_ptr, H5C__CURR_AUTO_RESIZE_RPT_FCN_VER, hit_rate, status,
                                        old_max_cache_size, new_max_cache_size, old_min_clean_size,
                                        new_min_clean_size);

    if (H5C_reset_cache_hit_rate_stats(cache_ptr) < 0)
        
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_reset_cache_hit_rate_stats failed");

done:
    
    assert(cache_ptr->resize_in_progress);
    if (!reentrant_call)
        cache_ptr->resize_in_progress = false;
    assert((!reentrant_call) || (cache_ptr->resize_in_progress));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__autoadjust__ageout(H5F_t *f, double hit_rate, enum H5C_resize_status *status_ptr,
                        size_t *new_max_cache_size_ptr, bool write_permitted)
{
    H5C_t *cache_ptr = f->shared->cache;
    size_t test_size;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(cache_ptr);
    assert((status_ptr) && (*status_ptr == in_spec));
    assert((new_max_cache_size_ptr) && (*new_max_cache_size_ptr == 0));

    
    if (cache_ptr->epoch_markers_active > cache_ptr->resize_ctl.epochs_before_eviction)
        if (H5C__autoadjust__ageout__remove_excess_markers(cache_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't remove excess epoch markers");

    if ((cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out) ||
        ((cache_ptr->resize_ctl.decr_mode == H5C_decr__age_out_with_threshold) &&
         (hit_rate >= cache_ptr->resize_ctl.upper_hr_threshold))) {

        if (cache_ptr->max_cache_size > cache_ptr->resize_ctl.min_size) {
            
            if (H5C__autoadjust__ageout__evict_aged_out_entries(f, write_permitted) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "error flushing aged out entries");

            
            if (cache_ptr->index_size < cache_ptr->max_cache_size) {
                if (cache_ptr->resize_ctl.apply_empty_reserve) {
                    test_size =
                        (size_t)(((double)cache_ptr->index_size) / (1 - cache_ptr->resize_ctl.empty_reserve));
                    if (test_size < cache_ptr->max_cache_size) {
                        *status_ptr             = decrease;
                        *new_max_cache_size_ptr = test_size;
                    }
                }
                else {
                    *status_ptr             = decrease;
                    *new_max_cache_size_ptr = cache_ptr->index_size;
                }

                if (*status_ptr == decrease) {
                    
                    if (*new_max_cache_size_ptr < cache_ptr->resize_ctl.min_size)
                        *new_max_cache_size_ptr = cache_ptr->resize_ctl.min_size;

                    
                    if ((cache_ptr->resize_ctl.apply_max_decrement) &&
                        ((cache_ptr->resize_ctl.max_decrement + *new_max_cache_size_ptr) <
                         cache_ptr->max_cache_size))
                        *new_max_cache_size_ptr =
                            cache_ptr->max_cache_size - cache_ptr->resize_ctl.max_decrement;
                }
            }
        }
        else
            *status_ptr = at_min_size;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__autoadjust__ageout__cycle_epoch_marker(H5C_t *cache_ptr)
{
    int    i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (cache_ptr->epoch_markers_active <= 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "No active epoch markers on entry?!?!?");

    
    i = cache_ptr->epoch_marker_ringbuf[cache_ptr->epoch_marker_ringbuf_first];
    cache_ptr->epoch_marker_ringbuf_first =
        (cache_ptr->epoch_marker_ringbuf_first + 1) % (H5C__MAX_EPOCH_MARKERS + 1);
    if (cache_ptr->epoch_marker_ringbuf_size <= 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ring buffer underflow");

    cache_ptr->epoch_marker_ringbuf_size -= 1;
    if (cache_ptr->epoch_marker_active[i] != true)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unused marker in LRU?!?");

    H5C__DLL_REMOVE((&((cache_ptr->epoch_markers)[i])), (cache_ptr)->LRU_head_ptr, (cache_ptr)->LRU_tail_ptr,
                    (cache_ptr)->LRU_list_len, (cache_ptr)->LRU_list_size, (FAIL))

    
    assert(cache_ptr->epoch_markers[i].addr == (haddr_t)i);
    assert(cache_ptr->epoch_markers[i].next == NULL);
    assert(cache_ptr->epoch_markers[i].prev == NULL);

    cache_ptr->epoch_marker_ringbuf_last =
        (cache_ptr->epoch_marker_ringbuf_last + 1) % (H5C__MAX_EPOCH_MARKERS + 1);
    cache_ptr->epoch_marker_ringbuf[cache_ptr->epoch_marker_ringbuf_last] = i;
    if (cache_ptr->epoch_marker_ringbuf_size >= H5C__MAX_EPOCH_MARKERS)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ring buffer overflow");

    cache_ptr->epoch_marker_ringbuf_size += 1;

    H5C__DLL_PREPEND(&(cache_ptr->epoch_markers[i]), cache_ptr->LRU_head_ptr, cache_ptr->LRU_tail_ptr,
                     cache_ptr->LRU_list_len, cache_ptr->LRU_list_size, FAIL)
done:

    FUNC_LEAVE_NOAPI(ret_value)

} 

static herr_t
H5C__autoadjust__ageout__evict_aged_out_entries(H5F_t *f, bool write_permitted)
{
    H5C_t             *cache_ptr = f->shared->cache;
    size_t             eviction_size_limit;
    size_t             bytes_evicted = 0;
    bool               prev_is_dirty = false;
    bool               restart_scan;
    H5C_cache_entry_t *entry_ptr;
    H5C_cache_entry_t *next_ptr;
    H5C_cache_entry_t *prev_ptr;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(cache_ptr);

    
    if (cache_ptr->resize_ctl.apply_max_decrement)
        eviction_size_limit = cache_ptr->resize_ctl.max_decrement;
    else
        eviction_size_limit = cache_ptr->index_size; 

    if (write_permitted) {
        restart_scan = false;
        entry_ptr    = cache_ptr->LRU_tail_ptr;
        while (entry_ptr != NULL && entry_ptr->type->id != H5AC_EPOCH_MARKER_ID &&
               bytes_evicted < eviction_size_limit) {
            bool skipping_entry = false;

            assert(!(entry_ptr->is_protected));
            assert(!(entry_ptr->is_read_only));
            assert((entry_ptr->ro_ref_count) == 0);

            next_ptr = entry_ptr->next;
            prev_ptr = entry_ptr->prev;

            if (prev_ptr != NULL)
                prev_is_dirty = prev_ptr->is_dirty;

            if (entry_ptr->is_dirty) {
                assert(!entry_ptr->prefetched_dirty);

                
                if (entry_ptr->tag_info && entry_ptr->tag_info->corked)
                    skipping_entry = true;
                else {
                    
                    cache_ptr->entries_removed_counter = 0;
                    cache_ptr->last_entry_removed_ptr  = NULL;

                    if (H5C__flush_single_entry(f, entry_ptr, H5C__NO_FLAGS_SET) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush entry");

                    if (cache_ptr->entries_removed_counter > 1 ||
                        cache_ptr->last_entry_removed_ptr == prev_ptr)
                        restart_scan = true;
                } 
            }     
            else if (!entry_ptr->prefetched_dirty) {
                bytes_evicted += entry_ptr->size;

                if (H5C__flush_single_entry(
                        f, entry_ptr, H5C__FLUSH_INVALIDATE_FLAG | H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush entry");
            } 
            else {
                assert(!entry_ptr->is_dirty);
                assert(entry_ptr->prefetched_dirty);

                skipping_entry = true;
            } 

            if (prev_ptr != NULL) {
                if (skipping_entry)
                    entry_ptr = prev_ptr;
                else if (restart_scan || (prev_ptr->is_dirty != prev_is_dirty) ||
                         (prev_ptr->next != next_ptr) || (prev_ptr->is_protected) || (prev_ptr->is_pinned)) {
                    
                    restart_scan = false;
                    entry_ptr    = cache_ptr->LRU_tail_ptr;

                    H5C__UPDATE_STATS_FOR_LRU_SCAN_RESTART(cache_ptr);
                } 
                else
                    entry_ptr = prev_ptr;
            } 
            else
                entry_ptr = NULL;
        } 

        
    } 
    else  {
        
        assert(H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS);
        entry_ptr = cache_ptr->LRU_tail_ptr;
        while (entry_ptr != NULL && ((entry_ptr->type)->id != H5AC_EPOCH_MARKER_ID) &&
               (bytes_evicted < eviction_size_limit)) {
            assert(!(entry_ptr->is_protected));

            prev_ptr = entry_ptr->prev;

            if (!(entry_ptr->is_dirty) && !(entry_ptr->prefetched_dirty))
                if (H5C__flush_single_entry(
                        f, entry_ptr, H5C__FLUSH_INVALIDATE_FLAG | H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush clean entry");

            
            entry_ptr = prev_ptr;
        } 
    }     

    if (cache_ptr->index_size < cache_ptr->max_cache_size)
        cache_ptr->cache_full = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__autoadjust__ageout__insert_new_marker(H5C_t *cache_ptr)
{
    int    i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (cache_ptr->epoch_markers_active >= cache_ptr->resize_ctl.epochs_before_eviction)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Already have a full complement of markers");

    
    i = 0;
    while ((cache_ptr->epoch_marker_active)[i] && i < H5C__MAX_EPOCH_MARKERS)
        i++;
    if (i >= H5C__MAX_EPOCH_MARKERS)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Can't find unused marker");

    assert(((cache_ptr->epoch_markers)[i]).addr == (haddr_t)i);
    assert(((cache_ptr->epoch_markers)[i]).next == NULL);
    assert(((cache_ptr->epoch_markers)[i]).prev == NULL);

    (cache_ptr->epoch_marker_active)[i] = true;

    cache_ptr->epoch_marker_ringbuf_last =
        (cache_ptr->epoch_marker_ringbuf_last + 1) % (H5C__MAX_EPOCH_MARKERS + 1);
    (cache_ptr->epoch_marker_ringbuf)[cache_ptr->epoch_marker_ringbuf_last] = i;
    if (cache_ptr->epoch_marker_ringbuf_size >= H5C__MAX_EPOCH_MARKERS)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ring buffer overflow");

    cache_ptr->epoch_marker_ringbuf_size += 1;

    H5C__DLL_PREPEND(&(cache_ptr->epoch_markers[i]), cache_ptr->LRU_head_ptr, cache_ptr->LRU_tail_ptr,
                     cache_ptr->LRU_list_len, cache_ptr->LRU_list_size, FAIL)

    cache_ptr->epoch_markers_active += 1;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__autoadjust__ageout__remove_all_markers(H5C_t *cache_ptr)
{
    int    ring_buf_index;
    int    i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    while (cache_ptr->epoch_markers_active > 0) {
        

        ring_buf_index = cache_ptr->epoch_marker_ringbuf_first;
        i              = (cache_ptr->epoch_marker_ringbuf)[ring_buf_index];

        cache_ptr->epoch_marker_ringbuf_first =
            (cache_ptr->epoch_marker_ringbuf_first + 1) % (H5C__MAX_EPOCH_MARKERS + 1);

        if (cache_ptr->epoch_marker_ringbuf_size <= 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ring buffer underflow");
        cache_ptr->epoch_marker_ringbuf_size -= 1;

        if (cache_ptr->epoch_marker_active[i] != true)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unused marker in LRU?!?");

        
        H5C__DLL_REMOVE(&(cache_ptr->epoch_markers[i]), cache_ptr->LRU_head_ptr, cache_ptr->LRU_tail_ptr,
                        cache_ptr->LRU_list_len, cache_ptr->LRU_list_size, FAIL)

        
        cache_ptr->epoch_marker_active[i] = false;

        assert(cache_ptr->epoch_markers[i].addr == (haddr_t)i);
        assert(cache_ptr->epoch_markers[i].next == NULL);
        assert(cache_ptr->epoch_markers[i].prev == NULL);

        
        cache_ptr->epoch_markers_active -= 1;

        assert(cache_ptr->epoch_markers_active == cache_ptr->epoch_marker_ringbuf_size);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__autoadjust__ageout__remove_excess_markers(H5C_t *cache_ptr)
{
    int    ring_buf_index;
    int    i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);

    if (cache_ptr->epoch_markers_active <= cache_ptr->resize_ctl.epochs_before_eviction)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "no excess markers on entry");

    while (cache_ptr->epoch_markers_active > cache_ptr->resize_ctl.epochs_before_eviction) {
        
        ring_buf_index = cache_ptr->epoch_marker_ringbuf_first;
        i              = (cache_ptr->epoch_marker_ringbuf)[ring_buf_index];

        cache_ptr->epoch_marker_ringbuf_first =
            (cache_ptr->epoch_marker_ringbuf_first + 1) % (H5C__MAX_EPOCH_MARKERS + 1);

        if (cache_ptr->epoch_marker_ringbuf_size <= 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "ring buffer underflow");
        cache_ptr->epoch_marker_ringbuf_size -= 1;

        if (cache_ptr->epoch_marker_active[i] != true)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "unused marker in LRU?!?");

        
        H5C__DLL_REMOVE(&(cache_ptr->epoch_markers[i]), cache_ptr->LRU_head_ptr, cache_ptr->LRU_tail_ptr,
                        cache_ptr->LRU_list_len, cache_ptr->LRU_list_size, FAIL)

        
        cache_ptr->epoch_marker_active[i] = false;

        assert(cache_ptr->epoch_markers[i].addr == (haddr_t)i);
        assert(cache_ptr->epoch_markers[i].next == NULL);
        assert(cache_ptr->epoch_markers[i].prev == NULL);

        
        cache_ptr->epoch_markers_active -= 1;

        assert(cache_ptr->epoch_markers_active == cache_ptr->epoch_marker_ringbuf_size);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__flash_increase_cache_size(H5C_t *cache_ptr, size_t old_entry_size, size_t new_entry_size)
{
    size_t                 new_max_cache_size = 0;
    size_t                 old_max_cache_size = 0;
    size_t                 new_min_clean_size = 0;
    size_t                 old_min_clean_size = 0;
    size_t                 space_needed;
    enum H5C_resize_status status = flash_increase; 
    double                 hit_rate;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);
    assert(cache_ptr->flash_size_increase_possible);
    assert(new_entry_size > cache_ptr->flash_size_increase_threshold);
    assert(old_entry_size < new_entry_size);

    if (old_entry_size >= new_entry_size)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "old_entry_size >= new_entry_size");

    space_needed = new_entry_size - old_entry_size;
    if (((cache_ptr->index_size + space_needed) > cache_ptr->max_cache_size) &&
        (cache_ptr->max_cache_size < cache_ptr->resize_ctl.max_size)) {
        switch (cache_ptr->resize_ctl.flash_incr_mode) {
            case H5C_flash_incr__off:
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL,
                            "flash_size_increase_possible but H5C_flash_incr__off?!");
                break;

            case H5C_flash_incr__add_space:
                if (cache_ptr->index_size < cache_ptr->max_cache_size) {
                    assert((cache_ptr->max_cache_size - cache_ptr->index_size) < space_needed);
                    space_needed -= cache_ptr->max_cache_size - cache_ptr->index_size;
                }
                space_needed       = (size_t)(((double)space_needed) * cache_ptr->resize_ctl.flash_multiple);
                new_max_cache_size = cache_ptr->max_cache_size + space_needed;
                break;

            default: 
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Unknown flash_incr_mode?!?!?");
                break;
        }

        if (new_max_cache_size > cache_ptr->resize_ctl.max_size)
            new_max_cache_size = cache_ptr->resize_ctl.max_size;
        assert(new_max_cache_size > cache_ptr->max_cache_size);

        new_min_clean_size = (size_t)((double)new_max_cache_size * cache_ptr->resize_ctl.min_clean_fraction);
        assert(new_min_clean_size <= new_max_cache_size);

        old_max_cache_size = cache_ptr->max_cache_size;
        old_min_clean_size = cache_ptr->min_clean_size;

        cache_ptr->max_cache_size = new_max_cache_size;
        cache_ptr->min_clean_size = new_min_clean_size;

        
        assert(cache_ptr->flash_size_increase_possible);

        switch (cache_ptr->resize_ctl.flash_incr_mode) {
            case H5C_flash_incr__off:
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL,
                            "flash_size_increase_possible but H5C_flash_incr__off?!");
                break;

            case H5C_flash_incr__add_space:
                cache_ptr->flash_size_increase_threshold =
                    (size_t)((double)cache_ptr->max_cache_size * cache_ptr->resize_ctl.flash_threshold);
                break;

            default: 
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Unknown flash_incr_mode?!?!?");
                break;
        }

        

        if (cache_ptr->resize_ctl.rpt_fcn != NULL) {
            
            if (H5C_get_cache_hit_rate(cache_ptr, &hit_rate) != SUCCEED)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Can't get hit rate");

            (cache_ptr->resize_ctl.rpt_fcn)(cache_ptr, H5C__CURR_AUTO_RESIZE_RPT_FCN_VER, hit_rate, status,
                                            old_max_cache_size, new_max_cache_size, old_min_clean_size,
                                            new_min_clean_size);
        }

        if (H5C_reset_cache_hit_rate_stats(cache_ptr) < 0)
            
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_reset_cache_hit_rate_stats failed");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__flush_invalidate_cache(H5F_t *f, unsigned flags)
{
    H5C_t     *cache_ptr;
    H5C_ring_t ring;
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(cache_ptr->slist_ptr);
    assert(cache_ptr->slist_enabled);

#ifdef H5C_DO_SANITY_CHECKS
    {
        int32_t  i;
        uint32_t index_len        = 0;
        uint32_t slist_len        = 0;
        size_t   index_size       = (size_t)0;
        size_t   clean_index_size = (size_t)0;
        size_t   dirty_index_size = (size_t)0;
        size_t   slist_size       = (size_t)0;

        assert(cache_ptr->index_ring_len[H5C_RING_UNDEFINED] == 0);
        assert(cache_ptr->index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
        assert(cache_ptr->clean_index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
        assert(cache_ptr->dirty_index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
        assert(cache_ptr->slist_ring_len[H5C_RING_UNDEFINED] == 0);
        assert(cache_ptr->slist_ring_size[H5C_RING_UNDEFINED] == (size_t)0);

        for (i = H5C_RING_USER; i < H5C_RING_NTYPES; i++) {
            index_len += cache_ptr->index_ring_len[i];
            index_size += cache_ptr->index_ring_size[i];
            clean_index_size += cache_ptr->clean_index_ring_size[i];
            dirty_index_size += cache_ptr->dirty_index_ring_size[i];

            slist_len += cache_ptr->slist_ring_len[i];
            slist_size += cache_ptr->slist_ring_size[i];
        } 

        assert(cache_ptr->index_len == index_len);
        assert(cache_ptr->index_size == index_size);
        assert(cache_ptr->clean_index_size == clean_index_size);
        assert(cache_ptr->dirty_index_size == dirty_index_size);
        assert(cache_ptr->slist_len == slist_len);
        assert(cache_ptr->slist_size == slist_size);
    }
#endif 

    
    if (cache_ptr->epoch_markers_active > 0)
        if (H5C__autoadjust__ageout__remove_all_markers(cache_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "error removing all epoch markers");

    
    ring = H5C_RING_USER;
    while (ring < H5C_RING_NTYPES) {
        if (H5C__flush_invalidate_ring(f, ring, flags) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "flush invalidate ring failed");
        ring++;
    } 

#ifndef NDEBUG
    
    if (!(flags & H5C__EVICT_ALLOW_LAST_PINS_FLAG)) {
        assert(cache_ptr->index_size == 0);
        assert(cache_ptr->clean_index_size == 0);
        assert(cache_ptr->pel_len == 0);
        assert(cache_ptr->pel_size == 0);
    } 
    else {
        H5C_cache_entry_t *entry_ptr; 
        unsigned           u;         

        
        
        for (u = H5C_RING_USER; u < H5C_RING_SB; u++) {
            assert(cache_ptr->index_ring_len[u] == 0);
            assert(cache_ptr->index_ring_size[u] == 0);
            assert(cache_ptr->clean_index_ring_size[u] == 0);
        } 

        
        entry_ptr = cache_ptr->pel_head_ptr;
        while (entry_ptr) {
            
            assert(entry_ptr->ring == H5C_RING_SB);

            
            entry_ptr = entry_ptr->next;
        } 
    }     

    assert(cache_ptr->dirty_index_size == 0);
    assert(cache_ptr->slist_len == 0);
    assert(cache_ptr->slist_size == 0);
    assert(cache_ptr->pl_len == 0);
    assert(cache_ptr->pl_size == 0);
    assert(cache_ptr->LRU_list_len == 0);
    assert(cache_ptr->LRU_list_size == 0);
#endif

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__flush_invalidate_ring(H5F_t *f, H5C_ring_t ring, unsigned flags)
{
    H5C_t             *cache_ptr;
    bool               restart_slist_scan;
    uint32_t           protected_entries = 0;
    int32_t            i;
    uint32_t           cur_ring_pel_len;
    uint32_t           old_ring_pel_len;
    unsigned           cooked_flags;
    unsigned           evict_flags;
    H5SL_node_t       *node_ptr       = NULL;
    H5C_cache_entry_t *entry_ptr      = NULL;
    H5C_cache_entry_t *next_entry_ptr = NULL;
#ifdef H5C_DO_SANITY_CHECKS
    uint32_t initial_slist_len  = 0;
    size_t   initial_slist_size = 0;
#endif 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(cache_ptr->slist_enabled);
    assert(cache_ptr->slist_ptr);
    assert(ring > H5C_RING_UNDEFINED);
    assert(ring < H5C_RING_NTYPES);

    assert(cache_ptr->epoch_markers_active == 0);

    
    cooked_flags = flags & H5C__FLUSH_CLEAR_ONLY_FLAG;
    evict_flags  = flags & H5C__EVICT_ALLOW_LAST_PINS_FLAG;

    

    
    entry_ptr        = cache_ptr->pel_head_ptr;
    cur_ring_pel_len = 0;
    while (entry_ptr != NULL) {
        assert(entry_ptr->ring >= ring);
        if (entry_ptr->ring == ring)
            cur_ring_pel_len++;

        entry_ptr = entry_ptr->next;
    } 
    old_ring_pel_len = cur_ring_pel_len;

    while (cache_ptr->index_ring_len[ring] > 0) {
        

#ifdef H5C_DO_SANITY_CHECKS
        
        initial_slist_len  = cache_ptr->slist_len;
        initial_slist_size = cache_ptr->slist_size;

        
        cache_ptr->slist_len_increase  = 0;
        cache_ptr->slist_size_increase = 0;
#endif 

        
        cache_ptr->slist_changed = false;

        
        restart_slist_scan = true;
        while (restart_slist_scan || (node_ptr != NULL)) {
            if (restart_slist_scan) {
                restart_slist_scan = false;

                
                node_ptr = H5SL_first(cache_ptr->slist_ptr);
                if (node_ptr == NULL)
                    
                    break;

                
                next_entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
                if (NULL == next_entry_ptr)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "next_entry_ptr == NULL ?!?!");

                assert(next_entry_ptr->is_dirty);
                assert(next_entry_ptr->in_slist);
                assert(next_entry_ptr->ring >= ring);
            } 

            entry_ptr = next_entry_ptr;

            
            assert(entry_ptr != NULL);
            assert(entry_ptr->in_slist);
            assert(entry_ptr->is_dirty);
            assert(entry_ptr->ring >= ring);

            
            node_ptr = H5SL_next(node_ptr);
            if (node_ptr != NULL) {
                next_entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
                if (NULL == next_entry_ptr)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "next_entry_ptr == NULL ?!?!");

                assert(next_entry_ptr->is_dirty);
                assert(next_entry_ptr->in_slist);
                assert(next_entry_ptr->ring >= ring);
                assert(entry_ptr != next_entry_ptr);
            } 
            else
                next_entry_ptr = NULL;

            
            if (((!entry_ptr->flush_me_last) ||
                 ((entry_ptr->flush_me_last) && (cache_ptr->num_last_entries >= cache_ptr->slist_len))) &&
                (entry_ptr->flush_dep_nchildren == 0) && (entry_ptr->ring == ring)) {
                if (entry_ptr->is_protected) {
                    
                    protected_entries++;
                } 
                else if (entry_ptr->is_pinned) {
                    if (H5C__flush_single_entry(f, entry_ptr, H5C__DURING_FLUSH_FLAG) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "dirty pinned entry flush failed");

                    if (cache_ptr->slist_changed) {
                        
                        restart_slist_scan       = true;
                        cache_ptr->slist_changed = false;
                        H5C__UPDATE_STATS_FOR_SLIST_SCAN_RESTART(cache_ptr);
                    } 
                }     
                else {
                    if (H5C__flush_single_entry(f, entry_ptr,
                                                (cooked_flags | H5C__DURING_FLUSH_FLAG |
                                                 H5C__FLUSH_INVALIDATE_FLAG |
                                                 H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG)) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "dirty entry flush destroy failed");

                    if (cache_ptr->slist_changed) {
                        
                        restart_slist_scan       = true;
                        cache_ptr->slist_changed = false;
                        H5C__UPDATE_STATS_FOR_SLIST_SCAN_RESTART(cache_ptr);
                    } 
                }     
            }         
        }             

#ifdef H5C_DO_SANITY_CHECKS
        

        if (node_ptr == NULL) {
            assert(cache_ptr->slist_len ==
                   (uint32_t)((int32_t)initial_slist_len + cache_ptr->slist_len_increase));
            assert(cache_ptr->slist_size ==
                   (size_t)((ssize_t)initial_slist_size + cache_ptr->slist_size_increase));
        } 
#endif    

        

        
        cache_ptr->entries_loaded_counter    = 0;
        cache_ptr->entries_inserted_counter  = 0;
        cache_ptr->entries_relocated_counter = 0;

        next_entry_ptr = cache_ptr->il_head;
        while (next_entry_ptr != NULL) {
            entry_ptr = next_entry_ptr;
            assert(entry_ptr->ring >= ring);

            next_entry_ptr = entry_ptr->il_next;

            if (((!entry_ptr->flush_me_last) ||
                 (entry_ptr->flush_me_last && (cache_ptr->num_last_entries >= cache_ptr->slist_len))) &&
                (entry_ptr->flush_dep_nchildren == 0) && (entry_ptr->ring == ring)) {

                if (entry_ptr->is_protected) {
                    
                    protected_entries++;

                    if (!entry_ptr->in_slist)
                        assert(!(entry_ptr->is_dirty));
                } 
                else if (!entry_ptr->is_pinned) {
                    
                    cache_ptr->entry_watched_for_removal = next_entry_ptr;
                    if (H5C__flush_single_entry(f, entry_ptr,
                                                (cooked_flags | H5C__DURING_FLUSH_FLAG |
                                                 H5C__FLUSH_INVALIDATE_FLAG |
                                                 H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG)) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Entry flush destroy failed");

                    
                    if (((NULL != next_entry_ptr) && (NULL == cache_ptr->entry_watched_for_removal)) ||
                        (cache_ptr->entries_loaded_counter > 0) ||
                        (cache_ptr->entries_inserted_counter > 0) ||
                        (cache_ptr->entries_relocated_counter > 0)) {

                        next_entry_ptr = cache_ptr->il_head;

                        cache_ptr->entries_loaded_counter    = 0;
                        cache_ptr->entries_inserted_counter  = 0;
                        cache_ptr->entries_relocated_counter = 0;

                        H5C__UPDATE_STATS_FOR_INDEX_SCAN_RESTART(cache_ptr);
                    } 
                    else
                        cache_ptr->entry_watched_for_removal = NULL;
                } 
            }     
        }         

        
        old_ring_pel_len = cur_ring_pel_len;
        entry_ptr        = cache_ptr->pel_head_ptr;
        cur_ring_pel_len = 0;

        while (entry_ptr != NULL) {
            assert(entry_ptr->ring >= ring);

            if (entry_ptr->ring == ring)
                cur_ring_pel_len++;

            entry_ptr = entry_ptr->next;
        } 

        
        if ((cur_ring_pel_len > 0) && (cur_ring_pel_len >= old_ring_pel_len)) {
            
            if (evict_flags)
                HGOTO_DONE(true);

            HGOTO_ERROR(
                H5E_CACHE, H5E_CANTFLUSH, FAIL,
                "Pinned entry count not decreasing, cur_ring_pel_len = %d, old_ring_pel_len = %d, ring = %d",
                (int)cur_ring_pel_len, (int)old_ring_pel_len, (int)ring);
        } 

        assert(protected_entries == cache_ptr->pl_len);

        if ((protected_entries > 0) && (protected_entries == cache_ptr->index_len))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL,
                        "Only protected entries left in cache, protected_entries = %d",
                        (int)protected_entries);
    } 

    
    for (i = (int)H5C_RING_UNDEFINED; i <= (int)ring; i++) {
        assert(cache_ptr->index_ring_len[i] == 0);
        assert(cache_ptr->index_ring_size[i] == (size_t)0);
        assert(cache_ptr->clean_index_ring_size[i] == (size_t)0);
        assert(cache_ptr->dirty_index_ring_size[i] == (size_t)0);

        assert(cache_ptr->slist_ring_len[i] == 0);
        assert(cache_ptr->slist_ring_size[i] == (size_t)0);
    } 

    assert(protected_entries <= cache_ptr->pl_len);

    if (protected_entries > 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Cache has protected entries");
    else if (cur_ring_pel_len > 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't unpin all pinned entries in ring");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__flush_ring(H5F_t *f, H5C_ring_t ring, unsigned flags)
{
    H5C_t             *cache_ptr = f->shared->cache;
    bool               flushed_entries_last_pass;
    bool               ignore_protected;
    bool               tried_to_flush_protected_entry = false;
    bool               restart_slist_scan;
    uint32_t           protected_entries = 0;
    H5SL_node_t       *node_ptr          = NULL;
    H5C_cache_entry_t *entry_ptr         = NULL;
    H5C_cache_entry_t *next_entry_ptr    = NULL;
#ifdef H5C_DO_SANITY_CHECKS
    uint32_t initial_slist_len  = 0;
    size_t   initial_slist_size = 0;
#endif 
    int    i;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(cache_ptr);
    assert(cache_ptr->slist_enabled);
    assert(cache_ptr->slist_ptr);
    assert((flags & H5C__FLUSH_INVALIDATE_FLAG) == 0);
    assert(ring > H5C_RING_UNDEFINED);
    assert(ring < H5C_RING_NTYPES);

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

    ignore_protected = ((flags & H5C__FLUSH_IGNORE_PROTECTED_FLAG) != 0);

    for (i = (int)H5C_RING_UNDEFINED; i < (int)ring; i++)
        assert(cache_ptr->slist_ring_len[i] == 0);

    assert(cache_ptr->flush_in_progress);

    
    flushed_entries_last_pass = true;

    
    cache_ptr->slist_changed = false;

    while ((cache_ptr->slist_ring_len[ring] > 0) && (protected_entries == 0) && (flushed_entries_last_pass)) {
        flushed_entries_last_pass = false;

#ifdef H5C_DO_SANITY_CHECKS
        
        initial_slist_len  = cache_ptr->slist_len;
        initial_slist_size = cache_ptr->slist_size;

        
        cache_ptr->slist_len_increase  = 0;
        cache_ptr->slist_size_increase = 0;

        
#endif 

        restart_slist_scan = true;
        while ((restart_slist_scan) || (node_ptr != NULL)) {
            if (restart_slist_scan) {
                restart_slist_scan = false;

                
                node_ptr = H5SL_first(cache_ptr->slist_ptr);
                if (node_ptr == NULL)
                    
                    break;

                
                next_entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
                if (NULL == next_entry_ptr)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "next_entry_ptr == NULL ?!?!");

                assert(next_entry_ptr->is_dirty);
                assert(next_entry_ptr->in_slist);
            } 

            entry_ptr = next_entry_ptr;

            
            assert(entry_ptr->in_slist);
            assert(entry_ptr->is_dirty);
            assert(entry_ptr->ring >= ring);

            
            node_ptr = H5SL_next(node_ptr);
            if (node_ptr != NULL) {
                next_entry_ptr = (H5C_cache_entry_t *)H5SL_item(node_ptr);
                if (NULL == next_entry_ptr)
                    HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "next_entry_ptr == NULL ?!?!");

                assert(next_entry_ptr->is_dirty);
                assert(next_entry_ptr->in_slist);
                assert(next_entry_ptr->ring >= ring);
                assert(entry_ptr != next_entry_ptr);
            } 
            else
                next_entry_ptr = NULL;

            if (((!entry_ptr->flush_me_last) ||
                 ((entry_ptr->flush_me_last) && cache_ptr->num_last_entries >= cache_ptr->slist_len)) &&
                ((entry_ptr->flush_dep_nchildren == 0) || (entry_ptr->flush_dep_ndirty_children == 0)) &&
                (entry_ptr->ring == ring)) {

                assert(entry_ptr->flush_dep_nunser_children == 0);

                if (entry_ptr->is_protected) {
                    
                    tried_to_flush_protected_entry = true;
                    protected_entries++;
                } 
                else {
                    if (H5C__flush_single_entry(f, entry_ptr, (flags | H5C__DURING_FLUSH_FLAG)) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't flush entry");

                    if (cache_ptr->slist_changed) {
                        
                        restart_slist_scan       = true;
                        cache_ptr->slist_changed = false;
                        H5C__UPDATE_STATS_FOR_SLIST_SCAN_RESTART(cache_ptr);
                    } 

                    flushed_entries_last_pass = true;
                } 
            }     
        }         

#ifdef H5C_DO_SANITY_CHECKS
        
        assert((uint32_t)((int32_t)initial_slist_len + cache_ptr->slist_len_increase) ==
               cache_ptr->slist_len);
        assert((size_t)((ssize_t)initial_slist_size + cache_ptr->slist_size_increase) ==
               cache_ptr->slist_size);
#endif 
    }  

    assert(protected_entries <= cache_ptr->pl_len);

    if (((cache_ptr->pl_len > 0) && !ignore_protected) || tried_to_flush_protected_entry)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "cache has protected items");

#ifdef H5C_DO_SANITY_CHECKS
    assert(cache_ptr->slist_ring_len[ring] == 0);
    assert(cache_ptr->slist_ring_size[ring] == 0);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__make_space_in_cache(H5F_t *f, size_t space_needed, bool write_permitted)
{
    H5C_t *cache_ptr = f->shared->cache;
#if H5C_COLLECT_CACHE_STATS
    int32_t clean_entries_skipped    = 0;
    int32_t dirty_pf_entries_skipped = 0;
    int32_t total_entries_scanned    = 0;
#endif 
    uint32_t           entries_examined = 0;
    uint32_t           initial_list_len;
    size_t             empty_space;
    bool               reentrant_call    = false;
    bool               prev_is_dirty     = false;
    bool               didnt_flush_entry = false;
    bool               restart_scan;
    H5C_cache_entry_t *entry_ptr;
    H5C_cache_entry_t *prev_ptr;
    H5C_cache_entry_t *next_ptr;
#ifndef NDEBUG
    uint32_t num_corked_entries = 0;
#endif
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cache_ptr);
    assert(cache_ptr->index_size == (cache_ptr->clean_index_size + cache_ptr->dirty_index_size));

    
    if (cache_ptr->msic_in_progress) {
        reentrant_call = true;
        HGOTO_DONE(SUCCEED);
    } 

    cache_ptr->msic_in_progress = true;

    if (write_permitted) {
        restart_scan     = false;
        initial_list_len = cache_ptr->LRU_list_len;
        entry_ptr        = cache_ptr->LRU_tail_ptr;

        if (cache_ptr->index_size >= cache_ptr->max_cache_size)
            empty_space = 0;
        else
            empty_space = cache_ptr->max_cache_size - cache_ptr->index_size;

        while ((((cache_ptr->index_size + space_needed) > cache_ptr->max_cache_size) ||
                ((empty_space + cache_ptr->clean_index_size) < (cache_ptr->min_clean_size))) &&
               (entries_examined <= (2 * initial_list_len)) && (entry_ptr != NULL)) {
            assert(!(entry_ptr->is_protected));
            assert(!(entry_ptr->is_read_only));
            assert((entry_ptr->ro_ref_count) == 0);

            next_ptr = entry_ptr->next;
            prev_ptr = entry_ptr->prev;

            if (prev_ptr != NULL)
                prev_is_dirty = prev_ptr->is_dirty;

            if (entry_ptr->is_dirty && (entry_ptr->tag_info && entry_ptr->tag_info->corked)) {
                
#ifndef NDEBUG
                ++num_corked_entries;
#endif
                didnt_flush_entry = true;
            }
            else if ((entry_ptr->type->id != H5AC_EPOCH_MARKER_ID) && !entry_ptr->flush_in_progress &&
                     !entry_ptr->prefetched_dirty) {
                didnt_flush_entry = false;
                if (entry_ptr->is_dirty) {
#if H5C_COLLECT_CACHE_STATS
                    if ((cache_ptr->index_size + space_needed) > cache_ptr->max_cache_size)
                        cache_ptr->entries_scanned_to_make_space++;
#endif 

                    
                    cache_ptr->entries_removed_counter = 0;
                    cache_ptr->last_entry_removed_ptr  = NULL;

                    if (H5C__flush_single_entry(f, entry_ptr, H5C__NO_FLAGS_SET) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush entry");

                    if ((cache_ptr->entries_removed_counter > 1) ||
                        (cache_ptr->last_entry_removed_ptr == prev_ptr))

                        restart_scan = true;
                }
                else if ((cache_ptr->index_size + space_needed) > cache_ptr->max_cache_size
#ifdef H5_HAVE_PARALLEL
                         && !(entry_ptr->coll_access)
#endif 
                ) {
#if H5C_COLLECT_CACHE_STATS
                    cache_ptr->entries_scanned_to_make_space++;
#endif 

                    if (H5C__flush_single_entry(f, entry_ptr,
                                                H5C__FLUSH_INVALIDATE_FLAG |
                                                    H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush entry");
                }
                else {
                    
#if H5C_COLLECT_CACHE_STATS
                    clean_entries_skipped++;
#endif 
                    didnt_flush_entry = true;
                }

#if H5C_COLLECT_CACHE_STATS
                total_entries_scanned++;
#endif 
            }
            else {

                
                didnt_flush_entry = true;

#if H5C_COLLECT_CACHE_STATS
                if (entry_ptr->prefetched_dirty)
                    dirty_pf_entries_skipped++;
#endif 
            }

            if (prev_ptr != NULL) {
                if (didnt_flush_entry)
                    
                    entry_ptr = prev_ptr;
                else if (restart_scan || prev_ptr->is_dirty != prev_is_dirty || prev_ptr->next != next_ptr ||
                         prev_ptr->is_protected || prev_ptr->is_pinned) {
                    
                    restart_scan = false;
                    entry_ptr    = cache_ptr->LRU_tail_ptr;
                    H5C__UPDATE_STATS_FOR_LRU_SCAN_RESTART(cache_ptr);
                }
                else
                    entry_ptr = prev_ptr;
            }
            else
                entry_ptr = NULL;

            entries_examined++;

            if (cache_ptr->index_size >= cache_ptr->max_cache_size)
                empty_space = 0;
            else
                empty_space = cache_ptr->max_cache_size - cache_ptr->index_size;

            assert(cache_ptr->index_size == (cache_ptr->clean_index_size + cache_ptr->dirty_index_size));
        }

#if H5C_COLLECT_CACHE_STATS
        cache_ptr->calls_to_msic++;

        cache_ptr->total_entries_skipped_in_msic += clean_entries_skipped;
        cache_ptr->total_dirty_pf_entries_skipped_in_msic += dirty_pf_entries_skipped;
        cache_ptr->total_entries_scanned_in_msic += total_entries_scanned;

        if (clean_entries_skipped > cache_ptr->max_entries_skipped_in_msic)
            cache_ptr->max_entries_skipped_in_msic = clean_entries_skipped;

        if (dirty_pf_entries_skipped > cache_ptr->max_dirty_pf_entries_skipped_in_msic)
            cache_ptr->max_dirty_pf_entries_skipped_in_msic = dirty_pf_entries_skipped;

        if (total_entries_scanned > cache_ptr->max_entries_scanned_in_msic)
            cache_ptr->max_entries_scanned_in_msic = total_entries_scanned;
#endif 

        
        assert((entries_examined > (2 * initial_list_len)) ||
               ((cache_ptr->pl_size + cache_ptr->pel_size + cache_ptr->min_clean_size) >
                cache_ptr->max_cache_size) ||
               ((cache_ptr->clean_index_size + empty_space) >= cache_ptr->min_clean_size) ||
               ((num_corked_entries)));
#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS

        assert((entries_examined > (2 * initial_list_len)) ||
               (cache_ptr->cLRU_list_size <= cache_ptr->clean_index_size));
        assert((entries_examined > (2 * initial_list_len)) ||
               (cache_ptr->dLRU_list_size <= cache_ptr->dirty_index_size));

#endif 
    }
    else {
        assert(H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS);

#if H5C_MAINTAIN_CLEAN_AND_DIRTY_LRU_LISTS
        initial_list_len = cache_ptr->cLRU_list_len;
        entry_ptr        = cache_ptr->cLRU_tail_ptr;

        while (((cache_ptr->index_size + space_needed) > cache_ptr->max_cache_size) &&
               (entries_examined <= initial_list_len) && (entry_ptr != NULL)) {
            assert(!(entry_ptr->is_protected));
            assert(!(entry_ptr->is_read_only));
            assert((entry_ptr->ro_ref_count) == 0);
            assert(!(entry_ptr->is_dirty));

            prev_ptr = entry_ptr->aux_prev;

            if (!entry_ptr->prefetched_dirty
#ifdef H5_HAVE_PARALLEL
                && !entry_ptr->coll_access
#endif 
            ) {
                if (H5C__flush_single_entry(
                        f, entry_ptr, H5C__FLUSH_INVALIDATE_FLAG | H5C__DEL_FROM_SLIST_ON_DESTROY_FLAG) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush entry");
            } 

            

            entry_ptr = prev_ptr;
            entries_examined++;
        }
#endif 
    }

done:
    
    assert(cache_ptr->msic_in_progress);
    if (!reentrant_call)
        cache_ptr->msic_in_progress = false;
    assert((!reentrant_call) || (cache_ptr->msic_in_progress));

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C__serialize_cache(H5F_t *f)
{
#ifdef H5C_DO_SANITY_CHECKS
    int      i;
    uint32_t index_len        = 0;
    size_t   index_size       = (size_t)0;
    size_t   clean_index_size = (size_t)0;
    size_t   dirty_index_size = (size_t)0;
    size_t   slist_size       = (size_t)0;
    uint32_t slist_len        = 0;
#endif 
    H5C_ring_t ring;
    H5C_t     *cache_ptr;
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(cache_ptr->slist_ptr);

#ifdef H5C_DO_SANITY_CHECKS
    assert(cache_ptr->index_ring_len[H5C_RING_UNDEFINED] == 0);
    assert(cache_ptr->index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
    assert(cache_ptr->clean_index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
    assert(cache_ptr->dirty_index_ring_size[H5C_RING_UNDEFINED] == (size_t)0);
    assert(cache_ptr->slist_ring_len[H5C_RING_UNDEFINED] == 0);
    assert(cache_ptr->slist_ring_size[H5C_RING_UNDEFINED] == (size_t)0);

    for (i = H5C_RING_USER; i < H5C_RING_NTYPES; i++) {
        index_len += cache_ptr->index_ring_len[i];
        index_size += cache_ptr->index_ring_size[i];
        clean_index_size += cache_ptr->clean_index_ring_size[i];
        dirty_index_size += cache_ptr->dirty_index_ring_size[i];

        slist_len += cache_ptr->slist_ring_len[i];
        slist_size += cache_ptr->slist_ring_size[i];
    } 

    assert(cache_ptr->index_len == index_len);
    assert(cache_ptr->index_size == index_size);
    assert(cache_ptr->clean_index_size == clean_index_size);
    assert(cache_ptr->dirty_index_size == dirty_index_size);
    assert(cache_ptr->slist_len == slist_len);
    assert(cache_ptr->slist_size == slist_size);
#endif 

#ifdef H5C_DO_EXTREME_SANITY_CHECKS
    if (H5C__validate_protected_entry_list(cache_ptr) < 0 || H5C__validate_pinned_entry_list(cache_ptr) < 0 ||
        H5C__validate_lru_list(cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "an extreme sanity check failed on entry");
#endif 

#ifndef NDEBUG
    
    {
        H5C_cache_entry_t *scan_ptr = NULL;

        scan_ptr = cache_ptr->il_head;
        while (scan_ptr != NULL) {
            scan_ptr->serialization_count = 0;
            scan_ptr                      = scan_ptr->il_next;
        } 
    }     
#endif

    
    assert(!cache_ptr->serialization_in_progress);
    cache_ptr->serialization_in_progress = true;

    
    ring = H5C_RING_USER;
    while (ring < H5C_RING_NTYPES) {
        assert(cache_ptr->close_warning_received);
        switch (ring) {
            case H5C_RING_USER:
                break;

            case H5C_RING_RDFSM:
                
                if (!cache_ptr->rdfsm_settled)
                    if (H5MF_settle_raw_data_fsm(f, &cache_ptr->rdfsm_settled) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "RD FSM settle failed");
                break;

            case H5C_RING_MDFSM:
                
                if (!cache_ptr->mdfsm_settled)
                    if (H5MF_settle_meta_data_fsm(f, &cache_ptr->mdfsm_settled) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "MD FSM settle failed");
                break;

            case H5C_RING_SBE:
            case H5C_RING_SB:
                break;

            default:
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Unknown ring?!?!");
                break;
        } 

        if (H5C__serialize_ring(f, ring) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "serialize ring failed");

        ring++;
    } 

#ifndef NDEBUG
    
    {
        H5C_cache_entry_t *scan_ptr = NULL;

        scan_ptr = cache_ptr->il_head;
        while (scan_ptr != NULL) {
            assert(scan_ptr->serialization_count <= 1);

            scan_ptr = scan_ptr->il_next;
        } 
    }     
#endif

done:
    cache_ptr->serialization_in_progress = false;
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5C__serialize_ring(H5F_t *f, H5C_ring_t ring)
{
    bool               done = false;
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(ring > H5C_RING_UNDEFINED);
    assert(ring < H5C_RING_NTYPES);

    assert(cache_ptr->serialization_in_progress);

    
    while (!done) {
        
        cache_ptr->entries_loaded_counter    = 0;
        cache_ptr->entries_inserted_counter  = 0;
        cache_ptr->entries_relocated_counter = 0;

        done      = true; 
        entry_ptr = cache_ptr->il_head;
        while (entry_ptr != NULL) {
            
            assert((entry_ptr->ring >= ring) || (entry_ptr->image_up_to_date));

            
            if (!entry_ptr->flush_me_last && entry_ptr->ring == ring) {

                
                if (!entry_ptr->image_up_to_date)
                    done = false;

                
                if (!entry_ptr->image_up_to_date && entry_ptr->flush_dep_nunser_children == 0) {
                    assert(entry_ptr->serialization_count == 0);

                    
                    if (H5C__serialize_single_entry(f, cache_ptr, entry_ptr) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "entry serialization failed");

                    assert(entry_ptr->flush_dep_nunser_children == 0);
                    assert(entry_ptr->serialization_count == 0);

#ifndef NDEBUG
                    
                    entry_ptr->serialization_count++;
#endif
                } 
            }     

            
            if ((cache_ptr->entries_loaded_counter > 0) || (cache_ptr->entries_inserted_counter > 0) ||
                (cache_ptr->entries_relocated_counter > 0)) {

#if H5C_COLLECT_CACHE_STATS
                H5C__UPDATE_STATS_FOR_INDEX_SCAN_RESTART(cache_ptr);
#endif 

                
                cache_ptr->entries_loaded_counter    = 0;
                cache_ptr->entries_inserted_counter  = 0;
                cache_ptr->entries_relocated_counter = 0;

                
                entry_ptr = cache_ptr->il_head;
            } 
            else
                
                entry_ptr = entry_ptr->il_next;
        } 
    }     

    
    cache_ptr->entries_loaded_counter    = 0;
    cache_ptr->entries_inserted_counter  = 0;
    cache_ptr->entries_relocated_counter = 0;

    
    entry_ptr = cache_ptr->il_head;
    while (entry_ptr != NULL) {
        assert(entry_ptr->ring > H5C_RING_UNDEFINED);
        assert(entry_ptr->ring < H5C_RING_NTYPES);
        assert((entry_ptr->ring >= ring) || (entry_ptr->image_up_to_date));

        if (entry_ptr->ring == ring) {
            if (entry_ptr->flush_me_last) {
                if (!entry_ptr->image_up_to_date) {
                    assert(entry_ptr->serialization_count == 0);
                    assert(entry_ptr->flush_dep_nunser_children == 0);

                    
                    if (H5C__serialize_single_entry(f, cache_ptr, entry_ptr) < 0)
                        HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "entry serialization failed");

                    
                    if ((cache_ptr->entries_loaded_counter > 0) ||
                        (cache_ptr->entries_inserted_counter > 0) ||
                        (cache_ptr->entries_relocated_counter > 0))
                        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL,
                                    "flush_me_last entry serialization triggered restart");

                    assert(entry_ptr->flush_dep_nunser_children == 0);
                    assert(entry_ptr->serialization_count == 0);
#ifndef NDEBUG
                    
                    entry_ptr->serialization_count++;
#endif
                } 
            }     
            else {
                assert(entry_ptr->image_up_to_date);
                assert(entry_ptr->serialization_count <= 1);
                assert(entry_ptr->flush_dep_nunser_children == 0);
            } 
        }     

        entry_ptr = entry_ptr->il_next;
    } 

done:
    assert(cache_ptr->serialization_in_progress);
    FUNC_LEAVE_NOAPI(ret_value)
} 
