/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDprivate.h" 
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Oprivate.h"  
#include "H5PBprivate.h" 
#include "H5VMprivate.h" 

typedef struct H5D_contig_readvv_sieve_ud_t {
    H5F_shared_t               *f_sh;         
    H5D_rdcdc_t                *dset_contig;  
    const H5D_contig_storage_t *store_contig; 
    unsigned char              *rbuf;         
} H5D_contig_readvv_sieve_ud_t;

typedef struct H5D_contig_readvv_ud_t {
    H5F_shared_t  *f_sh;      
    haddr_t        dset_addr; 
    unsigned char *rbuf;      
} H5D_contig_readvv_ud_t;

typedef struct H5D_contig_writevv_sieve_ud_t {
    H5F_shared_t               *f_sh;         
    H5D_rdcdc_t                *dset_contig;  
    const H5D_contig_storage_t *store_contig; 
    const unsigned char        *wbuf;         
} H5D_contig_writevv_sieve_ud_t;

typedef struct H5D_contig_writevv_ud_t {
    H5F_shared_t        *f_sh;      
    haddr_t              dset_addr; 
    const unsigned char *wbuf;      
} H5D_contig_writevv_ud_t;

static herr_t  H5D__contig_construct(H5F_t *f, H5D_t *dset);
static herr_t  H5D__contig_init(H5F_t *f, H5D_t *dset, hid_t dapl_id, bool open_op);
static herr_t  H5D__contig_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static herr_t  H5D__contig_mdio_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo);
static ssize_t H5D__contig_readvv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dinfo,
                                  size_t dset_max_nseq, size_t *dset_curr_seq, size_t dset_len_arr[],
                                  hsize_t dset_offset_arr[], size_t mem_max_nseq, size_t *mem_curr_seq,
                                  size_t mem_len_arr[], hsize_t mem_offset_arr[]);
static ssize_t H5D__contig_writevv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dinfo,
                                   size_t dset_max_nseq, size_t *dset_curr_seq, size_t dset_len_arr[],
                                   hsize_t dset_offset_arr[], size_t mem_max_nseq, size_t *mem_curr_seq,
                                   size_t mem_len_arr[], hsize_t mem_offset_arr[]);
static herr_t  H5D__contig_flush(H5D_t *dset);
static herr_t  H5D__contig_io_term(H5D_io_info_t *io_info, H5D_dset_io_info_t *di);

static herr_t H5D__contig_write_one(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info, hsize_t offset,
                                    size_t size);
static herr_t H5D__contig_may_use_select_io(H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                                            H5D_io_op_type_t op_type);

const H5D_layout_ops_t H5D_LOPS_CONTIG[1] = {{
    H5D__contig_construct,      
    H5D__contig_init,           
    H5D__contig_is_space_alloc, 
    H5D__contig_is_data_cached, 
    H5D__contig_io_init,        
    H5D__contig_mdio_init,      
    H5D__contig_read,           
    H5D__contig_write,          
    H5D__contig_readvv,         
    H5D__contig_writevv,        
    H5D__contig_flush,          
    H5D__contig_io_term,        
    NULL                        
}};

H5FL_BLK_DEFINE(sieve_buf);

H5FL_BLK_EXTERN(type_conv);

H5FL_EXTERN(H5D_piece_info_t);

herr_t
H5D__contig_alloc(H5F_t *f, H5O_storage_contig_t *storage )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(storage);

    
    if (HADDR_UNDEF == (storage->addr = H5MF_alloc(f, H5FD_MEM_DRAW, storage->size)))
        HGOTO_ERROR(H5E_IO, H5E_NOSPACE, FAIL, "unable to reserve file space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_fill(H5D_t *dset)
{
    H5D_io_info_t      ioinfo;       
    H5D_dset_io_info_t dset_info;    
    H5D_storage_t      store;        
    hssize_t           snpoints;     
    size_t             npoints;      
    hsize_t            offset;       
    size_t             max_temp_buf; 
#ifdef H5_HAVE_PARALLEL
    MPI_Comm mpi_comm = MPI_COMM_NULL; 
    int      mpi_rank = (-1);          
    int      mpi_code;                 
    bool     blocks_written = false;   
    bool     using_mpi =
        false; 
#endif         
    H5D_fill_buf_info_t fb_info;                
    bool                fb_info_init = false;   
    herr_t              ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset && H5D_CONTIGUOUS == dset->shared->layout.type);
    assert(H5_addr_defined(dset->shared->layout.storage.u.contig.addr));
    assert(dset->shared->layout.storage.u.contig.size > 0);
    assert(dset->shared->space);
    assert(dset->shared->type);

#ifdef H5_HAVE_PARALLEL
    
    if (H5F_HAS_FEATURE(dset->oloc.file, H5FD_FEAT_HAS_MPI)) {
        
        if (MPI_COMM_NULL == (mpi_comm = H5F_mpi_get_comm(dset->oloc.file)))
            HGOTO_ERROR(H5E_INTERNAL, H5E_MPI, FAIL, "Can't retrieve MPI communicator");

        
        if ((mpi_rank = H5F_mpi_get_rank(dset->oloc.file)) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_MPI, FAIL, "Can't retrieve MPI rank");

        
        using_mpi = true;
    }  
#endif 

    
    store.contig.dset_addr = dset->shared->layout.storage.u.contig.addr;
    store.contig.dset_size = dset->shared->layout.storage.u.contig.size;

    
    if ((snpoints = H5S_GET_EXTENT_NPOINTS(dset->shared->space)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "dataset has negative number of elements");
    H5_CHECKED_ASSIGN(npoints, size_t, snpoints, hssize_t);

    
    if (H5CX_get_max_temp_buf(&max_temp_buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve max. temp. buf size");

    
    if (H5D__fill_init(&fb_info, NULL, NULL, NULL, NULL, NULL, &dset->shared->dcpl_cache.fill,
                       dset->shared->type, npoints, max_temp_buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize fill buffer info");
    fb_info_init = true;

    
    offset = 0;

    
    ioinfo.op_type = H5D_IO_OP_WRITE;

    dset_info.dset      = (H5D_t *)dset;
    dset_info.store     = &store;
    dset_info.buf.cvp   = fb_info.fill_buf;
    dset_info.mem_space = NULL;
    ioinfo.dsets_info   = &dset_info;
    ioinfo.f_sh         = H5F_SHARED(dset->oloc.file);

    

    
    while (npoints > 0) {
        size_t curr_points; 
        size_t size;        

        
        curr_points = MIN(fb_info.elmts_per_buf, npoints);
        size        = curr_points * fb_info.file_elmt_size;

        
        if (fb_info.has_vlen_fill_type)
            
            if (H5D__fill_refill_vl(&fb_info, curr_points) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "can't refill fill value buffer");

#ifdef H5_HAVE_PARALLEL
        
        if (using_mpi) {
            
            
            if (H5_PAR_META_WRITE == mpi_rank) {
                if (H5D__contig_write_one(&ioinfo, &dset_info, offset, size) < 0) {
                    
                    blocks_written = true;
                    HDONE_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to write fill value to dataset");
                    break;
                }
            }

            
            blocks_written = true;
        } 
        else {
#endif 
            H5_CHECK_OVERFLOW(size, size_t, hsize_t);
            if (H5D__contig_write_one(&ioinfo, &dset_info, offset, size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to write fill value to dataset");
#ifdef H5_HAVE_PARALLEL
        } 
#endif    

        npoints -= curr_points;
        offset += size;
    } 

#ifdef H5_HAVE_PARALLEL
    
    
    if (using_mpi && blocks_written) {
        
        if (MPI_SUCCESS != (mpi_code = MPI_Barrier(mpi_comm)))
            HMPI_GOTO_ERROR(FAIL, "MPI_Barrier failed", mpi_code)
    }  
#endif 

done:
    
    if (fb_info_init && H5D__fill_term(&fb_info) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_delete(H5F_t *f, const H5O_storage_t *storage)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(storage);

    
    if (H5MF_xfree(f, H5FD_MEM_DRAW, storage->u.contig.addr, storage->u.contig.size) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to free contiguous storage space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_check(const H5F_t *f, const H5O_layout_t *layout, const H5S_extent_t *extent, const H5T_t *dt)
{
    hsize_t nelmts;              
    size_t  dt_size;             
    hsize_t data_size;           
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(layout);
    assert(extent);
    assert(dt);

    
    nelmts = H5S_extent_nelem(extent);

    
    if (0 == (dt_size = H5T_GET_SIZE(dt)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve size of datatype");

    
    data_size = nelmts * dt_size;

    
    if (nelmts != (data_size / dt_size))
        HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "size of dataset's storage overflowed");

    
    if (H5_addr_defined(layout->storage.u.contig.addr)) {
        haddr_t rel_eoa; 

        if (HADDR_UNDEF == (rel_eoa = H5F_get_eoa(f, H5FD_MEM_DRAW)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine file size");

        
        if (H5_addr_le((layout->storage.u.contig.addr + data_size), layout->storage.u.contig.addr))
            HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "invalid dataset size, likely file corruption");
        if (H5_addr_gt((layout->storage.u.contig.addr + data_size), rel_eoa))
            HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "invalid dataset size, likely file corruption");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_construct(H5F_t *f, H5D_t *dset)
{
    hssize_t snelmts;             
    hsize_t  nelmts;              
    size_t   dt_size;             
    hsize_t  tmp_size;            
    size_t   tmp_sieve_buf_size;  
    unsigned version;             
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);
    assert(dset->shared);

    

    
    for (u = 0; u < dset->shared->ndims; u++)
        if (dset->shared->max_dims[u] > dset->shared->curr_dims[u])
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                        "extendible contiguous non-external dataset not allowed");

    
    if ((snelmts = H5S_GET_EXTENT_NPOINTS(dset->shared->space)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve number of elements in dataspace");
    nelmts = (hsize_t)snelmts;

    
    if (0 == (dt_size = H5T_GET_SIZE(dset->shared->type)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve size of datatype");

    
    tmp_size = nelmts * dt_size;

    
    if (nelmts != (tmp_size / dt_size))
        HGOTO_ERROR(H5E_DATASET, H5E_OVERFLOW, FAIL, "size of dataset's storage overflowed");

    
    dset->shared->layout.storage.u.contig.size = tmp_size;

    
    tmp_sieve_buf_size = H5F_SIEVE_BUF_SIZE(f);

    
    if (tmp_size < tmp_sieve_buf_size)
        dset->shared->cache.contig.sieve_buf_size = tmp_size;
    else
        dset->shared->cache.contig.sieve_buf_size = tmp_sieve_buf_size;

    
    if (dset->shared->layout.version < H5O_LAYOUT_VERSION_3) {
        version = MAX(dset->shared->layout.version,
                      MIN(H5O_layout_ver_bounds[H5F_LOW_BOUND(f)], H5O_LAYOUT_VERSION_3));

        
        if (version > H5O_layout_ver_bounds[H5F_HIGH_BOUND(f)])
            HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, FAIL, "layout version out of bounds");

        dset->shared->layout.version = version;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_init(H5F_t *f, H5D_t *dset, hid_t H5_ATTR_UNUSED dapl_id, bool H5_ATTR_UNUSED open_op)
{
    size_t tmp_sieve_buf_size;  
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(dset);

    
    if (H5D__contig_check(f, &dset->shared->layout, H5S_GET_EXTENT(dset->shared->space), dset->shared->type) <
        0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "invalid dataset info");

    
    if (dset->shared->layout.version < 3) {
        hssize_t snelmts; 
        hsize_t  nelmts;  
        size_t   dt_size; 

        
        if ((snelmts = H5S_GET_EXTENT_NPOINTS(dset->shared->space)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve number of elements in dataspace");
        nelmts = (hsize_t)snelmts;

        
        if (0 == (dt_size = H5T_GET_SIZE(dset->shared->type)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve size of datatype");

        
        dset->shared->layout.storage.u.contig.size = nelmts * dt_size;
    }

    
    tmp_sieve_buf_size = H5F_SIEVE_BUF_SIZE(dset->oloc.file);

    
    if (dset->shared->layout.storage.u.contig.size < tmp_sieve_buf_size)
        dset->shared->cache.contig.sieve_buf_size = dset->shared->layout.storage.u.contig.size;
    else
        dset->shared->cache.contig.sieve_buf_size = tmp_sieve_buf_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5D__contig_is_space_alloc(const H5O_storage_t *storage)
{
    bool ret_value = false; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    
    ret_value = (bool)H5_addr_defined(storage->u.contig.addr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5D__contig_is_data_cached(const H5D_shared_t *shared_dset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(shared_dset);

    FUNC_LEAVE_NOAPI(shared_dset->cache.contig.sieve_size > 0)
} 

static herr_t
H5D__contig_io_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    H5D_t *dataset = dinfo->dset; 

    hssize_t old_offset[H5O_LAYOUT_NDIMS];  
    htri_t   file_space_normalized = false; 

    int sf_ndims; 

    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    dinfo->store->contig.dset_addr = dataset->shared->layout.storage.u.contig.addr;
    dinfo->store->contig.dset_size = dataset->shared->layout.storage.u.contig.size;

    
    dinfo->layout_io_info.contig_piece_info = NULL;

    
    dinfo->layout = &(dataset->shared->layout);

    
    if ((sf_ndims = H5S_GET_EXTENT_NDIMS(dinfo->file_space)) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTGET, FAIL, "unable to get dimension number");

    
    
    if ((file_space_normalized = H5S_hyper_normalize_offset(dinfo->file_space, old_offset)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to normalize dataspace by offset");

    
    if (dinfo->nelmts) {
        int               u;
        H5D_piece_info_t *new_piece_info; 

        
        H5S_t *tmp_fspace; 

        
        if (NULL == (tmp_fspace = H5S_copy(dinfo->file_space, true, false)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "unable to copy memory space");

        
        
        
        if (NULL == (new_piece_info = H5FL_MALLOC(H5D_piece_info_t))) {
            (void)H5S_close(tmp_fspace);
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't allocate chunk info");
        } 

        
        new_piece_info->index = 0;

        
        new_piece_info->fspace        = tmp_fspace;
        new_piece_info->fspace_shared = false;

        
        
        new_piece_info->mspace = dinfo->mem_space;

        
        new_piece_info->mspace_shared = true;

        
        new_piece_info->piece_points = dinfo->nelmts;

        
        for (u = 0; u < sf_ndims; u++)
            new_piece_info->scaled[u] = 0;
        new_piece_info->scaled[sf_ndims] = 0;

        
        new_piece_info->dset_info = dinfo;

        
        new_piece_info->faddr = dinfo->dset->shared->layout.storage.u.contig.addr;

        
        new_piece_info->in_place_tconv = false;
        new_piece_info->buf_off        = 0;

        new_piece_info->filtered_dset = dinfo->dset->shared->dcpl_cache.pline.nused > 0;

        
        if (io_info->use_select_io != H5D_SELECTION_IO_MODE_OFF &&
            !(dinfo->type_info.is_xform_noop && dinfo->type_info.is_conv_noop))
            H5D_INIT_PIECE_TCONV(io_info, dinfo, new_piece_info)

        
        dinfo->layout_io_info.contig_piece_info = new_piece_info;

        
        io_info->piece_count++;
    } 

    
    if (io_info->use_select_io != H5D_SELECTION_IO_MODE_OFF)
        if (H5D__contig_may_use_select_io(io_info, dinfo, io_info->op_type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if selection I/O is possible");

done:
    if (ret_value < 0) {
        if (H5D__contig_io_term(io_info, dinfo) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release dataset I/O info");
    } 

    if (file_space_normalized) {
        
        if (H5S_hyper_denormalize_offset(dinfo->file_space, old_offset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_BADSELECT, FAIL, "unable to normalize dataspace by offset");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_mdio_init(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    if (dinfo->layout_io_info.contig_piece_info) {
        assert(io_info->sel_pieces);
        assert(io_info->pieces_added < io_info->piece_count);

        
        io_info->sel_pieces[io_info->pieces_added] = dinfo->layout_io_info.contig_piece_info;

        
        io_info->pieces_added++;
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__contig_may_use_select_io(H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info,
                              H5D_io_op_type_t op_type)
{
    const H5D_t *dataset   = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_info->dset);
    assert(op_type == H5D_IO_OP_READ || op_type == H5D_IO_OP_WRITE);

    dataset = dset_info->dset;

    

    
    if (dset_info->layout_ops.readvv != H5D__contig_readvv) {
        io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
        io_info->no_selection_io_cause |= H5D_SEL_IO_NOT_CONTIGUOUS_OR_CHUNKED_DATASET;
    }
    else if ((op_type == H5D_IO_OP_READ && dataset->shared->cache.contig.sieve_dirty) ||
             (op_type == H5D_IO_OP_WRITE && dataset->shared->cache.contig.sieve_buf)) {
        io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
        io_info->no_selection_io_cause |= H5D_SEL_IO_CONTIGUOUS_SIEVE_BUFFER;
    }
    else {
        bool page_buf_enabled;

        assert(dset_info->layout_ops.writevv == H5D__contig_writevv);

        
        if (H5PB_enabled(io_info->f_sh, H5FD_MEM_DRAW, &page_buf_enabled) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if page buffer is enabled");
        if (page_buf_enabled) {
            io_info->use_select_io = H5D_SELECTION_IO_MODE_OFF;
            io_info->no_selection_io_cause |= H5D_SEL_IO_PAGE_BUFFER;
        }
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_read(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dinfo);
    assert(dinfo->buf.vp);
    assert(dinfo->mem_space);
    assert(dinfo->file_space);

    if (io_info->use_select_io == H5D_SELECTION_IO_MODE_ON) {
        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            size_t dst_type_size = dinfo->type_info.dst_type_size;

            
            if (H5F_shared_select_read(H5F_SHARED(dinfo->dset->oloc.file), H5FD_MEM_DRAW,
                                       dinfo->nelmts > 0 ? 1 : 0, &dinfo->mem_space, &dinfo->file_space,
                                       &(dinfo->store->contig.dset_addr), &dst_type_size,
                                       &(dinfo->buf.vp)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "contiguous selection read failed");
        }
        else {
            if (dinfo->layout_io_info.contig_piece_info) {
                
                assert(io_info->mem_spaces);
                assert(io_info->file_spaces);
                assert(io_info->addrs);
                assert(io_info->element_sizes);
                assert(io_info->rbufs);
                assert(io_info->pieces_added < io_info->piece_count);

                io_info->mem_spaces[io_info->pieces_added]    = dinfo->mem_space;
                io_info->file_spaces[io_info->pieces_added]   = dinfo->file_space;
                io_info->addrs[io_info->pieces_added]         = dinfo->store->contig.dset_addr;
                io_info->element_sizes[io_info->pieces_added] = dinfo->type_info.src_type_size;
                io_info->rbufs[io_info->pieces_added]         = dinfo->buf.vp;
                if (io_info->sel_pieces)
                    io_info->sel_pieces[io_info->pieces_added] = dinfo->layout_io_info.contig_piece_info;
                io_info->pieces_added++;
            }
        }

#ifdef H5_HAVE_PARALLEL
        
        io_info->actual_io_mode |= H5D_MPIO_CONTIGUOUS_COLLECTIVE;
#endif 
    }  
    else
        
        if ((dinfo->io_ops.single_read)(io_info, dinfo) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "contiguous read failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_write(H5D_io_info_t *io_info, H5D_dset_io_info_t *dinfo)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dinfo);
    assert(dinfo->buf.cvp);
    assert(dinfo->mem_space);
    assert(dinfo->file_space);

    if (io_info->use_select_io == H5D_SELECTION_IO_MODE_ON) {
        
        if (H5D_LAYOUT_CB_PERFORM_IO(io_info)) {
            size_t dst_type_size = dinfo->type_info.dst_type_size;

            
            if (H5F_shared_select_write(H5F_SHARED(dinfo->dset->oloc.file), H5FD_MEM_DRAW,
                                        dinfo->nelmts > 0 ? 1 : 0, &dinfo->mem_space, &dinfo->file_space,
                                        &(dinfo->store->contig.dset_addr), &dst_type_size,
                                        &(dinfo->buf.cvp)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "contiguous selection write failed");
        }
        else {
            if (dinfo->layout_io_info.contig_piece_info) {
                
                assert(io_info->mem_spaces);
                assert(io_info->file_spaces);
                assert(io_info->addrs);
                assert(io_info->element_sizes);
                assert(io_info->wbufs);
                assert(io_info->pieces_added < io_info->piece_count);

                io_info->mem_spaces[io_info->pieces_added]    = dinfo->mem_space;
                io_info->file_spaces[io_info->pieces_added]   = dinfo->file_space;
                io_info->addrs[io_info->pieces_added]         = dinfo->store->contig.dset_addr;
                io_info->element_sizes[io_info->pieces_added] = dinfo->type_info.dst_type_size;
                io_info->wbufs[io_info->pieces_added]         = dinfo->buf.cvp;
                if (io_info->sel_pieces)
                    io_info->sel_pieces[io_info->pieces_added] = dinfo->layout_io_info.contig_piece_info;
                io_info->pieces_added++;
            }
        }

#ifdef H5_HAVE_PARALLEL
        
        io_info->actual_io_mode |= H5D_MPIO_CONTIGUOUS_COLLECTIVE;
#endif 
    }  
    else
        
        if ((dinfo->io_ops.single_write)(io_info, dinfo) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "contiguous write failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_write_one(H5D_io_info_t *io_info, H5D_dset_io_info_t *dset_info, hsize_t offset, size_t size)
{
    hsize_t dset_off      = offset;  
    size_t  dset_len      = size;    
    size_t  dset_curr_seq = 0;       
    hsize_t mem_off       = 0;       
    size_t  mem_len       = size;    
    size_t  mem_curr_seq  = 0;       
    herr_t  ret_value     = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(io_info);

    if (H5D__contig_writevv(io_info, dset_info, (size_t)1, &dset_curr_seq, &dset_len, &dset_off, (size_t)1,
                            &mem_curr_seq, &mem_len, &mem_off) < 0)
        HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "vector write failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_readvv_sieve_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_contig_readvv_sieve_ud_t *udata =
        (H5D_contig_readvv_sieve_ud_t *)_udata;     
    H5F_shared_t *f_sh        = udata->f_sh;        
    H5D_rdcdc_t  *dset_contig = udata->dset_contig; 
    const H5D_contig_storage_t *store_contig =
        udata->store_contig; 
    unsigned char *buf;      
    haddr_t        addr;     
    haddr_t sieve_start = HADDR_UNDEF, sieve_end = HADDR_UNDEF; 
    haddr_t contig_end;                                         
    size_t  sieve_size = (size_t)-1;                            
    haddr_t rel_eoa;                                            
    hsize_t max_data;                                           
    hsize_t min;                 
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (dset_contig->sieve_buf != NULL) {
        sieve_start = dset_contig->sieve_loc;
        sieve_size  = dset_contig->sieve_size;
        sieve_end   = sieve_start + sieve_size;
    } 

    
    addr = store_contig->dset_addr + dst_off;

    
    buf = udata->rbuf + src_off;

    
    if (NULL == dset_contig->sieve_buf) {
        
        if (len > dset_contig->sieve_buf_size) {
            if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, addr, len, buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");
        } 
        else {
            
            if (NULL == (dset_contig->sieve_buf = H5FL_BLK_CALLOC(sieve_buf, dset_contig->sieve_buf_size)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");

            
            dset_contig->sieve_loc = addr;

            
            if (HADDR_UNDEF == (rel_eoa = H5F_shared_get_eoa(f_sh, H5FD_MEM_DRAW)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine file size");

            
            max_data = store_contig->dset_size - dst_off;

            
            min = MIN3(rel_eoa - dset_contig->sieve_loc, max_data, dset_contig->sieve_buf_size);
            H5_CHECKED_ASSIGN(dset_contig->sieve_size, size_t, min, hsize_t);

            
            if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, dset_contig->sieve_loc, dset_contig->sieve_size,
                                      dset_contig->sieve_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");

            
            H5MM_memcpy(buf, dset_contig->sieve_buf, len);

            
            dset_contig->sieve_dirty = false;
        } 
    }     
    else {
        
        contig_end = addr + len - 1;

        
        if (addr >= sieve_start && contig_end < sieve_end) {
            unsigned char *base_sieve_buf = dset_contig->sieve_buf + (addr - sieve_start);

            
            H5MM_memcpy(buf, base_sieve_buf, len);
        } 
        
        else {
            
            if (len > dset_contig->sieve_buf_size) {
                
                if ((sieve_start >= addr && sieve_start < (contig_end + 1)) ||
                    ((sieve_end - 1) >= addr && (sieve_end - 1) < (contig_end + 1))) {
                    
                    if (dset_contig->sieve_dirty) {
                        
                        if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, sieve_start, sieve_size,
                                                   dset_contig->sieve_buf) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

                        
                        dset_contig->sieve_dirty = false;
                    } 
                }     

                
                if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, addr, len, buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");
            } 
            
            else {
                
                if (dset_contig->sieve_dirty) {
                    
                    if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, sieve_start, sieve_size,
                                               dset_contig->sieve_buf) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

                    
                    dset_contig->sieve_dirty = false;
                } 

                
                dset_contig->sieve_loc = addr;

                
                if (HADDR_UNDEF == (rel_eoa = H5F_shared_get_eoa(f_sh, H5FD_MEM_DRAW)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine file size");

                
                max_data = store_contig->dset_size - dst_off;

                
                min = MIN3(rel_eoa - dset_contig->sieve_loc, max_data, dset_contig->sieve_buf_size);
                H5_CHECKED_ASSIGN(dset_contig->sieve_size, size_t, min, hsize_t);

                
                if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, dset_contig->sieve_loc,
                                          dset_contig->sieve_size, dset_contig->sieve_buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");

                
                H5MM_memcpy(buf, dset_contig->sieve_buf, len);

                
                dset_contig->sieve_dirty = false;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_readvv_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_contig_readvv_ud_t *udata = (H5D_contig_readvv_ud_t *)_udata; 
    herr_t                  ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    if (H5F_shared_block_read(udata->f_sh, H5FD_MEM_DRAW, (udata->dset_addr + dst_off), len,
                              (udata->rbuf + src_off)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5D__contig_readvv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info, size_t dset_max_nseq,
                   size_t *dset_curr_seq, size_t dset_len_arr[], hsize_t dset_off_arr[], size_t mem_max_nseq,
                   size_t *mem_curr_seq, size_t mem_len_arr[], hsize_t mem_off_arr[])
{
    ssize_t ret_value = -1; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_curr_seq);
    assert(dset_len_arr);
    assert(dset_off_arr);
    assert(mem_curr_seq);
    assert(mem_len_arr);
    assert(mem_off_arr);

    
    if (H5F_SHARED_HAS_FEATURE(io_info->f_sh, H5FD_FEAT_DATA_SIEVE)) {
        H5D_contig_readvv_sieve_ud_t udata; 

        
        udata.f_sh         = io_info->f_sh;
        udata.dset_contig  = &(dset_info->dset->shared->cache.contig);
        udata.store_contig = &(dset_info->store->contig);
        udata.rbuf         = (unsigned char *)dset_info->buf.vp;

        
        if ((ret_value =
                 H5VM_opvv(dset_max_nseq, dset_curr_seq, dset_len_arr, dset_off_arr, mem_max_nseq,
                           mem_curr_seq, mem_len_arr, mem_off_arr, H5D__contig_readvv_sieve_cb, &udata)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPERATE, FAIL, "can't perform vectorized sieve buffer read");
    } 
    else {
        H5D_contig_readvv_ud_t udata; 

        
        udata.f_sh      = io_info->f_sh;
        udata.dset_addr = dset_info->store->contig.dset_addr;
        udata.rbuf      = (unsigned char *)dset_info->buf.vp;

        
        if ((ret_value = H5VM_opvv(dset_max_nseq, dset_curr_seq, dset_len_arr, dset_off_arr, mem_max_nseq,
                                   mem_curr_seq, mem_len_arr, mem_off_arr, H5D__contig_readvv_cb, &udata)) <
            0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPERATE, FAIL, "can't perform vectorized read");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_writevv_sieve_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_contig_writevv_sieve_ud_t *udata =
        (H5D_contig_writevv_sieve_ud_t *)_udata;    
    H5F_shared_t *f_sh        = udata->f_sh;        
    H5D_rdcdc_t  *dset_contig = udata->dset_contig; 
    const H5D_contig_storage_t *store_contig =
        udata->store_contig;   
    const unsigned char *buf;  
    haddr_t              addr; 
    haddr_t sieve_start = HADDR_UNDEF, sieve_end = HADDR_UNDEF; 
    haddr_t contig_end;                                         
    size_t  sieve_size = (size_t)-1;                            
    haddr_t rel_eoa;                                            
    hsize_t max_data;                                           
    hsize_t min;                 
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (dset_contig->sieve_buf != NULL) {
        sieve_start = dset_contig->sieve_loc;
        sieve_size  = dset_contig->sieve_size;
        sieve_end   = sieve_start + sieve_size;
    } 

    
    addr = store_contig->dset_addr + dst_off;

    
    buf = udata->wbuf + src_off;

    
    if (NULL == dset_contig->sieve_buf) {
        
        if (len > dset_contig->sieve_buf_size) {
            if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, addr, len, buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");
        } 
        else {
            
            if (NULL == (dset_contig->sieve_buf = H5FL_BLK_CALLOC(sieve_buf, dset_contig->sieve_buf_size)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");

            
            if (dset_contig->sieve_size > len)
                memset(dset_contig->sieve_buf + len, 0, (dset_contig->sieve_size - len));

            
            dset_contig->sieve_loc = addr;

            
            if (HADDR_UNDEF == (rel_eoa = H5F_shared_get_eoa(f_sh, H5FD_MEM_DRAW)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine file size");

            
            max_data = store_contig->dset_size - dst_off;

            
            min = MIN3(rel_eoa - dset_contig->sieve_loc, max_data, dset_contig->sieve_buf_size);
            H5_CHECKED_ASSIGN(dset_contig->sieve_size, size_t, min, hsize_t);

            
            if (dset_contig->sieve_size > len) {
                
                if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, dset_contig->sieve_loc,
                                          dset_contig->sieve_size, dset_contig->sieve_buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");
            } 

            
            H5MM_memcpy(dset_contig->sieve_buf, buf, len);

            
            dset_contig->sieve_dirty = true;

            
            sieve_start = dset_contig->sieve_loc;
            sieve_size  = dset_contig->sieve_size;
            sieve_end   = sieve_start + sieve_size;
        } 
    }     
    else {
        
        contig_end = addr + len - 1;

        
        if (addr >= sieve_start && contig_end < sieve_end) {
            unsigned char *base_sieve_buf = dset_contig->sieve_buf + (addr - sieve_start);

            
            H5MM_memcpy(base_sieve_buf, buf, len);

            
            dset_contig->sieve_dirty = true;
        } 
        
        else {
            
            if (len > dset_contig->sieve_buf_size) {
                
                if ((sieve_start >= addr && sieve_start < (contig_end + 1)) ||
                    ((sieve_end - 1) >= addr && (sieve_end - 1) < (contig_end + 1))) {
                    
                    if (dset_contig->sieve_dirty) {
                        
                        if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, sieve_start, sieve_size,
                                                   dset_contig->sieve_buf) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

                        
                        dset_contig->sieve_dirty = false;
                    } 

                    
                    dset_contig->sieve_loc  = HADDR_UNDEF;
                    dset_contig->sieve_size = 0;
                } 

                
                if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, addr, len, buf) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");
            } 
            
            else {
                
                if (((addr + len) == sieve_start || addr == sieve_end) &&
                    (len + sieve_size) <= dset_contig->sieve_buf_size && dset_contig->sieve_dirty) {
                    
                    if ((addr + len) == sieve_start) {
                        
                        memmove(dset_contig->sieve_buf + len, dset_contig->sieve_buf,
                                dset_contig->sieve_size);

                        
                        H5MM_memcpy(dset_contig->sieve_buf, buf, len);

                        
                        dset_contig->sieve_loc = addr;

                    } 
                    
                    else {
                        
                        H5MM_memcpy(dset_contig->sieve_buf + sieve_size, buf, len);
                    } 

                    
                    dset_contig->sieve_size += len;
                } 
                
                else {
                    
                    if (dset_contig->sieve_dirty) {
                        
                        if (H5F_shared_block_write(f_sh, H5FD_MEM_DRAW, sieve_start, sieve_size,
                                                   dset_contig->sieve_buf) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

                        
                        dset_contig->sieve_dirty = false;
                    } 

                    
                    dset_contig->sieve_loc = addr;

                    
                    if (HADDR_UNDEF == (rel_eoa = H5F_shared_get_eoa(f_sh, H5FD_MEM_DRAW)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine file size");

                    
                    max_data = store_contig->dset_size - dst_off;

                    
                    min = MIN3(rel_eoa - dset_contig->sieve_loc, max_data, dset_contig->sieve_buf_size);
                    H5_CHECKED_ASSIGN(dset_contig->sieve_size, size_t, min, hsize_t);

                    
                    if (dset_contig->sieve_size > len) {
                        
                        if (H5F_shared_block_read(f_sh, H5FD_MEM_DRAW, dset_contig->sieve_loc,
                                                  dset_contig->sieve_size, dset_contig->sieve_buf) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "block read failed");
                    } 

                    
                    H5MM_memcpy(dset_contig->sieve_buf, buf, len);

                    
                    dset_contig->sieve_dirty = true;
                } 
            }     
        }         
    }             

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_writevv_cb(hsize_t dst_off, hsize_t src_off, size_t len, void *_udata)
{
    H5D_contig_writevv_ud_t *udata =
        (H5D_contig_writevv_ud_t *)_udata; 
    herr_t ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    if (H5F_shared_block_write(udata->f_sh, H5FD_MEM_DRAW, (udata->dset_addr + dst_off), len,
                               (udata->wbuf + src_off)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "block write failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5D__contig_writevv(const H5D_io_info_t *io_info, const H5D_dset_io_info_t *dset_info, size_t dset_max_nseq,
                    size_t *dset_curr_seq, size_t dset_len_arr[], hsize_t dset_off_arr[], size_t mem_max_nseq,
                    size_t *mem_curr_seq, size_t mem_len_arr[], hsize_t mem_off_arr[])
{
    ssize_t ret_value = -1; 

    FUNC_ENTER_PACKAGE

    
    assert(io_info);
    assert(dset_info);
    assert(dset_curr_seq);
    assert(dset_len_arr);
    assert(dset_off_arr);
    assert(mem_curr_seq);
    assert(mem_len_arr);
    assert(mem_off_arr);

    
    if (H5F_SHARED_HAS_FEATURE(io_info->f_sh, H5FD_FEAT_DATA_SIEVE)) {
        H5D_contig_writevv_sieve_ud_t udata; 

        
        udata.f_sh         = io_info->f_sh;
        udata.dset_contig  = &(dset_info->dset->shared->cache.contig);
        udata.store_contig = &(dset_info->store->contig);
        udata.wbuf         = (const unsigned char *)dset_info->buf.cvp;

        
        if ((ret_value =
                 H5VM_opvv(dset_max_nseq, dset_curr_seq, dset_len_arr, dset_off_arr, mem_max_nseq,
                           mem_curr_seq, mem_len_arr, mem_off_arr, H5D__contig_writevv_sieve_cb, &udata)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPERATE, FAIL, "can't perform vectorized sieve buffer write");
    } 
    else {
        H5D_contig_writevv_ud_t udata; 

        
        udata.f_sh      = io_info->f_sh;
        udata.dset_addr = dset_info->store->contig.dset_addr;
        udata.wbuf      = (const unsigned char *)dset_info->buf.cvp;

        
        if ((ret_value = H5VM_opvv(dset_max_nseq, dset_curr_seq, dset_len_arr, dset_off_arr, mem_max_nseq,
                                   mem_curr_seq, mem_len_arr, mem_off_arr, H5D__contig_writevv_cb, &udata)) <
            0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTOPERATE, FAIL, "can't perform vectorized read");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_flush(H5D_t *dset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    if (H5D__flush_sieve_buf(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush sieve buffer");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__contig_io_term(H5D_io_info_t H5_ATTR_UNUSED *io_info, H5D_dset_io_info_t *di)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(di);

    
    if (di->layout_io_info.contig_piece_info) {
        if (H5D__free_piece_info(di->layout_io_info.contig_piece_info, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't free piece info");
        di->layout_io_info.contig_piece_info = NULL;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__contig_copy(H5F_t *f_src, const H5O_storage_contig_t *storage_src, H5F_t *f_dst,
                 H5O_storage_contig_t *storage_dst, H5T_t *dt_src, H5O_copy_t *cpy_info)
{
    haddr_t       addr_src;                                    
    haddr_t       addr_dst;                                    
    H5T_path_t   *tpath_src_mem = NULL, *tpath_mem_dst = NULL; 
    H5T_t        *dt_dst      = NULL;                          
    H5T_t        *dt_mem      = NULL;                          
    size_t        src_dt_size = 0;                             
    size_t        mem_dt_size = 0;                             
    size_t        dst_dt_size = 0;                             
    size_t        max_dt_size;                                 
    size_t        nelmts = 0;                                  
    size_t        src_nbytes;                                  
    size_t        mem_nbytes;                                  
    size_t        dst_nbytes;                                  
    hsize_t       total_src_nbytes;                            
    size_t        buf_size;                                    
    void         *buf         = NULL;                          
    void         *bkg         = NULL;                          
    void         *reclaim_buf = NULL;                          
    H5S_t        *buf_space   = NULL;                          
    hsize_t       buf_dim[1]  = {0};                           
    bool          is_vlen     = false; 
    bool          fix_ref     = false; 
    H5D_shared_t *shared_fo =
        (H5D_shared_t *)cpy_info->shared_fo; 
    bool    try_sieve   = false;             
    haddr_t sieve_start = HADDR_UNDEF;       
    haddr_t sieve_end   = HADDR_UNDEF;       
    herr_t  ret_value   = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    assert(f_src);
    assert(storage_src);
    assert(f_dst);
    assert(storage_dst);
    assert(dt_src);

    
    if (H5D__contig_alloc(f_dst, storage_dst) < 0)
        HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to allocate contiguous storage");

    
    
    total_src_nbytes = storage_dst->size;
    H5_CHECK_OVERFLOW(total_src_nbytes, hsize_t, size_t);
    buf_size = MIN(H5D_TEMP_BUF_SIZE, (size_t)total_src_nbytes);

    
    if (H5T_detect_class(dt_src, H5T_VLEN, false) > 0) {
        
        if (NULL == (dt_mem = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");

        
        if (NULL == (dt_dst = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to copy");
        if (H5T_set_loc(dt_dst, H5F_VOL_OBJ(f_dst), H5T_LOC_DISK) < 0) {
            (void)H5T_close_real(dt_dst);
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "cannot mark datatype on disk");
        } 

        
        if (NULL == (tpath_src_mem = H5T_path_find(dt_src, dt_mem)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to convert between src and mem datatypes");
        if (NULL == (tpath_mem_dst = H5T_path_find(dt_mem, dt_dst)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to convert between mem and dst datatypes");

        
        if (0 == (src_dt_size = H5T_get_size(dt_src)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        if (0 == (mem_dt_size = H5T_get_size(dt_mem)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(src_dt_size, mem_dt_size);
        if (0 == (dst_dt_size = H5T_get_size(dt_dst)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to determine datatype size");
        max_dt_size = MAX(max_dt_size, dst_dt_size);

        
        if (0 == (nelmts = buf_size / max_dt_size))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "element size too large");

        
        src_nbytes = nelmts * src_dt_size;
        dst_nbytes = nelmts * dst_dt_size;
        mem_nbytes = nelmts * mem_dt_size;

        
        buf_size = nelmts * max_dt_size;

        
        buf_dim[0] = nelmts;

        
        if (NULL == (buf_space = H5S_create_simple((unsigned)1, buf_dim, NULL)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create simple dataspace");

        
        is_vlen = true;
    } 
    else {
        
        if (H5T_get_class(dt_src, false) == H5T_REFERENCE) {
            
            if (f_src != f_dst)
                fix_ref = true;
        } 

        
        src_nbytes = dst_nbytes = mem_nbytes = buf_size;
    } 

    
    assert(buf_size);
    if (NULL == (buf = H5FL_BLK_MALLOC(type_conv, buf_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for copy buffer");

    
    if (is_vlen || fix_ref) {
        if (NULL == (reclaim_buf = H5FL_BLK_MALLOC(type_conv, buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for copy buffer");

        
        if (NULL == (bkg = H5FL_BLK_MALLOC(type_conv, buf_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed for copy buffer");
    } 

    
    addr_src = storage_src->addr;
    addr_dst = storage_dst->addr;

    
    if (H5F_HAS_FEATURE(f_src, H5FD_FEAT_DATA_SIEVE) && shared_fo && shared_fo->cache.contig.sieve_buf) {
        try_sieve   = true;
        sieve_start = shared_fo->cache.contig.sieve_loc;
        sieve_end   = sieve_start + shared_fo->cache.contig.sieve_size;
    }

    while (total_src_nbytes > 0) {
        
        if (total_src_nbytes < src_nbytes) {
            
            src_nbytes = (size_t)total_src_nbytes;

            
            if (is_vlen) {
                
                nelmts     = src_nbytes / src_dt_size;
                dst_nbytes = nelmts * dst_dt_size;
                mem_nbytes = nelmts * mem_dt_size;

                
                buf_dim[0] = nelmts;

                
                if (H5S_set_extent_real(buf_space, buf_dim) < 0)
                    HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSET, FAIL, "unable to change buffer dataspace size");
            } 
            else
                
                dst_nbytes = mem_nbytes = src_nbytes;
        } 

        
        if (try_sieve && (addr_src >= sieve_start) && ((addr_src + src_nbytes - 1) < sieve_end)) {
            unsigned char *base_sieve_buf = shared_fo->cache.contig.sieve_buf + (addr_src - sieve_start);

            H5MM_memcpy(buf, base_sieve_buf, src_nbytes);
        }
        else
            
            if (H5F_block_read(f_src, H5FD_MEM_DRAW, addr_src, src_nbytes, buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "unable to read raw data");

        
        if (is_vlen) {
            
            if (H5T_convert(tpath_src_mem, dt_src, dt_mem, nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

            
            H5MM_memcpy(reclaim_buf, buf, mem_nbytes);

            
            memset(bkg, 0, buf_size);

            
            if (H5T_convert(tpath_mem_dst, dt_mem, dt_dst, nelmts, (size_t)0, (size_t)0, buf, bkg) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "datatype conversion failed");

            
            if (H5T_reclaim(dt_mem, buf_space, reclaim_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to reclaim variable-length data");
        } 
        else if (fix_ref) {
            
            if (cpy_info->expand_ref) {
                
                if (H5O_copy_expand_ref(f_src, dt_src, buf, buf_size, f_dst, bkg, cpy_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy reference attribute");

                
                H5MM_memcpy(buf, bkg, buf_size);
            } 
            else
                
                memset(buf, 0, src_nbytes);
        } 

        
        if (H5F_block_write(f_dst, H5FD_MEM_DRAW, addr_dst, dst_nbytes, buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to write raw data");

        
        addr_src += src_nbytes;
        addr_dst += dst_nbytes;
        total_src_nbytes -= src_nbytes;
    } 

done:
    if (dt_dst && (H5T_close(dt_dst) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (dt_mem && (H5T_close(dt_mem) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary datatype");
    if (buf_space && H5S_close(buf_space) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "can't close temporary dataspace");
    if (buf)
        buf = H5FL_BLK_FREE(type_conv, buf);
    if (reclaim_buf)
        reclaim_buf = H5FL_BLK_FREE(type_conv, reclaim_buf);
    if (bkg)
        bkg = H5FL_BLK_FREE(type_conv, bkg);

    FUNC_LEAVE_NOAPI(ret_value)
} 
