/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

H5FL_DEFINE_STATIC(H5EA_dblock_t);

H5EA_dblock_t *
H5EA__dblock_alloc(H5EA_hdr_t *hdr, void *parent, size_t nelmts)
{
    H5EA_dblock_t *dblock    = NULL; 
    H5EA_dblock_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(parent);
    assert(nelmts > 0);

    
    if (NULL == (dblock = H5FL_CALLOC(H5EA_dblock_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array data block");

    
    if (H5EA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    dblock->hdr = hdr;

    
    dblock->parent = parent;
    dblock->nelmts = nelmts;

    
    if (nelmts > hdr->dblk_page_nelmts) {
        
        dblock->npages = nelmts / hdr->dblk_page_nelmts;
        assert(nelmts == (dblock->npages * hdr->dblk_page_nelmts));
    } 
    else {
        
        if (NULL == (dblock->elmts = H5EA__hdr_alloc_elmts(hdr, nelmts)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for data block element buffer");
    } 

    
    ret_value = dblock;

done:
    if (!ret_value)
        if (dblock && H5EA__dblock_dest(dblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5EA__dblock_create(H5EA_hdr_t *hdr, void *parent, bool *stats_changed, hsize_t dblk_off, size_t nelmts)
{
    H5EA_dblock_t *dblock = NULL;     
    haddr_t        dblock_addr;       
    bool           inserted  = false; 
    haddr_t        ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(stats_changed);
    assert(nelmts > 0);

    
    if (NULL == (dblock = H5EA__dblock_alloc(hdr, parent, nelmts)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for extensible array data block");

    
    dblock->size = H5EA_DBLOCK_SIZE(dblock);

    
    dblock->block_off = dblk_off;

    
    if (HADDR_UNDEF == (dblock_addr = H5MF_alloc(hdr->f, H5FD_MEM_EARRAY_DBLOCK, (hsize_t)dblock->size)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "file allocation failed for extensible array data block");
    dblock->addr = dblock_addr;

    
    if (!dblock->npages)
        
        if ((hdr->cparam.cls->fill)(dblock->elmts, (size_t)dblock->nelmts) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "can't set extensible array data block elements to class's fill value");

    
    if (H5AC_insert_entry(hdr->f, H5AC_EARRAY_DBLOCK, dblock_addr, dblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINSERT, HADDR_UNDEF,
                    "can't add extensible array data block to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add extensible array entry as child of array proxy");
        dblock->top_proxy = hdr->top_proxy;
    } 

    
    hdr->stats.stored.ndata_blks++;
    hdr->stats.stored.data_blk_size += dblock->size;

    
    hdr->stats.stored.nelmts += nelmts;

    
    *stats_changed = true;

    
    ret_value = dblock_addr;

done:
    if (!H5_addr_defined(ret_value))
        if (dblock) {
            
            if (inserted)
                if (H5AC_remove_entry(dblock) < 0)
                    HDONE_ERROR(H5E_EARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove extensible array data block from cache");

            
            if (H5_addr_defined(dblock->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_EARRAY_DBLOCK, dblock->addr, (hsize_t)dblock->size) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to release extensible array data block");

            
            if (H5EA__dblock_dest(dblock) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to destroy extensible array data block");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5EA__dblock_sblk_idx(const H5EA_hdr_t *hdr, hsize_t idx)
{
    unsigned sblk_idx = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(idx >= hdr->cparam.idx_blk_elmts);

    
    idx -= hdr->cparam.idx_blk_elmts;

    
    H5_CHECK_OVERFLOW(idx,  hsize_t,  uint64_t);
    sblk_idx = H5VM_log2_gen((uint64_t)((idx / hdr->cparam.data_blk_min_elmts) + 1));

    FUNC_LEAVE_NOAPI(sblk_idx)
} 

H5EA_dblock_t *
H5EA__dblock_protect(H5EA_hdr_t *hdr, void *parent, haddr_t dblk_addr, size_t dblk_nelmts, unsigned flags)
{
    H5EA_dblock_t         *dblock; 
    H5EA_dblock_cache_ud_t udata;  
    H5EA_dblock_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(dblk_addr));
    assert(dblk_nelmts);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.hdr       = hdr;
    udata.parent    = parent;
    udata.nelmts    = dblk_nelmts;
    udata.dblk_addr = dblk_addr;

    
    if (NULL ==
        (dblock = (H5EA_dblock_t *)H5AC_protect(hdr->f, H5AC_EARRAY_DBLOCK, dblk_addr, &udata, flags)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect extensible array data block, address = %llu",
                    (unsigned long long)dblk_addr);

    
    if (hdr->top_proxy && NULL == dblock->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, NULL,
                        "unable to add extensible array entry as child of array proxy");
        dblock->top_proxy = hdr->top_proxy;
    }

    
    ret_value = dblock;

done:

    
    if (!ret_value) {
        
        if (dblock &&
            H5AC_unprotect(hdr->f, H5AC_EARRAY_DBLOCK, dblock->addr, dblock, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect extensible array data block, address = %llu",
                        (unsigned long long)dblock->addr);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblock_unprotect(H5EA_dblock_t *dblock, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    if (H5AC_unprotect(dblock->hdr->f, H5AC_EARRAY_DBLOCK, dblock->addr, dblock, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect extensible array data block, address = %llu",
                    (unsigned long long)dblock->addr);

done:

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblock_delete(H5EA_hdr_t *hdr, void *parent, haddr_t dblk_addr, size_t dblk_nelmts)
{
    H5EA_dblock_t *dblock    = NULL; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(parent);
    assert(H5_addr_defined(dblk_addr));
    assert(dblk_nelmts > 0);

    
    if (NULL == (dblock = H5EA__dblock_protect(hdr, parent, dblk_addr, dblk_nelmts, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array data block, address = %llu",
                    (unsigned long long)dblk_addr);

    
    if (dblk_nelmts > hdr->dblk_page_nelmts) {
        size_t  npages = dblk_nelmts / hdr->dblk_page_nelmts; 
        haddr_t dblk_page_addr;                               
        size_t  dblk_page_size;                               
        size_t  u;                                            

        
        dblk_page_addr = dblk_addr + H5EA_DBLOCK_PREFIX_SIZE(dblock);
        dblk_page_size = (hdr->dblk_page_nelmts * hdr->cparam.raw_elmt_size) + H5EA_SIZEOF_CHKSUM;

        
        for (u = 0; u < npages; u++) {
            
            
            if (H5AC_expunge_entry(hdr->f, H5AC_EARRAY_DBLK_PAGE, dblk_page_addr, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTEXPUNGE, FAIL,
                            "unable to remove array data block page from metadata cache");

            
            dblk_page_addr += dblk_page_size;
        } 
    }     

done:
    
    if (dblock && H5EA__dblock_unprotect(dblock, H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG |
                                                     H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblock_dest(H5EA_dblock_t *dblock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);
    assert(!dblock->has_hdr_depend);

    
    if (dblock->hdr) {
        
        if (dblock->elmts && !dblock->npages) {
            
            assert(dblock->nelmts > 0);
            if (H5EA__hdr_free_elmts(dblock->hdr, dblock->nelmts, dblock->elmts) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL,
                            "unable to free extensible array data block element buffer");
            dblock->elmts  = NULL;
            dblock->nelmts = 0;
        } 

        
        if (H5EA__hdr_decr(dblock->hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        dblock->hdr = NULL;
    } 

    
    assert(NULL == dblock->top_proxy);

    
    dblock = H5FL_FREE(H5EA_dblock_t, dblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
