/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5FLprivate.h" 

H5FL_DEFINE_STATIC(H5FA_dblk_page_t);

H5FL_BLK_DEFINE(page_elmts);

H5FA_dblk_page_t *
H5FA__dblk_page_alloc(H5FA_hdr_t *hdr, size_t nelmts)
{
    H5FA_dblk_page_t *dblk_page = NULL; 
    H5FA_dblk_page_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (dblk_page = H5FL_CALLOC(H5FA_dblk_page_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for fixed array data block page");

    
    if (H5FA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    dblk_page->hdr = hdr;

    
    dblk_page->nelmts = nelmts;

    
    if (NULL == (dblk_page->elmts = H5FL_BLK_MALLOC(page_elmts, nelmts * hdr->cparam.cls->nat_elmt_size)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for data block page element buffer");

    
    ret_value = dblk_page;

done:

    if (!ret_value)
        if (dblk_page && H5FA__dblk_page_dest(dblk_page) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblk_page_create(H5FA_hdr_t *hdr, haddr_t addr, size_t nelmts)
{
    H5FA_dblk_page_t *dblk_page = NULL;  
    bool              inserted  = false; 
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

#ifdef H5FA_DEBUG
    Rfprintf(Rstderr, "%s: Called, addr = %" PRIuHADDR "\n", __func__, addr);
#endif 

    
    assert(hdr);

    
    if (NULL == (dblk_page = H5FA__dblk_page_alloc(hdr, nelmts)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, FAIL,
                    "memory allocation failed for fixed array data block page");

    
    dblk_page->addr = addr;
    dblk_page->size = H5FA_DBLK_PAGE_SIZE(hdr, nelmts);
#ifdef H5FA_DEBUG
    Rfprintf(Rstderr, "%s: dblk_page->size = %llu\n", __func__, (unsigned long long)dblk_page->size);
#endif 

    
    if ((hdr->cparam.cls->fill)(dblk_page->elmts, nelmts) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, FAIL,
                    "can't set fixed array data block page elements to class's fill value");

    
    if (H5AC_insert_entry(hdr->f, H5AC_FARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINSERT, FAIL, "can't add fixed array data block page to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblk_page) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, FAIL,
                        "unable to add fixed array entry as child of array proxy");
        dblk_page->top_proxy = hdr->top_proxy;
    } 

done:
    if (ret_value < 0)
        if (dblk_page) {
            
            if (inserted)
                if (H5AC_remove_entry(dblk_page) < 0)
                    HDONE_ERROR(H5E_FARRAY, H5E_CANTREMOVE, FAIL,
                                "unable to remove fixed array data block page from cache");

            
            if (H5FA__dblk_page_dest(dblk_page) < 0)
                HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, FAIL, "unable to destroy fixed array data block page");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FA_dblk_page_t *
H5FA__dblk_page_protect(H5FA_hdr_t *hdr, haddr_t dblk_page_addr, size_t dblk_page_nelmts, unsigned flags)
{
    H5FA_dblk_page_t         *dblk_page = NULL; 
    H5FA_dblk_page_cache_ud_t udata;            
    H5FA_dblk_page_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

#ifdef H5FA_DEBUG
    Rfprintf(Rstderr, "%s: Called\n", __func__);
#endif 

    
    assert(hdr);
    assert(H5_addr_defined(dblk_page_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.hdr            = hdr;
    udata.nelmts         = dblk_page_nelmts;
    udata.dblk_page_addr = dblk_page_addr;

    
    if (NULL == (dblk_page = (H5FA_dblk_page_t *)H5AC_protect(hdr->f, H5AC_FARRAY_DBLK_PAGE, dblk_page_addr,
                                                              &udata, flags)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect fixed array data block page, address = %llu",
                    (unsigned long long)dblk_page_addr);

    
    if (hdr->top_proxy && NULL == dblk_page->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblk_page) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, NULL,
                        "unable to add fixed array entry as child of array proxy");
        dblk_page->top_proxy = hdr->top_proxy;
    } 

    
    ret_value = dblk_page;

done:

    
    if (!ret_value) {
        
        if (dblk_page &&
            H5AC_unprotect(hdr->f, H5AC_FARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect fixed array data block page, address = %llu",
                        (unsigned long long)dblk_page->addr);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblk_page_unprotect(H5FA_dblk_page_t *dblk_page, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

#ifdef H5FA_DEBUG
    Rfprintf(Rstderr, "%s: Called\n", __func__);
#endif 

    
    assert(dblk_page);

    
    if (H5AC_unprotect(dblk_page->hdr->f, H5AC_FARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, cache_flags) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect fixed array data block page, address = %llu",
                    (unsigned long long)dblk_page->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblk_page_dest(H5FA_dblk_page_t *dblk_page)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk_page);

    
    if (dblk_page->hdr) {
        
        if (dblk_page->elmts) {
            
            dblk_page->elmts = H5FL_BLK_FREE(page_elmts, dblk_page->elmts);
        } 

        
        if (H5FA__hdr_decr(dblk_page->hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        dblk_page->hdr = NULL;
    } 

    
    assert(NULL == dblk_page->top_proxy);

    
    dblk_page = H5FL_FREE(H5FA_dblk_page_t, dblk_page);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
