/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5SLprivate.h" 

#define H5F_EFC_TAG_DEFAULT   (-1)
#define H5F_EFC_TAG_LOCK      (-2)
#define H5F_EFC_TAG_CLOSE     (-3)
#define H5F_EFC_TAG_DONTCLOSE (-4)

typedef struct H5F_efc_ent_t {
    char                 *name;     
    H5F_t                *file;     
    struct H5F_efc_ent_t *LRU_next; 
    struct H5F_efc_ent_t *LRU_prev; 
    unsigned              nopen;    
} H5F_efc_ent_t;

struct H5F_efc_t {
    H5SL_t        *slist;      
    H5F_efc_ent_t *LRU_head;   
    H5F_efc_ent_t *LRU_tail;   
    unsigned       nfiles;     
    unsigned       max_nfiles; 
    unsigned       nrefs;      
    int            tag;        
    H5F_shared_t  *tmp_next;   
};

static herr_t H5F__efc_open_file(bool try, H5F_t **file, const char *name, unsigned flags, hid_t fcpl_id,
                                 hid_t fapl_id);
static herr_t H5F__efc_release_real(H5F_efc_t *efc);
static herr_t H5F__efc_remove_ent(H5F_efc_t *efc, H5F_efc_ent_t *ent);
static void   H5F__efc_try_close_tag1(H5F_shared_t *sf, H5F_shared_t **tail);
static void   H5F__efc_try_close_tag2(H5F_shared_t *sf, H5F_shared_t **tail);

H5FL_DEFINE_STATIC(H5F_efc_ent_t);
H5FL_DEFINE_STATIC(H5F_efc_t);

H5F_efc_t *
H5F__efc_create(unsigned max_nfiles)
{
    H5F_efc_t *efc       = NULL; 
    H5F_efc_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(max_nfiles > 0);

    
    if (NULL == (efc = H5FL_CALLOC(H5F_efc_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    efc->max_nfiles = max_nfiles;

    
    efc->tag = H5F_EFC_TAG_DEFAULT;

    
    ret_value = efc;

done:
    if (ret_value == NULL && efc)
        efc = H5FL_FREE(H5F_efc_t, efc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__efc_open_file(bool try, H5F_t **_file, const char *name, unsigned flags, hid_t fcpl_id, hid_t fapl_id)
{
    H5F_t *file      = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    *_file = NULL;

    
    if (H5F_open(try, &file, name, flags, fcpl_id, fapl_id) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, FAIL, "can't open file");

    
    if (NULL == file) {
        assert(try);
        HGOTO_DONE(SUCCEED);
    }

    
    if (H5F__post_open(file) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "can't finish opening file");

    
    file->nopen_objs++;

    
    *_file = file;

done:
    if (ret_value < 0)
        if (file)
            if (H5F_try_close(file, NULL) < 0)
                HDONE_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close external file");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__efc_open(bool try, H5F_efc_t *efc, H5F_t **_file, const char *name, unsigned flags, hid_t fcpl_id,
              hid_t fapl_id)
{
    H5F_efc_ent_t        *ent       = NULL;    
    bool                  open_file = false;   
    H5P_genplist_t       *plist;               
    H5VL_connector_prop_t connector_prop;      
    herr_t                ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(_file);
    assert(name);

    
    *_file = NULL;

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_FILE, H5E_BADTYPE, FAIL, "not a file access property list");
    if (H5P_peek(plist, H5F_ACS_VOL_CONN_NAME, &connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get VOL connector info");

    
    if (H5CX_set_vol_connector_prop(&connector_prop) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTSET, FAIL, "can't set VOL connector info in API context");

    
    if (!efc) {
        if (H5F__efc_open_file(try, _file, name, flags, fcpl_id, fapl_id) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, FAIL, "can't try opening file");

        
        if (NULL == *_file)
            assert(try);

        
        HGOTO_DONE(SUCCEED);
    } 

    
    if (efc->slist) {
        if (efc->nfiles > 0)
            ent = (H5F_efc_ent_t *)H5SL_search(efc->slist, name);
    } 
    else {
        assert(efc->nfiles == 0);
        if (NULL == (efc->slist = H5SL_create(H5SL_TYPE_STR, NULL)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTCREATE, FAIL, "can't create skip list");
    } 

    
    if (ent) {
        assert(efc->LRU_head);
        assert(efc->LRU_tail);

        
        if (ent->LRU_prev) {
            assert(efc->LRU_head != ent);

            
            if (ent->LRU_next)
                ent->LRU_next->LRU_prev = ent->LRU_prev;
            else {
                assert(efc->LRU_tail == ent);
                efc->LRU_tail = ent->LRU_prev;
            } 
            ent->LRU_prev->LRU_next = ent->LRU_next;

            
            ent->LRU_next           = efc->LRU_head;
            ent->LRU_next->LRU_prev = ent;
            ent->LRU_prev           = NULL;
            efc->LRU_head           = ent;
        } 

        
        ent->nopen++;
    } 
    else {
        
        if (efc->nfiles == efc->max_nfiles) {
            
            for (ent = efc->LRU_tail; ent && ent->nopen; ent = ent->LRU_prev)
                ;

            
            if (ent) {
                if (H5F__efc_remove_ent(efc, ent) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_CANTREMOVE, FAIL,
                                "can't remove entry from external file cache");

                
            } 
            else {
                
                if (H5F__efc_open_file(try, _file, name, flags, fcpl_id, fapl_id) < 0)
                    HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, FAIL, "can't try opening file");

                
                if (NULL == *_file)
                    assert(try);

                
                HGOTO_DONE(SUCCEED);
            } 
        }     
        else
            
            if (NULL == (ent = H5FL_MALLOC(H5F_efc_ent_t)))
                HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL, "memory allocation failed");

        
        ent->file = NULL;
        ent->name = NULL;

        
        if (H5F__efc_open_file(try, &ent->file, name, flags, fcpl_id, fapl_id) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, FAIL, "can't try opening file");

        
        if (NULL == ent->file) {
            
            assert(try);

            ent = H5FL_FREE(H5F_efc_ent_t, ent);
            HGOTO_DONE(SUCCEED);
        }
        else
            open_file = true;

        
        if (NULL == (ent->name = H5MM_strdup(name)))
            HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, FAIL, "memory allocation failed");

        
        
        if (H5SL_insert(efc->slist, ent, ent->name) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTINSERT, FAIL, "can't insert entry into skip list");

        
        ent->LRU_next = efc->LRU_head;
        if (ent->LRU_next)
            ent->LRU_next->LRU_prev = ent;
        ent->LRU_prev = NULL;
        efc->LRU_head = ent;
        if (!efc->LRU_tail) {
            assert(!ent->LRU_next);
            efc->LRU_tail = ent;
        } 

        
        ent->nopen = 1;

        
        efc->nfiles++;
        if (ent->file->shared->efc)
            ent->file->shared->efc->nrefs++;
    } 

    
    *_file = ent->file;

done:
    if (ret_value < 0)
        if (ent) {
            if (open_file) {
                ent->file->nopen_objs--;
                if (H5F_try_close(ent->file, NULL) < 0)
                    HDONE_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close external file");
            } 
            ent->name = H5MM_xfree(ent->name);
            ent       = H5FL_FREE(H5F_efc_ent_t, ent);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_efc_close(H5F_t *parent, H5F_t *file)
{
    H5F_efc_t     *efc       = NULL;    
    H5F_efc_ent_t *ent       = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(parent);
    assert(parent->shared);
    assert(file);
    assert(file->shared);

    
    efc = parent->shared->efc;

    
    if (!efc) {
        file->nopen_objs--;
        if (H5F_try_close(file, NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close external file");

        HGOTO_DONE(SUCCEED);
    } 

    
    for (ent = efc->LRU_head; ent && ent->file != file; ent = ent->LRU_next)
        ;
    if (!ent) {
        file->nopen_objs--;
        if (H5F_try_close(file, NULL) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close external file");
    } 
    else
        
        ent->nopen--;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5F__efc_max_nfiles(H5F_efc_t *efc)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(efc);
    assert(efc->max_nfiles > 0);

    FUNC_LEAVE_NOAPI(efc->max_nfiles)
} 

static herr_t
H5F__efc_release_real(H5F_efc_t *efc)
{
    H5F_efc_ent_t *ent       = NULL;    
    H5F_efc_ent_t *prev_ent  = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(efc);

    
    assert((efc->tag == H5F_EFC_TAG_DEFAULT) || (efc->tag == H5F_EFC_TAG_CLOSE));
    efc->tag = H5F_EFC_TAG_LOCK;

    
    ent = efc->LRU_head;
    while (ent)
        if (!ent->nopen) {
            if (H5F__efc_remove_ent(efc, ent) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTREMOVE, FAIL, "can't remove entry from external file cache");

            
            prev_ent = ent;
            ent      = ent->LRU_next;
            prev_ent = H5FL_FREE(H5F_efc_ent_t, prev_ent);
        } 
        else
            
            ent = ent->LRU_next;

    
    efc->tag = H5F_EFC_TAG_DEFAULT;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__efc_release(H5F_efc_t *efc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(efc);

    
    if (H5F__efc_release_real(efc) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "can't remove entry from external file cache");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F__efc_destroy(H5F_efc_t *efc)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(efc);

    if (efc->nfiles > 0) {
        
        if (H5F__efc_release_real(efc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "can't release external file cache");

        
        if (efc->nfiles > 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "can't destroy EFC after incomplete release");
    } 

    assert(efc->nfiles == 0);
    assert(efc->LRU_head == NULL);
    assert(efc->LRU_tail == NULL);

    
    if (efc->slist)
        if (H5SL_close(efc->slist) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "can't close skip list");

    
    (void)H5FL_FREE(H5F_efc_t, efc);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5F__efc_remove_ent(H5F_efc_t *efc, H5F_efc_ent_t *ent)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(efc);
    assert(efc->slist);
    assert(ent);

    
    if (ent != H5SL_remove(efc->slist, ent->name))
        HGOTO_ERROR(H5E_FILE, H5E_CANTDELETE, FAIL, "can't delete entry from skip list");

    
    if (ent->LRU_next)
        ent->LRU_next->LRU_prev = ent->LRU_prev;
    else {
        assert(efc->LRU_tail == ent);
        efc->LRU_tail = ent->LRU_prev;
    } 
    if (ent->LRU_prev)
        ent->LRU_prev->LRU_next = ent->LRU_next;
    else {
        assert(efc->LRU_head == ent);
        efc->LRU_head = ent->LRU_next;
    } 

    
    efc->nfiles--;
    if (ent->file->shared->efc)
        ent->file->shared->efc->nrefs--;

    
    ent->name = (char *)H5MM_xfree(ent->name);

    
    ent->file->nopen_objs--;
    if (H5F_try_close(ent->file, NULL) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTCLOSEFILE, FAIL, "can't close external file");
    ent->file = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void
H5F__efc_try_close_tag1(H5F_shared_t *sf, H5F_shared_t **tail)
{
    H5F_efc_ent_t *ent = NULL; 
    H5F_shared_t  *esf;        

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sf);
    assert(sf->efc);
    assert((sf->efc->tag > 0) || (sf->nrefs == sf->efc->nrefs));
    assert(sf->efc->tag != H5F_EFC_TAG_LOCK);
    assert(tail);
    assert(*tail);

    
    for (ent = sf->efc->LRU_head; ent; ent = ent->LRU_next) {
        esf = ent->file->shared;

        if (esf->efc) {
            
            assert(esf->efc->tag != 0);

            
            if (esf->efc->tag > 0)
                esf->efc->tag--;
            
            else if ((esf->nrefs == esf->efc->nrefs) && (esf->efc->tag != H5F_EFC_TAG_LOCK) &&
                     !(ent->nopen)) {
                
                assert(esf->efc->tmp_next == NULL);

                
                if (esf->nrefs > 1) {
                    (*tail)->efc->tmp_next = esf;
                    *tail                  = esf;
                    esf->efc->tag          = (int)esf->nrefs - 1;
                } 

                
                H5F__efc_try_close_tag1(ent->file->shared, tail);
            } 
        }     
    }         

    FUNC_LEAVE_NOAPI_VOID
} 

static void
H5F__efc_try_close_tag2(H5F_shared_t *sf, H5F_shared_t **tail)
{
    H5F_efc_ent_t *ent = NULL; 
    H5F_shared_t  *esf;        

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sf);
    assert(sf->efc);

    
    for (ent = sf->efc->LRU_head; ent; ent = ent->LRU_next) {
        esf = ent->file->shared;

        
        if ((esf->efc) &&
            ((esf->efc->tag == H5F_EFC_TAG_CLOSE) ||
             ((esf->efc->tag == H5F_EFC_TAG_DEFAULT) && (esf->nrefs == esf->efc->nrefs) && !(ent->nopen)))) {
            
            assert(((esf->nrefs > 1) && ((esf->efc->tag == H5F_EFC_TAG_CLOSE))) ||
                   ((esf->nrefs == 1) && (esf->efc->tag == H5F_EFC_TAG_DEFAULT)));

            
            if (esf->efc->tag != H5F_EFC_TAG_DONTCLOSE) {
                
                if (esf->efc->tag == H5F_EFC_TAG_CLOSE) {
                    esf->efc->tag          = H5F_EFC_TAG_DONTCLOSE;
                    esf->efc->tmp_next     = NULL;
                    (*tail)->efc->tmp_next = esf;
                    *tail                  = esf;
                } 

                
                H5F__efc_try_close_tag2(esf, tail);
            } 
        }     
    }         

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5F__efc_try_close(H5F_t *f)
{
    H5F_shared_t *tail; 
    H5F_shared_t *uncloseable_head =
        NULL; 
    H5F_shared_t *uncloseable_tail =
        NULL;           
    H5F_shared_t *sf;   
    H5F_shared_t *next; 
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);
    assert(f->shared->efc);
    assert(f->shared->nrefs > f->shared->efc->nrefs);
    assert(f->shared->nrefs > 1);
    assert(f->shared->efc->tag < 0);

    if (f->shared->efc->tag == H5F_EFC_TAG_CLOSE) {
        
        if (H5F__efc_release_real(f->shared->efc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "can't release external file cache");

        
        assert(f->shared->efc->nfiles == 0);

        HGOTO_DONE(SUCCEED);
    } 

    
    
    
    
    if ((f->shared->nrefs != f->shared->efc->nrefs + 1) || (f->shared->efc->tag == H5F_EFC_TAG_DONTCLOSE) ||
        (f->shared->efc->nfiles == 0))
        
        HGOTO_DONE(SUCCEED);

    
    
    assert(f->shared->efc->tag == H5F_EFC_TAG_DEFAULT);
    assert(f->shared->efc->tmp_next == NULL);

    
    tail = f->shared;

    
    f->shared->efc->tag = (int)f->shared->efc->nrefs;

    
    H5F__efc_try_close_tag1(f->shared, &tail);

    
    if (f->shared->efc->tag > 0) {
        sf = f->shared;
        while (sf) {
            next              = sf->efc->tmp_next;
            sf->efc->tag      = H5F_EFC_TAG_DEFAULT;
            sf->efc->tmp_next = NULL;
            sf                = next;
        } 
        HGOTO_DONE(SUCCEED);
    } 

    
    sf   = f->shared;
    tail = NULL;
    while (sf) {
        assert(sf->efc->tag >= 0);
        next = sf->efc->tmp_next;
        if (sf->efc->tag > 0) {
            
            assert(tail);
            tail->efc->tmp_next = sf->efc->tmp_next;
            sf->efc->tmp_next   = NULL;

            
            if (!uncloseable_head)
                uncloseable_head = sf;
            else
                uncloseable_tail->efc->tmp_next = sf;
            uncloseable_tail = sf;

            
            sf->efc->tag = H5F_EFC_TAG_DONTCLOSE;
        } 
        else {
            sf->efc->tag = H5F_EFC_TAG_CLOSE;
            tail         = sf;
        } 
        sf = next;
    } 

    
    sf = uncloseable_head;
    if (sf) {
        tail = uncloseable_tail;
        assert(tail);
        while (sf != tail->efc->tmp_next) {
            H5F__efc_try_close_tag2(sf, &uncloseable_tail);
            sf = sf->efc->tmp_next;
        } 
    }     

    
    if (f->shared->efc->tag == H5F_EFC_TAG_CLOSE) {
        if (H5F__efc_release_real(f->shared->efc) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTRELEASE, FAIL, "can't release external file cache");

        
        assert(f->shared->nrefs == 1);
    } 

    
    if (uncloseable_head) {
        sf = uncloseable_head;
        while (sf) {
            next = sf->efc->tmp_next;
            assert(sf->efc->tag == H5F_EFC_TAG_DONTCLOSE);
            sf->efc->tag      = H5F_EFC_TAG_DEFAULT;
            sf->efc->tmp_next = NULL;
            sf                = next;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
