/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Fmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FDprivate.h" 

unsigned
H5F_shared_get_intent(const H5F_shared_t *f_sh)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f_sh);

    FUNC_LEAVE_NOAPI(f_sh->flags)
} 

unsigned
H5F_get_intent(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared->flags)
} 

H5F_libver_t
H5F_get_low_bound(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared->low_bound)
} 

H5F_libver_t
H5F_get_high_bound(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared->high_bound)
} 

char *
H5F_get_open_name(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->open_name);

    FUNC_LEAVE_NOAPI(f->open_name)
} 

char *
H5F_get_actual_name(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->actual_name);

    FUNC_LEAVE_NOAPI(f->actual_name)
} 

char *
H5F_get_extpath(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);
    assert(f->shared->extpath);

    FUNC_LEAVE_NOAPI(f->shared->extpath)
} 

H5F_shared_t *
H5F_get_shared(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared)
} 

bool
H5F_same_shared(const H5F_t *f1, const H5F_t *f2)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f1);
    assert(f1->shared);
    assert(f2);
    assert(f2->shared);

    FUNC_LEAVE_NOAPI(f1->shared == f2->shared)
} 

unsigned
H5F_get_nopen_objs(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->nopen_objs)
} 

bool
H5F_file_id_exists(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->id_exists)
} 

H5F_t *
H5F_get_parent(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->parent)
} 

unsigned
H5F_get_nmounts(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->nmounts)
} 

unsigned
H5F_get_read_attempts(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared->read_attempts)
} 

hid_t
H5F_get_fcpl(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->fcpl_id)
} 

uint8_t
H5F_sizeof_addr(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sizeof_addr)
} 

uint8_t
H5F_sizeof_size(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sizeof_size)
} 

haddr_t
H5F_get_sohm_addr(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sohm_addr)
} 

unsigned
H5F_get_sohm_vers(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sohm_vers)
} 

unsigned
H5F_get_sohm_nindexes(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sohm_nindexes)
} 

unsigned
H5F_sym_leaf_k(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    FUNC_LEAVE_NOAPI(f->shared->sblock->sym_leaf_k)
} 

bool
H5F_get_min_dset_ohdr(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->shared->crt_dset_min_ohdr_flag)
} 

unsigned
H5F_kvalue(const H5F_t *f, const H5B_class_t *type)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);
    assert(type);

    FUNC_LEAVE_NOAPI(f->shared->sblock->btree_k[type->id])
} 

unsigned
H5F_get_nrefs(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->nrefs)
} 

size_t
H5F_rdcc_nslots(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->rdcc_nslots)
} 

size_t
H5F_rdcc_nbytes(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->rdcc_nbytes)
} 

double
H5F_rdcc_w0(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->rdcc_w0)
} 

haddr_t
H5F_get_base_addr(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    FUNC_LEAVE_NOAPI(f->shared->sblock->base_addr)
} 

H5UC_t *
H5F_grp_btree_shared(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->grp_btree_shared)
} 

size_t
H5F_sieve_buf_size(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->sieve_buf_size)
} 

unsigned
H5F_gc_ref(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->gc_ref)
} 

H5F_close_degree_t
H5F_get_fc_degree(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->fc_degree)
} 

bool
H5F_get_evict_on_close(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->evict_on_close)
} 

bool
H5F_store_msg_crt_idx(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->store_msg_crt_idx)
} 

bool
H5F_shared_has_feature(const H5F_shared_t *f_sh, unsigned feature)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f_sh);

    FUNC_LEAVE_NOAPI((bool)(f_sh->lf->feature_flags & feature))
} 

bool
H5F_has_feature(const H5F_t *f, unsigned feature)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI((bool)(f->shared->lf->feature_flags & feature))
} 

hid_t
H5F_get_driver_id(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);
    assert(f->shared->lf);

    FUNC_LEAVE_NOAPI(f->shared->lf->driver_id)
} 

herr_t
H5F_get_fileno(const H5F_t *f, unsigned long *filenum)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    assert(f);
    assert(f->shared);
    assert(f->shared->lf);
    assert(filenum);

    
    if (H5FD_get_fileno(f->shared->lf, filenum) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_BADRANGE, FAIL, "can't retrieve fileno");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5F_shared_get_eoa(const H5F_shared_t *f_sh, H5FD_mem_t type)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    assert(f_sh);

    
    if (HADDR_UNDEF == (ret_value = H5FD_get_eoa(f_sh->lf, type)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "driver get_eoa request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5F_get_eoa(const H5F_t *f, H5FD_mem_t type)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    assert(f);
    assert(f->shared);

    
    if (HADDR_UNDEF == (ret_value = H5FD_get_eoa(f->shared->lf, type)))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "driver get_eoa request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5F_shared_get_file_driver(const H5F_shared_t *f_sh, H5FD_t **file_handle)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f_sh);
    assert(file_handle);

    *file_handle = f_sh->lf;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5F_get_vfd_handle(const H5F_t *file, hid_t fapl, void **file_handle)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file_handle);

    
    if (H5FD_get_vfd_handle(file->shared->lf, fapl, file_handle) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get file handle for file driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5F_is_tmp_addr(const H5F_t *f, haddr_t addr)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(H5_addr_le(f->shared->tmp_addr, addr))
} 

bool
H5F_use_tmp_space(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->use_tmp_space)
} 

#ifdef H5_HAVE_PARALLEL

H5P_coll_md_read_flag_t
H5F_coll_md_read(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f && f->shared);

    FUNC_LEAVE_NOAPI(f->shared->coll_md_read)
} 

H5P_coll_md_read_flag_t
H5F_shared_coll_md_read(const H5F_shared_t *f_sh)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f_sh);

    FUNC_LEAVE_NOAPI(f_sh->coll_md_read)
} 

herr_t
H5F_shared_get_mpi_file_sync_required(const H5F_shared_t *f_sh, bool *flag )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(f_sh);
    assert(flag);

    
    if ((ret_value = H5FD_mpi_get_file_sync_required(f_sh->lf, flag)) < 0)
        HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "driver get_file_sync_required request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

bool
H5F_use_mdc_logging(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->use_mdc_logging)
} 

bool
H5F_start_mdc_log_on_access(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->start_mdc_log_on_access)
} 

char *
H5F_mdc_log_location(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->mdc_log_location)
} 

hsize_t
H5F_get_alignment(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->alignment)
} 

hsize_t
H5F_get_threshold(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->threshold)
} 

hsize_t
H5F_get_pgend_meta_thres(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->pgend_meta_thres)
} 

bool
H5F_get_point_of_no_return(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->point_of_no_return)
} 

bool
H5F_get_null_fsm_addr(const H5F_t *f)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->null_fsm_addr)
} 

H5VL_object_t *
H5F_get_vol_obj(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);

    FUNC_LEAVE_NOAPI(f->vol_obj)
} 

herr_t
H5F__get_cont_info(const H5F_t *f, H5VL_file_cont_info_t *info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(f->shared);

    
    if (info->version != H5VL_CONTAINER_INFO_VERSION)
        HGOTO_ERROR(H5E_FILE, H5E_VERSION, FAIL, "wrong container info version #");

    
    info->feature_flags = 0; 
    info->token_size    = H5F_SIZEOF_ADDR(f);
    info->blob_id_size  = H5HG_HEAP_ID_SIZE(f);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5F_get_use_file_locking(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->use_file_locking)
} 

bool
H5F_has_vector_select_io(const H5F_t *f, bool is_write)
{
    bool ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    if (is_write)
        ret_value = (f->shared->lf->cls->write_vector != NULL || f->shared->lf->cls->write_selection != NULL);
    else
        ret_value = (f->shared->lf->cls->read_vector != NULL || f->shared->lf->cls->read_selection != NULL);

    FUNC_LEAVE_NOAPI(ret_value)
} 

uint64_t
H5F_get_rfic_flags(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(f);
    assert(f->shared);

    FUNC_LEAVE_NOAPI(f->shared->rfic_flags)
} 
