/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5MMprivate.h" 
#include "H5WBprivate.h" 

#define H5G_FHEAP_MAN_WIDTH            4
#define H5G_FHEAP_MAN_START_BLOCK_SIZE 512
#define H5G_FHEAP_MAN_MAX_DIRECT_SIZE  (64 * 1024)
#define H5G_FHEAP_MAN_MAX_INDEX        32
#define H5G_FHEAP_MAN_START_ROOT_ROWS  1
#define H5G_FHEAP_CHECKSUM_DBLOCKS     true
#define H5G_FHEAP_MAX_MAN_SIZE         (4 * 1024)

#define H5G_NAME_BT2_NODE_SIZE  512
#define H5G_NAME_BT2_MERGE_PERC 40
#define H5G_NAME_BT2_SPLIT_PERC 100

#define H5G_CORDER_BT2_NODE_SIZE  512
#define H5G_CORDER_BT2_MERGE_PERC 40
#define H5G_CORDER_BT2_SPLIT_PERC 100

#define H5G_LINK_BUF_SIZE 128

typedef struct {
    H5G_link_table_t *ltable;   
    size_t            curr_lnk; 
} H5G_dense_bt_ud_t;

typedef struct {
    
    H5F_t  *f;     
    H5HF_t *fheap; 
    hsize_t count; 

    
    hsize_t           skip;    
    H5G_lib_iterate_t op;      
    void             *op_data; 

    
    int op_ret; 
} H5G_bt2_ud_it_t;

typedef struct {
    
    H5F_t *f; 

    
    H5O_link_t *lnk; 
} H5G_fh_ud_it_t;

typedef struct {
    
    H5G_bt2_ud_common_t common;          
    bool                rem_from_fheap;  
    haddr_t             corder_bt2_addr; 
    H5RS_str_t         *grp_full_path_r; 
    bool                replace_names;   
} H5G_bt2_ud_rm_t;

typedef struct {
    
    H5F_t      *f;               
    haddr_t     corder_bt2_addr; 
    H5RS_str_t *grp_full_path_r; 
    bool        replace_names;   
} H5G_fh_ud_rm_t;

typedef struct {
    
    H5F_t      *f;               
    H5HF_t     *fheap;           
    H5_index_t  idx_type;        
    haddr_t     other_bt2_addr;  
    H5RS_str_t *grp_full_path_r; 
} H5G_bt2_ud_rmbi_t;

typedef struct {
    
    H5F_t *f; 

    
    H5O_link_t *lnk; 
} H5G_fh_ud_rmbi_t;

typedef struct {
    
    H5F_t  *f;     
    H5HF_t *fheap; 

    
    char  *name;      
    size_t name_size; 

    
    size_t name_len; 
} H5G_bt2_ud_gnbi_t;

typedef struct {
    
    H5F_t *f; 

    
    char  *name;      
    size_t name_size; 

    
    size_t name_len; 
} H5G_fh_ud_gnbi_t;

typedef struct {
    
    H5F_t  *f;     
    H5HF_t *fheap; 

    
    H5O_link_t *lnk; 
} H5G_bt2_ud_lbi_t;

typedef struct {
    
    H5F_t *f; 

    
    H5O_link_t *lnk; 
} H5G_fh_ud_lbi_t;

herr_t
H5G__dense_create(H5F_t *f, H5O_linfo_t *linfo, const H5O_pline_t *pline)
{
    H5HF_create_t fheap_cparam;        
    H5B2_create_t bt2_cparam;          
    H5HF_t       *fheap      = NULL;   
    H5B2_t       *bt2_name   = NULL;   
    H5B2_t       *bt2_corder = NULL;   
    size_t        fheap_id_len;        
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);

    
    
    memset(&fheap_cparam, 0, sizeof(fheap_cparam));
    fheap_cparam.managed.width            = H5G_FHEAP_MAN_WIDTH;
    fheap_cparam.managed.start_block_size = H5G_FHEAP_MAN_START_BLOCK_SIZE;
    fheap_cparam.managed.max_direct_size  = H5G_FHEAP_MAN_MAX_DIRECT_SIZE;
    fheap_cparam.managed.max_index        = H5G_FHEAP_MAN_MAX_INDEX;
    fheap_cparam.managed.start_root_rows  = H5G_FHEAP_MAN_START_ROOT_ROWS;
    fheap_cparam.checksum_dblocks         = H5G_FHEAP_CHECKSUM_DBLOCKS;
    fheap_cparam.max_man_size             = H5G_FHEAP_MAX_MAN_SIZE;
    if (pline)
        fheap_cparam.pline = *pline;

    
    if (NULL == (fheap = H5HF_create(f, &fheap_cparam)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create fractal heap");

    
    if (H5HF_get_heap_addr(fheap, &(linfo->fheap_addr)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get fractal heap address");

    
    if (H5HF_get_id_len(fheap, &fheap_id_len) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGETSIZE, FAIL, "can't get fractal heap ID length");
    assert(fheap_id_len == H5G_DENSE_FHEAP_ID_LEN);

    
    memset(&bt2_cparam, 0, sizeof(bt2_cparam));
    bt2_cparam.cls       = H5G_BT2_NAME;
    bt2_cparam.node_size = (size_t)H5G_NAME_BT2_NODE_SIZE;
    H5_CHECK_OVERFLOW(fheap_id_len,  hsize_t,  uint32_t);
    bt2_cparam.rrec_size = 4 +                     
                           (uint32_t)fheap_id_len; 
    bt2_cparam.split_percent = H5G_NAME_BT2_SPLIT_PERC;
    bt2_cparam.merge_percent = H5G_NAME_BT2_MERGE_PERC;
    if (NULL == (bt2_name = H5B2_create(f, &bt2_cparam, NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create v2 B-tree for name index");

    
    if (H5B2_get_addr(bt2_name, &(linfo->name_bt2_addr)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get v2 B-tree address for name index");

    
    if (linfo->index_corder) {
        
        memset(&bt2_cparam, 0, sizeof(bt2_cparam));
        bt2_cparam.cls       = H5G_BT2_CORDER;
        bt2_cparam.node_size = (size_t)H5G_CORDER_BT2_NODE_SIZE;
        H5_CHECK_OVERFLOW(fheap_id_len,  hsize_t,  uint32_t);
        bt2_cparam.rrec_size = 8 +                     
                               (uint32_t)fheap_id_len; 
        bt2_cparam.split_percent = H5G_CORDER_BT2_SPLIT_PERC;
        bt2_cparam.merge_percent = H5G_CORDER_BT2_MERGE_PERC;
        if (NULL == (bt2_corder = H5B2_create(f, &bt2_cparam, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create v2 B-tree for creation order index");

        
        if (H5B2_get_addr(bt2_corder, &(linfo->corder_bt2_addr)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get v2 B-tree address for creation order index");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_insert(H5F_t *f, const H5O_linfo_t *linfo, const H5O_link_t *lnk)
{
    H5G_bt2_ud_ins_t udata;                       
    H5HF_t          *fheap      = NULL;           
    H5B2_t          *bt2_name   = NULL;           
    H5B2_t          *bt2_corder = NULL;           
    size_t           link_size;                   
    H5WB_t          *wb = NULL;                   
    uint8_t          link_buf[H5G_LINK_BUF_SIZE]; 
    void            *link_ptr  = NULL;            
    herr_t           ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(lnk);

    
    if ((link_size = H5O_msg_raw_size(f, H5O_LINK_ID, false, lnk)) == 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGETSIZE, FAIL, "can't get link size");

    
    if (NULL == (wb = H5WB_wrap(link_buf, sizeof(link_buf))))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't wrap buffer");

    
    if (NULL == (link_ptr = H5WB_actual(wb, link_size)))
        HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, FAIL, "can't get actual buffer");

    
    if (H5O_msg_encode(f, H5O_LINK_ID, false, (unsigned char *)link_ptr, lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTENCODE, FAIL, "can't encode link");

    
    if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (H5HF_insert(fheap, link_size, link_ptr, udata.id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert link into fractal heap");

    
    if (NULL == (bt2_name = H5B2_open(f, linfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.common.f             = f;
    udata.common.fheap         = fheap;
    udata.common.name          = lnk->name;
    udata.common.name_hash     = H5_checksum_lookup3(lnk->name, strlen(lnk->name), 0);
    udata.common.corder        = lnk->corder;
    udata.common.found_op      = NULL;
    udata.common.found_op_data = NULL;
    

    
    if (H5B2_insert(bt2_name, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert record into v2 B-tree");

    
    if (linfo->index_corder) {
        
        assert(H5_addr_defined(linfo->corder_bt2_addr));
        if (NULL == (bt2_corder = H5B2_open(f, linfo->corder_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for creation order index");

        
        if (H5B2_insert(bt2_corder, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert record into v2 B-tree");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_lookup_cb(const void *_lnk, void *_user_lnk)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_lnk; 
    H5O_link_t       *user_lnk  = (H5O_link_t *)_user_lnk;  
    herr_t            ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(user_lnk);

    
    if (H5O_msg_copy(H5O_LINK_ID, lnk, user_lnk) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_lookup(H5F_t *f, const H5O_linfo_t *linfo, const char *name, bool *found, H5O_link_t *lnk)
{
    H5G_bt2_ud_common_t udata;               
    H5HF_t             *fheap     = NULL;    
    H5B2_t             *bt2_name  = NULL;    
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(name && *name);
    assert(found);
    assert(lnk);

    
    if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (NULL == (bt2_name = H5B2_open(f, linfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.name          = name;
    udata.name_hash     = H5_checksum_lookup3(name, strlen(name), 0);
    udata.found_op      = H5G__dense_lookup_cb; 
    udata.found_op_data = lnk;

    
    if (H5B2_find(bt2_name, &udata, found, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to locate link in name index");

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_lookup_by_idx_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5G_fh_ud_lbi_t *udata     = (H5G_fh_ud_lbi_t *)_udata; 
    H5O_link_t      *tmp_lnk   = NULL;                      
    herr_t           ret_value = SUCCEED;                   

    FUNC_ENTER_PACKAGE

    
    if (NULL == (tmp_lnk = (H5O_link_t *)H5O_msg_decode(udata->f, NULL, H5O_LINK_ID, obj_len,
                                                        (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTDECODE, FAIL, "can't decode link");

    
    if (NULL == H5O_msg_copy(H5O_LINK_ID, tmp_lnk, udata->lnk))
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");

done:
    
    if (tmp_lnk)
        H5O_msg_free(H5O_LINK_ID, tmp_lnk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_lookup_by_idx_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5G_dense_bt2_name_rec_t *record    = (const H5G_dense_bt2_name_rec_t *)_record;
    H5G_bt2_ud_lbi_t               *bt2_udata = (H5G_bt2_ud_lbi_t *)_bt2_udata; 
    H5G_fh_ud_lbi_t                 fh_udata;                 
    int                             ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    
    fh_udata.f   = bt2_udata->f;
    fh_udata.lnk = bt2_udata->lnk;

    
    if (H5HF_op(bt2_udata->fheap, record->id, H5G__dense_lookup_by_idx_fh_cb, &fh_udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPERATE, H5_ITER_ERROR, "link found callback failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_lookup_by_idx(H5F_t *f, const H5O_linfo_t *linfo, H5_index_t idx_type, H5_iter_order_t order,
                         hsize_t n, H5O_link_t *lnk)
{
    H5HF_t          *fheap  = NULL;       
    H5G_link_table_t ltable = {0, NULL};  
    H5B2_t          *bt2    = NULL;       
    haddr_t          bt2_addr;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(lnk);

    
    if (idx_type == H5_INDEX_NAME) {
        
        bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = linfo->corder_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE && !H5_addr_defined(bt2_addr)) {
        bt2_addr = linfo->name_bt2_addr;
        assert(H5_addr_defined(bt2_addr));
    } 

    
    if (H5_addr_defined(bt2_addr)) {
        H5G_bt2_ud_lbi_t udata; 

        
        if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f     = f;
        udata.fheap = fheap;
        udata.lnk   = lnk;

        
        if (H5B2_index(bt2, order, n, H5G__dense_lookup_by_idx_bt2_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to locate link in index");
    }      
    else { 
        
        if (H5G__dense_build_table(f, linfo, idx_type, order, &ltable) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "error building table of links");

        
        if (n >= ltable.nlinks)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

        
        if (NULL == H5O_msg_copy(H5O_LINK_ID, &ltable.lnks[n], lnk))
            HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_build_table_cb(const H5O_link_t *lnk, void *_udata)
{
    H5G_dense_bt_ud_t *udata     = (H5G_dense_bt_ud_t *)_udata; 
    herr_t             ret_value = H5_ITER_CONT;                

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);
    assert(udata->curr_lnk < udata->ltable->nlinks);

    
    if (H5O_msg_copy(H5O_LINK_ID, lnk, &(udata->ltable->lnks[udata->curr_lnk])) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");

    
    udata->curr_lnk++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_build_table(H5F_t *f, const H5O_linfo_t *linfo, H5_index_t idx_type, H5_iter_order_t order,
                       H5G_link_table_t *ltable)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(ltable);

    
    H5_CHECK_OVERFLOW(linfo->nlinks,  hsize_t,  size_t);
    ltable->nlinks = (size_t)linfo->nlinks;

    
    if (ltable->nlinks > 0) {
        H5G_dense_bt_ud_t udata; 

        
        if ((ltable->lnks = (H5O_link_t *)H5MM_calloc(sizeof(H5O_link_t) * ltable->nlinks)) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        for (size_t i = 0; i < ltable->nlinks; i++)
            ltable->lnks[i].type = H5L_TYPE_ERROR;

        
        udata.ltable   = ltable;
        udata.curr_lnk = 0;

        
        if (H5G__dense_iterate(f, linfo, H5_INDEX_NAME, H5_ITER_NATIVE, (hsize_t)0, NULL,
                               H5G__dense_build_table_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "error iterating over links");

        
        if (H5G__link_sort_table(ltable, idx_type, order) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTSORT, FAIL, "error sorting link messages");
    } 
    else
        ltable->lnks = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_iterate_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5G_fh_ud_it_t *udata     = (H5G_fh_ud_it_t *)_udata; 
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    
    if (NULL == (udata->lnk = (H5O_link_t *)H5O_msg_decode(udata->f, NULL, H5O_LINK_ID, obj_len,
                                                           (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTDECODE, FAIL, "can't decode link");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_iterate_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5G_dense_bt2_name_rec_t *record    = (const H5G_dense_bt2_name_rec_t *)_record;
    H5G_bt2_ud_it_t                *bt2_udata = (H5G_bt2_ud_it_t *)_bt2_udata; 
    herr_t                          ret_value = H5_ITER_CONT;                  

    FUNC_ENTER_PACKAGE

    
    if (bt2_udata->skip > 0)
        --bt2_udata->skip;
    else {
        H5G_fh_ud_it_t fh_udata; 

        
        
        fh_udata.f = bt2_udata->f;

        
        if (H5HF_op(bt2_udata->fheap, record->id, H5G__dense_iterate_fh_cb, &fh_udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPERATE, H5_ITER_ERROR, "heap op callback failed");

        
        ret_value = (bt2_udata->op)(fh_udata.lnk, bt2_udata->op_data);

        
        H5O_msg_free(H5O_LINK_ID, fh_udata.lnk);
    } 

    
    
    bt2_udata->count++;

    
    if (ret_value < 0)
        HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_iterate(H5F_t *f, const H5O_linfo_t *linfo, H5_index_t idx_type, H5_iter_order_t order,
                   hsize_t skip, hsize_t *last_lnk, H5G_lib_iterate_t op, void *op_data)
{
    H5HF_t          *fheap  = NULL;      
    H5G_link_table_t ltable = {0, NULL}; 
    H5B2_t          *bt2    = NULL;      
    haddr_t          bt2_addr;           
    herr_t           ret_value = FAIL;   

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(op);

    
    if (idx_type == H5_INDEX_NAME) {
        
        bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = linfo->corder_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE && !H5_addr_defined(bt2_addr)) {
        assert(H5_addr_defined(linfo->name_bt2_addr));
        bt2_addr = linfo->name_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE) {
        H5G_bt2_ud_it_t udata; 

        
        assert(H5_addr_defined(bt2_addr));

        
        if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f       = f;
        udata.fheap   = fheap;
        udata.skip    = skip;
        udata.count   = 0;
        udata.op      = op;
        udata.op_data = op_data;

        
        
        if ((ret_value = H5B2_iterate(bt2, H5G__dense_iterate_bt2_cb, &udata)) < 0)
            HERROR(H5E_SYM, H5E_BADITER, "link iteration failed");

        
        if (last_lnk)
            *last_lnk = udata.count;
    } 
    else {
        
        if (H5G__dense_build_table(f, linfo, idx_type, order, &ltable) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "error building table of links");

        
        if ((ret_value = H5G__link_iterate_table(&ltable, skip, last_lnk, op, op_data)) < 0)
            HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_get_name_by_idx_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5G_fh_ud_gnbi_t *udata = (H5G_fh_ud_gnbi_t *)_udata; 
    H5O_link_t       *lnk;                                
    herr_t            ret_value = SUCCEED;                

    FUNC_ENTER_PACKAGE

    
    if (NULL == (lnk = (H5O_link_t *)H5O_msg_decode(udata->f, NULL, H5O_LINK_ID, obj_len,
                                                    (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTDECODE, FAIL, "can't decode link");

    
    udata->name_len = strlen(lnk->name);

    
    if (udata->name) {
        strncpy(udata->name, lnk->name, MIN((udata->name_len + 1), udata->name_size));
        if (udata->name_len >= udata->name_size)
            udata->name[udata->name_size - 1] = '\0';
    } 

    
    H5O_msg_free(H5O_LINK_ID, lnk);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_get_name_by_idx_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5G_dense_bt2_name_rec_t *record    = (const H5G_dense_bt2_name_rec_t *)_record;
    H5G_bt2_ud_gnbi_t              *bt2_udata = (H5G_bt2_ud_gnbi_t *)_bt2_udata; 
    H5G_fh_ud_gnbi_t                fh_udata;            
    herr_t                          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    
    fh_udata.f         = bt2_udata->f;
    fh_udata.name      = bt2_udata->name;
    fh_udata.name_size = bt2_udata->name_size;

    
    if (H5HF_op(bt2_udata->fheap, record->id, H5G__dense_get_name_by_idx_fh_cb, &fh_udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPERATE, FAIL, "link found callback failed");

    
    bt2_udata->name_len = fh_udata.name_len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_get_name_by_idx(H5F_t *f, H5O_linfo_t *linfo, H5_index_t idx_type, H5_iter_order_t order,
                           hsize_t n, char *name, size_t name_size, size_t *name_len)
{
    H5HF_t          *fheap  = NULL;       
    H5G_link_table_t ltable = {0, NULL};  
    H5B2_t          *bt2    = NULL;       
    haddr_t          bt2_addr;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);

    
    if (idx_type == H5_INDEX_NAME) {
        
        bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = linfo->corder_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE && !H5_addr_defined(bt2_addr)) {
        bt2_addr = linfo->name_bt2_addr;
        assert(H5_addr_defined(bt2_addr));
    } 

    
    if (H5_addr_defined(bt2_addr)) {
        H5G_bt2_ud_gnbi_t udata; 

        
        if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f         = f;
        udata.fheap     = fheap;
        udata.name      = name;
        udata.name_size = name_size;

        
        if (H5B2_index(bt2, order, n, H5G__dense_get_name_by_idx_bt2_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTLIST, FAIL, "can't locate object in v2 B-tree");

        
        *name_len = udata.name_len;
    }      
    else { 
        
        if (H5G__dense_build_table(f, linfo, idx_type, order, &ltable) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "error building table of links");

        
        if (n >= ltable.nlinks)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

        
        *name_len = strlen(ltable.lnks[n].name);

        
        if (name) {
            strncpy(name, ltable.lnks[n].name, MIN((*name_len + 1), name_size));
            if (*name_len >= name_size)
                name[name_size - 1] = '\0';
        } 
    }     

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_remove_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5G_fh_ud_rm_t *udata     = (H5G_fh_ud_rm_t *)_udata; 
    H5O_link_t     *lnk       = NULL;                     
    H5B2_t         *bt2       = NULL;                     
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    if (NULL == (lnk = (H5O_link_t *)H5O_msg_decode(udata->f, NULL, H5O_LINK_ID, obj_len,
                                                    (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTDECODE, FAIL, "can't decode link");

    
    if (H5_addr_defined(udata->corder_bt2_addr)) {
        H5G_bt2_ud_common_t bt2_udata; 

        
        if (NULL == (bt2 = H5B2_open(udata->f, udata->corder_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for creation order index");

        
        assert(lnk->corder_valid);
        bt2_udata.corder = lnk->corder;

        
        if (H5B2_remove(bt2, &bt2_udata, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL,
                        "unable to remove link from creation order index v2 B-tree");
    } 

    
    if (udata->replace_names)
        if (H5G__link_name_replace(udata->f, udata->grp_full_path_r, lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTRENAME, FAIL, "unable to rename open objects");

    
    
    if (H5O_link_delete(udata->f, NULL, lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete link");

done:
    
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");
    if (lnk)
        H5O_msg_free(H5O_LINK_ID, lnk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_remove_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5G_dense_bt2_name_rec_t *record    = (const H5G_dense_bt2_name_rec_t *)_record;
    H5G_bt2_ud_rm_t                *bt2_udata = (H5G_bt2_ud_rm_t *)_bt2_udata; 
    H5G_fh_ud_rm_t                  fh_udata;            
    herr_t                          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    fh_udata.f               = bt2_udata->common.f;
    fh_udata.corder_bt2_addr = bt2_udata->corder_bt2_addr;
    fh_udata.grp_full_path_r = bt2_udata->grp_full_path_r;
    fh_udata.replace_names   = bt2_udata->replace_names;

    
    if (H5HF_op(bt2_udata->common.fheap, record->id, H5G__dense_remove_fh_cb, &fh_udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPERATE, FAIL, "link removal callback failed");

    
    if (bt2_udata->rem_from_fheap)
        if (H5HF_remove(bt2_udata->common.fheap, record->id) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove link from fractal heap");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_remove(H5F_t *f, const H5O_linfo_t *linfo, H5RS_str_t *grp_full_path_r, const char *name)
{
    H5HF_t         *fheap = NULL;        
    H5G_bt2_ud_rm_t udata;               
    H5B2_t         *bt2       = NULL;    
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);
    assert(name && *name);

    
    if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (NULL == (bt2 = H5B2_open(f, linfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.common.f             = f;
    udata.common.fheap         = fheap;
    udata.common.name          = name;
    udata.common.name_hash     = H5_checksum_lookup3(name, strlen(name), 0);
    udata.common.found_op      = NULL;
    udata.common.found_op_data = NULL;
    udata.rem_from_fheap       = true;
    udata.corder_bt2_addr      = linfo->corder_bt2_addr;
    udata.grp_full_path_r      = grp_full_path_r;
    udata.replace_names        = true;

    
    if (H5B2_remove(bt2, &udata, H5G__dense_remove_bt2_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove link from name index v2 B-tree");

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_remove_by_idx_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5G_fh_ud_rmbi_t *udata     = (H5G_fh_ud_rmbi_t *)_udata; 
    herr_t            ret_value = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    if (NULL == (udata->lnk = (H5O_link_t *)H5O_msg_decode(udata->f, NULL, H5O_LINK_ID, obj_len,
                                                           (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTDECODE, H5_ITER_ERROR, "can't decode link");

    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__dense_remove_by_idx_bt2_cb(const void *_record, void *_bt2_udata)
{
    H5G_bt2_ud_rmbi_t *bt2_udata = (H5G_bt2_ud_rmbi_t *)_bt2_udata; 
    H5G_fh_ud_rmbi_t   fh_udata;            
    H5B2_t            *bt2 = NULL;          
    const uint8_t     *heap_id;             
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (bt2_udata->idx_type == H5_INDEX_NAME) {
        const H5G_dense_bt2_name_rec_t *record = (const H5G_dense_bt2_name_rec_t *)_record;

        
        heap_id = record->id;
    } 
    else {
        const H5G_dense_bt2_corder_rec_t *record = (const H5G_dense_bt2_corder_rec_t *)_record;

        assert(bt2_udata->idx_type == H5_INDEX_CRT_ORDER);

        
        heap_id = record->id;
    } 

    
    fh_udata.f   = bt2_udata->f;
    fh_udata.lnk = NULL;

    
    if (H5HF_op(bt2_udata->fheap, heap_id, H5G__dense_remove_by_idx_fh_cb, &fh_udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPERATE, FAIL, "link removal callback failed");
    assert(fh_udata.lnk);

    
    if (H5_addr_defined(bt2_udata->other_bt2_addr)) {
        H5G_bt2_ud_common_t other_bt2_udata; 

        
        if (bt2_udata->idx_type == H5_INDEX_NAME) {
            
            other_bt2_udata.corder = fh_udata.lnk->corder;
        } 
        else {
            assert(bt2_udata->idx_type == H5_INDEX_CRT_ORDER);

            
            other_bt2_udata.f     = bt2_udata->f;
            other_bt2_udata.fheap = bt2_udata->fheap;
            other_bt2_udata.name  = fh_udata.lnk->name;
            other_bt2_udata.name_hash =
                H5_checksum_lookup3(fh_udata.lnk->name, strlen(fh_udata.lnk->name), 0);
            other_bt2_udata.found_op      = NULL;
            other_bt2_udata.found_op_data = NULL;
        } 

        
        if (NULL == (bt2 = H5B2_open(bt2_udata->f, bt2_udata->other_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for 'other' index");

        

        
        if (H5B2_remove(bt2, &other_bt2_udata, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, H5_ITER_ERROR,
                        "unable to remove link from 'other' index v2 B-tree");
    } 

    
    if (H5G__link_name_replace(bt2_udata->f, bt2_udata->grp_full_path_r, fh_udata.lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTRENAME, FAIL, "unable to rename open objects");

    
    
    if (H5O_link_delete(bt2_udata->f, NULL, fh_udata.lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete link");

    
    H5O_msg_free(H5O_LINK_ID, fh_udata.lnk);

    
    if (H5HF_remove(bt2_udata->fheap, heap_id) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove link from fractal heap");

done:
    
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for 'other' index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_remove_by_idx(H5F_t *f, const H5O_linfo_t *linfo, H5RS_str_t *grp_full_path_r, H5_index_t idx_type,
                         H5_iter_order_t order, hsize_t n)
{
    H5HF_t          *fheap  = NULL;       
    H5G_link_table_t ltable = {0, NULL};  
    H5B2_t          *bt2    = NULL;       
    haddr_t          bt2_addr;            
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);

    
    if (idx_type == H5_INDEX_NAME) {
        
        bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = linfo->corder_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE && !H5_addr_defined(bt2_addr)) {
        bt2_addr = linfo->name_bt2_addr;
        assert(H5_addr_defined(bt2_addr));
    } 

    
    if (H5_addr_defined(bt2_addr)) {
        H5G_bt2_ud_rmbi_t udata; 

        
        if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f               = f;
        udata.fheap           = fheap;
        udata.idx_type        = idx_type;
        udata.other_bt2_addr  = idx_type == H5_INDEX_NAME ? linfo->corder_bt2_addr : linfo->name_bt2_addr;
        udata.grp_full_path_r = grp_full_path_r;

        
        if (H5B2_remove_by_idx(bt2, order, n, H5G__dense_remove_by_idx_bt2_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove link from indexed v2 B-tree");
    }      
    else { 
        
        if (H5G__dense_build_table(f, linfo, idx_type, order, &ltable) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "error building table of links");

        
        if (n >= ltable.nlinks)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

        
        if (H5G__dense_remove(f, linfo, grp_full_path_r, ltable.lnks[n].name) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove link from dense storage");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__dense_delete(H5F_t *f, H5O_linfo_t *linfo, bool adj_link)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(linfo);

    
    
    if (adj_link) {
        H5HF_t         *fheap = NULL; 
        H5G_bt2_ud_rm_t udata;        

        
        if (NULL == (fheap = H5HF_open(f, linfo->fheap_addr)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        udata.common.f             = f;
        udata.common.fheap         = fheap;
        udata.common.name          = NULL;
        udata.common.name_hash     = 0;
        udata.common.found_op      = NULL;
        udata.common.found_op_data = NULL;
        udata.rem_from_fheap       = false; 
        udata.corder_bt2_addr      = linfo->corder_bt2_addr;
        udata.grp_full_path_r      = NULL;
        udata.replace_names        = false;

        
        if (H5B2_delete(f, linfo->name_bt2_addr, NULL, H5G__dense_remove_bt2_cb, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree for name index");

        
        if (H5HF_close(fheap) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    } 
    else {
        
        if (H5B2_delete(f, linfo->name_bt2_addr, NULL, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree for name index");
    } 
    linfo->name_bt2_addr = HADDR_UNDEF;

    
    if (linfo->index_corder) {
        
        assert(H5_addr_defined(linfo->corder_bt2_addr));
        if (H5B2_delete(f, linfo->corder_bt2_addr, NULL, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree for creation order index");
        linfo->corder_bt2_addr = HADDR_UNDEF;
    } 
    else
        assert(!H5_addr_defined(linfo->corder_bt2_addr));

    
    if (H5HF_delete(f, linfo->fheap_addr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete fractal heap");
    linfo->fheap_addr = HADDR_UNDEF;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
