/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5Opkg.h"      

#include "H5VLnative_private.h" 

static hid_t  H5O__open_api_common(hid_t loc_id, const char *name, hid_t lapl_id, void **token_ptr,
                                   H5VL_object_t **_vol_obj_ptr);
static hid_t  H5O__open_by_idx_api_common(hid_t loc_id, const char *group_name, H5_index_t idx_type,
                                          H5_iter_order_t order, hsize_t n, hid_t lapl_id, void **token_ptr,
                                          H5VL_object_t **_vol_obj_ptr);
static herr_t H5O__get_info_by_name_api_common(hid_t loc_id, const char *name, H5O_info2_t *oinfo ,
                                               unsigned fields, hid_t lapl_id, void **token_ptr,
                                               H5VL_object_t **_vol_obj_ptr);
static herr_t H5O__copy_api_common(hid_t src_loc_id, const char *src_name, hid_t dst_loc_id,
                                   const char *dst_name, hid_t ocpypl_id, hid_t lcpl_id, void **token_ptr,
                                   H5VL_object_t **_vol_obj_ptr);
static herr_t H5O__flush_api_common(hid_t obj_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr);
static herr_t H5O__refresh_api_common(hid_t oid, void **token_ptr, H5VL_object_t **_vol_obj_ptr);
static htri_t H5O__close_check_type(hid_t object_id);

static hid_t
H5O__open_api_common(hid_t loc_id, const char *name, hid_t lapl_id, void **token_ptr,
                     H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5I_type_t        opened_type;
    void             *opened_obj = NULL;
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID;

    FUNC_ENTER_PACKAGE

    

    
    
    if (H5VL_setup_name_args(loc_id, name, false, lapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");

    
    if (NULL == (opened_obj = H5VL_object_open(*vol_obj_ptr, &loc_params, &opened_type,
                                               H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open object");

    
    if ((ret_value = H5VL_register(opened_type, opened_obj, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to atomize object handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Oopen(hid_t loc_id, const char *name, hid_t lapl_id)
{
    hid_t ret_value = H5I_INVALID_HID;

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value = H5O__open_api_common(loc_id, name, lapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to synchronously open object");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Oopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id, const char *name,
              hid_t lapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5O__open_api_common(loc_id, name, lapl_id, token_ptr, &vol_obj)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to asynchronously open object");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE7(__func__, "*s*sIui*sii", app_file, app_func, app_line, loc_id, name, lapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on object ID");
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        }

done:
    FUNC_LEAVE_API(ret_value)
} 

static hid_t
H5O__open_by_idx_api_common(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                            hsize_t n, hid_t lapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5I_type_t        opened_type;
    void             *opened_obj = NULL;
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID;

    FUNC_ENTER_PACKAGE

    
    
    
    if (H5VL_setup_idx_args(loc_id, group_name, idx_type, order, n, false, lapl_id, vol_obj_ptr,
                            &loc_params) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_CANTSET, H5I_INVALID_HID, "can't set object access arguments");

    
    if (NULL == (opened_obj = H5VL_object_open(*vol_obj_ptr, &loc_params, &opened_type,
                                               H5P_DATASET_XFER_DEFAULT, token_ptr)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open object");

    
    if ((ret_value = H5VL_register(opened_type, opened_obj, H5VL_OBJ_CONNECTOR(*vol_obj_ptr), true)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5Oopen_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
               hid_t lapl_id)
{
    hid_t ret_value = H5I_INVALID_HID;

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if ((ret_value =
             H5O__open_by_idx_api_common(loc_id, group_name, idx_type, order, n, lapl_id, NULL, NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to synchronously open object");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Oopen_by_idx_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                     const char *group_name, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
                     hid_t lapl_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if ((ret_value = H5O__open_by_idx_api_common(loc_id, group_name, idx_type, order, n, lapl_id, token_ptr,
                                                 &vol_obj)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to asynchronously open object");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE10(__func__, "*s*sIui*sIiIohii", app_file, app_func, app_line, loc_id, group_name, idx_type, order, n, lapl_id, es_id)) < 0) {
            
            if (H5I_dec_app_ref_always_close(ret_value) < 0)
                HDONE_ERROR(H5E_OHDR, H5E_CANTDEC, H5I_INVALID_HID, "can't decrement count on object ID");
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, H5I_INVALID_HID, "can't insert token into event set");
        }

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Oopen_by_token(hid_t loc_id, H5O_token_t token)
{
    H5VL_object_t    *vol_obj;                     
    H5I_type_t        vol_obj_type = H5I_BADID;    
    H5I_type_t        opened_type;                 
    void             *opened_obj = NULL;           
    H5VL_loc_params_t loc_params;                  
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5O_IS_TOKEN_UNDEF(token))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "can't open H5O_TOKEN_UNDEF");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    loc_params.type                        = H5VL_OBJECT_BY_TOKEN;
    loc_params.loc_data.loc_by_token.token = &token;
    loc_params.obj_type                    = vol_obj_type;

    
    if (NULL == (opened_obj = H5VL_object_open(vol_obj, &loc_params, &opened_type, H5P_DATASET_XFER_DEFAULT,
                                               H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open object");

    
    if ((ret_value = H5VL_register(opened_type, opened_obj, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register object handle");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5O__copy_api_common(hid_t src_loc_id, const char *src_name, hid_t dst_loc_id, const char *dst_name,
                     hid_t ocpypl_id, hid_t lcpl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_loc_params_t loc_params2;

    
    H5VL_object_t    *vol_obj1 = NULL; 
    H5VL_loc_params_t loc_params1;

    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (!src_name || !*src_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no source name specified");
    if (!dst_name || !*dst_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no destination name specified");

    
    if (H5P_DEFAULT == lcpl_id)
        lcpl_id = H5P_LINK_CREATE_DEFAULT;
    else if (true != H5P_isa_class(lcpl_id, H5P_LINK_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not link creation property list");

    
    if (H5P_DEFAULT == ocpypl_id)
        ocpypl_id = H5P_OBJECT_COPY_DEFAULT;
    else if (true != H5P_isa_class(ocpypl_id, H5P_OBJECT_COPY))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not object copy property list");

    
    H5CX_set_lcpl(lcpl_id);

    
    if (H5VL_setup_loc_args(src_loc_id, &vol_obj1, &loc_params1) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set object access arguments");

    
    if (NULL == (*vol_obj_ptr = H5VL_vol_object(dst_loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    loc_params2.type     = H5VL_OBJECT_BY_SELF;
    loc_params2.obj_type = H5I_get_type(dst_loc_id);

    
    if (H5VL_object_copy(vol_obj1, &loc_params1, src_name, *vol_obj_ptr, &loc_params2, dst_name, ocpypl_id,
                         lcpl_id, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Ocopy(hid_t src_loc_id, const char *src_name, hid_t dst_loc_id, const char *dst_name, hid_t ocpypl_id,
        hid_t lcpl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5O__copy_api_common(src_loc_id, src_name, dst_loc_id, dst_name, ocpypl_id, lcpl_id, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to synchronously copy object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ocopy_async(const char *app_file, const char *app_func, unsigned app_line, hid_t src_loc_id,
              const char *src_name, hid_t dst_loc_id, const char *dst_name, hid_t ocpypl_id, hid_t lcpl_id,
              hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5O__copy_api_common(src_loc_id, src_name, dst_loc_id, dst_name, ocpypl_id, lcpl_id, token_ptr,
                             &vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to asynchronously copy object");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE10(__func__, "*s*sIui*si*siii", app_file, app_func, app_line, src_loc_id, src_name, dst_loc_id, dst_name, ocpypl_id, lcpl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5O__flush_api_common(hid_t obj_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_object_specific_args_t vol_cb_args;          
    H5VL_loc_params_t           loc_params;           
    herr_t                      ret_value = SUCCEED;  

    FUNC_ENTER_PACKAGE

    
    if (H5VL_setup_loc_args(obj_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set object access arguments");

    
    vol_cb_args.op_type           = H5VL_OBJECT_FLUSH;
    vol_cb_args.args.flush.obj_id = obj_id;

    
    if (H5VL_object_specific(*vol_obj_ptr, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Oflush(hid_t obj_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5O__flush_api_common(obj_id, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to synchronously flush object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oflush_async(const char *app_file, const char *app_func, unsigned app_line, hid_t obj_id, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5O__flush_api_common(obj_id, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to asynchronously flush object");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, obj_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5O__refresh_api_common(hid_t oid, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_object_specific_args_t vol_cb_args;          
    H5VL_loc_params_t           loc_params;           
    herr_t                      ret_value = SUCCEED;  

    FUNC_ENTER_PACKAGE

    
    if (H5VL_setup_loc_args(oid, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set object access arguments");

    
    vol_cb_args.op_type             = H5VL_OBJECT_REFRESH;
    vol_cb_args.args.refresh.obj_id = oid;

    
    if (H5VL_object_specific(*vol_obj_ptr, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to refresh object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Orefresh(hid_t oid)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5O__refresh_api_common(oid, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to synchronously refresh object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Orefresh_async(const char *app_file, const char *app_func, unsigned app_line, hid_t oid, hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5O__refresh_api_common(oid, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to asynchronously refresh object");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, oid, es_id)) < 0)
            
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Olink(hid_t obj_id, hid_t new_loc_id, const char *new_name, hid_t lcpl_id, hid_t lapl_id)
{
    H5VL_object_t          *vol_obj1 = NULL; 
    H5VL_object_t          *vol_obj2 = NULL; 
    H5VL_link_create_args_t vol_cb_args;     
    H5VL_loc_params_t       new_loc_params;
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (new_loc_id == H5L_SAME_LOC)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL,
                    "cannot use H5L_SAME_LOC when only one location is specified");
    if (!new_name || !*new_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");

#if H5_SIZEOF_SIZE_T > H5_SIZEOF_INT32_T
    if (strlen(new_name) > H5L_MAX_LINK_NAME_LEN)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "name too long");
#endif 
    if (lcpl_id != H5P_DEFAULT && (true != H5P_isa_class(lcpl_id, H5P_LINK_CREATE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a link creation property list");

    
    if (H5P_DEFAULT == lcpl_id)
        lcpl_id = H5P_LINK_CREATE_DEFAULT;

    
    H5CX_set_lcpl(lcpl_id);

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, obj_id, true) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    new_loc_params.type                         = H5VL_OBJECT_BY_NAME;
    new_loc_params.obj_type                     = H5I_get_type(new_loc_id);
    new_loc_params.loc_data.loc_by_name.name    = new_name;
    new_loc_params.loc_data.loc_by_name.lapl_id = lapl_id;

    
    if (NULL == (vol_obj1 = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    if (H5L_SAME_LOC != new_loc_id)
        
        if (NULL == (vol_obj2 = H5VL_vol_object(new_loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (vol_obj1 && vol_obj2) {
        htri_t same_connector;

        
        if ((same_connector =
                 H5VL_conn_same_class(H5VL_OBJ_CONNECTOR(vol_obj1), H5VL_OBJ_CONNECTOR(vol_obj2))) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");
        if (!same_connector)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL,
                        "Objects are accessed through different VOL connectors and can't be linked");
    } 

    
    vol_cb_args.op_type                            = H5VL_LINK_CREATE_HARD;
    vol_cb_args.args.hard.curr_obj                 = H5VL_OBJ_DATA(vol_obj1);
    vol_cb_args.args.hard.curr_loc_params.type     = H5VL_OBJECT_BY_SELF;
    vol_cb_args.args.hard.curr_loc_params.obj_type = H5I_get_type(obj_id);

    
    if (H5VL_link_create(&vol_cb_args, vol_obj2, &new_loc_params, lcpl_id, lapl_id, H5P_DATASET_XFER_DEFAULT,
                         H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCREATE, FAIL, "unable to create link");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oincr_refcount(hid_t object_id)
{
    H5VL_object_t              *vol_obj;     
    H5VL_object_specific_args_t vol_cb_args; 
    H5VL_loc_params_t           loc_params;  
    herr_t                      ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(object_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5CX_set_loc(object_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    vol_cb_args.op_type              = H5VL_OBJECT_CHANGE_REF_COUNT;
    vol_cb_args.args.change_rc.delta = 1;

    
    if (H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "modifying object link count failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Odecr_refcount(hid_t object_id)
{
    H5VL_object_t              *vol_obj;             
    H5VL_object_specific_args_t vol_cb_args;         
    H5VL_loc_params_t           loc_params;          
    herr_t                      ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(object_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5CX_set_loc(object_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    vol_cb_args.op_type              = H5VL_OBJECT_CHANGE_REF_COUNT;
    vol_cb_args.args.change_rc.delta = -1;

    
    if (H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "modifying object link count failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5Oexists_by_name(hid_t loc_id, const char *name, hid_t lapl_id)
{
    H5VL_object_t              *vol_obj;            
    H5VL_object_specific_args_t vol_cb_args;        
    H5VL_loc_params_t           loc_params;         
    bool                        obj_exists = false; 
    htri_t                      ret_value  = FAIL;  

    FUNC_ENTER_API(FAIL)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    vol_cb_args.op_type            = H5VL_OBJECT_EXISTS;
    vol_cb_args.args.exists.exists = &obj_exists;

    
    if (H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to determine if '%s' exists", name);

    
    ret_value = (htri_t)obj_exists;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info3(hid_t loc_id, H5O_info2_t *oinfo , unsigned fields)
{
    H5VL_object_t         *vol_obj;     
    H5VL_object_get_args_t vol_cb_args; 
    H5VL_loc_params_t      loc_params;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
    vol_cb_args.args.get_info.oinfo  = oinfo;
    vol_cb_args.args.get_info.fields = fields;

    
    if (H5VL_object_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get data model info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5O__get_info_by_name_api_common(hid_t loc_id, const char *name, H5O_info2_t *oinfo , unsigned fields,
                                 hid_t lapl_id, void **token_ptr, H5VL_object_t **_vol_obj_ptr)
{
    H5VL_object_t  *tmp_vol_obj = NULL; 
    H5VL_object_t **vol_obj_ptr =
        (_vol_obj_ptr ? _vol_obj_ptr : &tmp_vol_obj); 
    H5VL_object_get_args_t vol_cb_args;               
    H5VL_loc_params_t      loc_params;                
    herr_t                 ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    
    if (H5VL_setup_name_args(loc_id, name, false, lapl_id, vol_obj_ptr, &loc_params) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set object access arguments");

    
    vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
    vol_cb_args.args.get_info.oinfo  = oinfo;
    vol_cb_args.args.get_info.fields = fields;

    
    if (H5VL_object_get(*vol_obj_ptr, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, token_ptr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get data model info for object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Oget_info_by_name3(hid_t loc_id, const char *name, H5O_info2_t *oinfo , unsigned fields,
                     hid_t lapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5O__get_info_by_name_api_common(loc_id, name, oinfo, fields, lapl_id, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't synchronously retrieve object info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_name_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                          const char *name, H5O_info2_t *oinfo , unsigned fields, hid_t lapl_id,
                          hid_t es_id)
{
    H5VL_object_t *vol_obj   = NULL;            
    void          *token     = NULL;            
    void         **token_ptr = H5_REQUEST_NULL; 
    herr_t         ret_value = SUCCEED;         

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id)
        token_ptr = &token; 

    
    if (H5O__get_info_by_name_api_common(loc_id, name, oinfo, fields, lapl_id, token_ptr, &vol_obj) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't asynchronously retrieve object info");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE9(__func__, "*s*sIui*s*!Iuii", app_file, app_func, app_line, loc_id, name, oinfo, fields, lapl_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_info_by_idx3(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                    hsize_t n, H5O_info2_t *oinfo , unsigned fields, hid_t lapl_id)
{
    H5VL_object_t         *vol_obj;     
    H5VL_object_get_args_t vol_cb_args; 
    H5VL_loc_params_t      loc_params;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!group_name || !*group_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = group_name;
    loc_params.loc_data.loc_by_idx.idx_type = idx_type;
    loc_params.loc_data.loc_by_idx.order    = order;
    loc_params.loc_data.loc_by_idx.n        = n;
    loc_params.loc_data.loc_by_idx.lapl_id  = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    vol_cb_args.op_type              = H5VL_OBJECT_GET_INFO;
    vol_cb_args.args.get_info.oinfo  = oinfo;
    vol_cb_args.args.get_info.fields = fields;

    
    if (H5VL_object_get(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get data model info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_native_info(hid_t loc_id, H5O_native_info_t *oinfo , unsigned fields)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_NATIVE_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    obj_opt_args.get_native_info.fields = fields;
    obj_opt_args.get_native_info.ninfo  = oinfo;
    vol_cb_args.op_type                 = H5VL_NATIVE_OBJECT_GET_NATIVE_INFO;
    vol_cb_args.args                    = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get native file format info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_native_info_by_name(hid_t loc_id, const char *name, H5O_native_info_t *oinfo , unsigned fields,
                           hid_t lapl_id)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "oinfo parameter cannot be NULL");
    if (fields & ~H5O_NATIVE_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    obj_opt_args.get_native_info.fields = fields;
    obj_opt_args.get_native_info.ninfo  = oinfo;
    vol_cb_args.op_type                 = H5VL_NATIVE_OBJECT_GET_NATIVE_INFO;
    vol_cb_args.args                    = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get native file format info for object: '%s'", name);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oget_native_info_by_idx(hid_t loc_id, const char *group_name, H5_index_t idx_type, H5_iter_order_t order,
                          hsize_t n, H5O_native_info_t *oinfo , unsigned fields, hid_t lapl_id)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!group_name || !*group_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name specified");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!oinfo)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no info struct");
    if (fields & ~H5O_NATIVE_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_IDX;
    loc_params.loc_data.loc_by_idx.name     = group_name;
    loc_params.loc_data.loc_by_idx.idx_type = idx_type;
    loc_params.loc_data.loc_by_idx.order    = order;
    loc_params.loc_data.loc_by_idx.n        = n;
    loc_params.loc_data.loc_by_idx.lapl_id  = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    obj_opt_args.get_native_info.fields = fields;
    obj_opt_args.get_native_info.ninfo  = oinfo;
    vol_cb_args.op_type                 = H5VL_NATIVE_OBJECT_GET_NATIVE_INFO;
    vol_cb_args.args                    = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get native file format info for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oset_comment(hid_t obj_id, const char *comment)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    if (H5CX_set_loc(obj_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(obj_id);

    
    obj_opt_args.set_comment.comment = comment;
    vol_cb_args.op_type              = H5VL_NATIVE_OBJECT_SET_COMMENT;
    vol_cb_args.args                 = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set comment for object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oset_comment_by_name(hid_t loc_id, const char *name, const char *comment, hid_t lapl_id)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no name");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, true) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    obj_opt_args.set_comment.comment = comment;
    vol_cb_args.op_type              = H5VL_NATIVE_OBJECT_SET_COMMENT;
    vol_cb_args.args                 = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set comment for object: '%s'", name);

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Oget_comment(hid_t obj_id, char *comment , size_t bufsize)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    size_t                             comment_len = 0;  
    ssize_t                            ret_value   = -1; 

    FUNC_ENTER_API((-1))

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid location identifier");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(obj_id);

    
    obj_opt_args.get_comment.buf         = comment;
    obj_opt_args.get_comment.buf_size    = bufsize;
    obj_opt_args.get_comment.comment_len = &comment_len;
    vol_cb_args.op_type                  = H5VL_NATIVE_OBJECT_GET_COMMENT;
    vol_cb_args.args                     = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, (-1), "can't get comment for object");

    
    ret_value = (ssize_t)comment_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Oget_comment_by_name(hid_t loc_id, const char *name, char *comment , size_t bufsize, hid_t lapl_id)
{
    H5VL_object_t                     *vol_obj;      
    H5VL_optional_args_t               vol_cb_args;  
    H5VL_native_object_optional_args_t obj_opt_args; 
    H5VL_loc_params_t                  loc_params;
    size_t                             comment_len = 0;  
    ssize_t                            ret_value   = -1; 

    FUNC_ENTER_API((-1))

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, (-1), "no name");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, (-1), "can't set access property list info");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "invalid location identifier");

    
    obj_opt_args.get_comment.buf         = comment;
    obj_opt_args.get_comment.buf_size    = bufsize;
    obj_opt_args.get_comment.comment_len = &comment_len;
    vol_cb_args.op_type                  = H5VL_NATIVE_OBJECT_GET_COMMENT;
    vol_cb_args.args                     = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, (-1), "can't get comment for object: '%s'", name);

    
    ret_value = (ssize_t)comment_len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit3(hid_t obj_id, H5_index_t idx_type, H5_iter_order_t order, H5O_iterate2_t op, void *op_data,
          unsigned fields)
{
    H5VL_object_t              *vol_obj;     
    H5VL_object_specific_args_t vol_cb_args; 
    H5VL_loc_params_t           loc_params;  
    herr_t                      ret_value;   

    FUNC_ENTER_API(FAIL)

    
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(obj_id);

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = op;
    vol_cb_args.args.visit.op_data  = op_data;
    vol_cb_args.args.visit.fields   = fields;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object iteration failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ovisit_by_name3(hid_t loc_id, const char *obj_name, H5_index_t idx_type, H5_iter_order_t order,
                  H5O_iterate2_t op, void *op_data, unsigned fields, hid_t lapl_id)
{
    H5VL_object_t              *vol_obj;     
    H5VL_object_specific_args_t vol_cb_args; 
    H5VL_loc_params_t           loc_params;  
    herr_t                      ret_value;   

    FUNC_ENTER_API(FAIL)

    
    if (!obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be NULL");
    if (!*obj_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "obj_name parameter cannot be an empty string");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");
    if (!op)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no callback operator specified");
    if (fields & ~H5O_INFO_ALL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid fields");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params.type                         = H5VL_OBJECT_BY_NAME;
    loc_params.loc_data.loc_by_name.name    = obj_name;
    loc_params.loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params.obj_type                     = H5I_get_type(loc_id);

    
    vol_cb_args.op_type             = H5VL_OBJECT_VISIT;
    vol_cb_args.args.visit.idx_type = idx_type;
    vol_cb_args.args.visit.order    = order;
    vol_cb_args.args.visit.op       = op;
    vol_cb_args.args.visit.op_data  = op_data;
    vol_cb_args.args.visit.fields   = fields;

    
    if ((ret_value = H5VL_object_specific(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT,
                                          H5_REQUEST_NULL)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object iteration failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

static htri_t
H5O__close_check_type(hid_t object_id)
{
    htri_t ret_value = true; 

    FUNC_ENTER_PACKAGE

    
    switch (H5I_get_type(object_id)) {
        case H5I_GROUP:
        case H5I_DATATYPE:
        case H5I_DATASET:
        case H5I_MAP:
            if (H5I_object(object_id) == NULL)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a valid object");
            break;

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_DONE(false);
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Oclose(hid_t object_id)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (H5O__close_check_type(object_id) <= 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "not a valid object");

    if (H5I_dec_app_ref(object_id) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "unable to close object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Oclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t object_id, hid_t es_id)
{
    H5VL_object_t    *vol_obj   = NULL;            
    H5VL_connector_t *connector = NULL;            
    void             *token     = NULL;            
    void            **token_ptr = H5_REQUEST_NULL; 
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    
    if (H5O__close_check_type(object_id) <= 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "not a valid object");

    
    if (H5ES_NONE != es_id) {
        
        if (NULL == (vol_obj = H5VL_vol_object(object_id)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get VOL object for object");

        
        connector = H5VL_OBJ_CONNECTOR(vol_obj);
        H5VL_conn_inc_rc(connector);

        
        token_ptr = &token;
    } 

    
    if (H5I_dec_app_ref_async(object_id, token_ptr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCLOSEFILE, FAIL, "decrementing object ID failed");

    
    if (NULL != token)
        
        if (H5ES_insert(es_id, H5VL_OBJ_CONNECTOR(vol_obj), token,
                        H5ARG_TRACE5(__func__, "*s*sIuii", app_file, app_func, app_line, object_id, es_id)) < 0)
            
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "can't insert token into event set");

done:
    if (connector && H5VL_conn_dec_rc(connector) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTDEC, FAIL, "can't decrement ref count on connector");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5O__disable_mdc_flushes(H5O_loc_t *oloc)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    if (H5AC_cork(oloc->file, oloc->addr, H5AC__SET_CORK, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCORK, FAIL, "unable to cork object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Odisable_mdc_flushes(hid_t object_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    H5VL_loc_params_t    loc_params;          
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_is_file_object(object_id) != true)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "ID is not a file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object ID");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(object_id);

    
    vol_cb_args.op_type = H5VL_NATIVE_OBJECT_DISABLE_MDC_FLUSHES;
    vol_cb_args.args    = NULL;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCORK, FAIL, "unable to cork object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5O__enable_mdc_flushes(H5O_loc_t *oloc)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    if (H5AC_cork(oloc->file, oloc->addr, H5AC__UNCORK, NULL) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNCORK, FAIL, "unable to uncork object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Oenable_mdc_flushes(hid_t object_id)
{
    H5VL_object_t       *vol_obj;             
    H5VL_optional_args_t vol_cb_args;         
    H5VL_loc_params_t    loc_params;          
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_is_file_object(object_id) != true)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "ID is not a file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object ID");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(object_id);

    
    vol_cb_args.op_type = H5VL_NATIVE_OBJECT_ENABLE_MDC_FLUSHES;
    vol_cb_args.args    = NULL;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNCORK, FAIL, "unable to uncork object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5O__are_mdc_flushes_disabled(const H5O_loc_t *oloc, bool *are_disabled)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(are_disabled);

    if (H5AC_cork(oloc->file, oloc->addr, H5AC__GET_CORKED, are_disabled) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to retrieve object's cork status");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Oare_mdc_flushes_disabled(hid_t object_id, bool *are_disabled)
{
    H5VL_object_t                     *vol_obj;             
    H5VL_optional_args_t               vol_cb_args;         
    H5VL_native_object_optional_args_t obj_opt_args;        
    H5VL_loc_params_t                  loc_params;          
    herr_t                             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (!are_disabled)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location from ID");

    
    if (H5I_is_file_object(object_id) != true)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "ID is not a file object");

    
    if (NULL == (vol_obj = H5VL_vol_object(object_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object ID");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(object_id);

    
    obj_opt_args.are_mdc_flushes_disabled.flag = are_disabled;
    vol_cb_args.op_type                        = H5VL_NATIVE_OBJECT_ARE_MDC_FLUSHES_DISABLED;
    vol_cb_args.args                           = &obj_opt_args;

    
    if (H5VL_object_optional(vol_obj, &loc_params, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) <
        0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to retrieve object's cork status");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Otoken_cmp(hid_t loc_id, const H5O_token_t *token1, const H5O_token_t *token2, int *cmp_value)
{
    H5VL_object_t *vol_obj;             
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    if (NULL == cmp_value)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid cmp_value pointer");

    
    if (H5VL_token_cmp(vol_obj, token1, token2, cmp_value) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOMPARE, FAIL, "object token comparison failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Otoken_to_str(hid_t loc_id, const H5O_token_t *token, char **token_str)
{
    H5VL_object_t *vol_obj;             
    H5I_type_t     vol_obj_type;        
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    if (NULL == token)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid token pointer");
    if (NULL == token_str)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid token string pointer");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get underlying VOL object type");

    
    if (H5VL_token_to_str(vol_obj, vol_obj_type, token, token_str) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTSERIALIZE, FAIL, "object token serialization failed");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Otoken_from_str(hid_t loc_id, const char *token_str, H5O_token_t *token)
{
    H5VL_object_t *vol_obj;             
    H5I_type_t     vol_obj_type;        
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");
    if (NULL == token)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid token pointer");
    if (NULL == token_str)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid token string pointer");

    
    if ((vol_obj_type = H5I_get_type(loc_id)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get underlying VOL object type");

    
    if (H5VL_token_from_str(vol_obj, vol_obj_type, token_str, token) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, FAIL, "object token deserialization failed");

done:
    FUNC_LEAVE_API(ret_value)
} 
