/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Opkg.h"      
#include "H5MMprivate.h" 

static void  *H5O__drvinfo_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                  size_t p_size, const uint8_t *p);
static herr_t H5O__drvinfo_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                  const void *_mesg);
static void  *H5O__drvinfo_copy(const void *_mesg, void *_dest);
static size_t H5O__drvinfo_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__drvinfo_reset(void *_mesg);
static herr_t H5O__drvinfo_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_DRVINFO[1] = {{
    H5O_DRVINFO_ID,        
    "driver info",         
    sizeof(H5O_drvinfo_t), 
    0,                     
    H5O__drvinfo_decode,   
    H5O__drvinfo_encode,   
    H5O__drvinfo_copy,     
    H5O__drvinfo_size,     
    H5O__drvinfo_reset,    
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    H5O__drvinfo_debug     
}};

#define H5O_DRVINFO_VERSION 0

static void *
H5O__drvinfo_decode(H5F_t H5_ATTR_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                    unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                    const uint8_t *p)
{
    H5O_drvinfo_t *mesg      = NULL;           
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_DRVINFO_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (mesg = (H5O_drvinfo_t *)H5MM_calloc(sizeof(H5O_drvinfo_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for driver info message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 8, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5MM_memcpy(mesg->name, p, 8);
    mesg->name[8] = '\0';
    p += 8;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->len);
    if (0 == mesg->len)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "message length can't be zero");

    
    if (NULL == (mesg->buf = (uint8_t *)H5MM_malloc(mesg->len)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for driver info buffer");

    
    if (H5_IS_BUFFER_OVERFLOW(p, mesg->len, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5MM_memcpy(mesg->buf, p, mesg->len);

    
    ret_value = (void *)mesg;

done:
    if (!ret_value && mesg) {
        H5MM_xfree(mesg->buf);
        H5MM_xfree(mesg);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__drvinfo_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size,
                    uint8_t *p, const void *_mesg)
{
    const H5O_drvinfo_t *mesg = (const H5O_drvinfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg);

    
    *p++ = H5O_DRVINFO_VERSION;
    H5MM_memcpy(p, mesg->name, 8);
    p += 8;
    assert(mesg->len <= 65535);
    UINT16ENCODE(p, mesg->len);
    H5MM_memcpy(p, mesg->buf, mesg->len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__drvinfo_copy(const void *_mesg, void *_dest)
{
    const H5O_drvinfo_t *mesg      = (const H5O_drvinfo_t *)_mesg;
    H5O_drvinfo_t       *dest      = (H5O_drvinfo_t *)_dest;
    void                *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    if (!dest && NULL == (dest = (H5O_drvinfo_t *)H5MM_malloc(sizeof(H5O_drvinfo_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for shared message table message");

    
    *dest = *mesg;

    
    if (NULL == (dest->buf = (uint8_t *)H5MM_malloc(mesg->len))) {
        if (dest != _dest)
            dest = (H5O_drvinfo_t *)H5MM_xfree(dest);
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    } 
    H5MM_memcpy(dest->buf, mesg->buf, mesg->len);

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__drvinfo_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_drvinfo_t *mesg      = (const H5O_drvinfo_t *)_mesg;
    size_t               ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    ret_value = 1 +        
                8 +        
                2 +        
                mesg->len; 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__drvinfo_reset(void *_mesg)
{
    H5O_drvinfo_t *mesg = (H5O_drvinfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);

    
    mesg->buf = (uint8_t *)H5MM_xfree(mesg->buf);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__drvinfo_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_drvinfo_t *mesg = (const H5O_drvinfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Driver name:", mesg->name);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Buffer size:", (unsigned long long)mesg->len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
