/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

hid_t
H5Tcomplex_create(hid_t base_type_id)
{
    H5T_t *base      = NULL;            
    H5T_t *dt        = NULL;            
    hid_t  ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    if (NULL == (base = (H5T_t *)H5I_object_verify(base_type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid base datatype ID");

    if (NULL == (dt = H5T__complex_create(base)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5I_INVALID_HID,
                    "can't create complex number datatype from base datatype");

    if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register datatype");

done:
    FUNC_LEAVE_API(ret_value);
}

H5T_t *
H5T__complex_create(const H5T_t *base)
{
    H5T_t *dt        = NULL; 
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(base);

    
    if (base->shared->type != H5T_FLOAT)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, NULL, "base datatype is not a H5T_FLOAT datatype");
    if (base->shared->size == 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, NULL, "invalid base datatype size");
    if (base->shared->size > SIZE_MAX / 2)
        HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, NULL,
                    "base datatype size too large - new datatype size would overflow");

    
    if (NULL == (dt = H5T__alloc()))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, NULL, "memory allocation failed");
    dt->shared->type = H5T_COMPLEX;
    dt->shared->size = 2 * base->shared->size;

    if (NULL == (dt->shared->parent = H5T_copy(base, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, NULL, "can't copy base datatype");

    
    dt->shared->u.cplx.form = H5T_COMPLEX_RECTANGULAR; 

    
    dt->shared->version = MAX(base->shared->version, H5O_DTYPE_VERSION_5);

    ret_value = dt;

done:
    if (!ret_value)
        if (dt && H5T_close(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "can't close datatype");

    FUNC_LEAVE_NOAPI(ret_value)
}
