% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_analytic_css.R
\name{calc_analytic_css}
\alias{calc_analytic_css}
\title{Calculate the analytic steady state plasma concentration.}
\usage{
calc_analytic_css(
  chem.name = NULL,
  chem.cas = NULL,
  dtxsid = NULL,
  parameters = NULL,
  species = "human",
  daily.dose = NULL,
  dose = 1,
  dose.units = "mg/kg/day",
  route = "oral",
  output.units = "uM",
  model = "pbtk",
  concentration = "plasma",
  suppress.messages = FALSE,
  tissue = NULL,
  bioactive.free.invivo = FALSE,
  IVIVE = NULL,
  parameterize.args.list = list(),
  ...
)
}
\arguments{
\item{chem.name}{Either the chemical name, CAS number, or the parameters must 
be specified.}

\item{chem.cas}{Either the chemical name, CAS number, or the parameters must 
be specified.}

\item{dtxsid}{EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})
the chemical must be identified by either CAS, name, or DTXSIDs}

\item{parameters}{Chemical parameters from parameterize_pbtk (for model = 
'pbtk'), parameterize_3comp (for model = '3compartment), 
parameterize_1comp(for model = '1compartment') or parameterize_steadystate 
(for model = '3compartmentss'), overrides chem.name and chem.cas.}

\item{species}{Species desired (either "Rat", "Rabbit", "Dog", "Mouse", or
default "Human").}

\item{daily.dose}{Total daily dose, mg/kg BW.}

\item{dose}{The amount of chemial to which the individual is exposed.}

\item{dose.units}{The units associated with the dose received.}

\item{route}{Route of exposure (either "oral", "iv", or "inhalation"
default "oral").}

\item{output.units}{Units for returned concentrations, defaults to uM 
(specify units = "uM") but can also be mg/L.}

\item{model}{Model used in calculation,'gas_pbtk' for the gas pbtk model, 
'pbtk' for the multiple compartment model,
'3compartment' for the three compartment model, '3compartmentss' for 
the three compartment steady state model, and '1compartment' for one 
compartment model.}

\item{concentration}{Desired concentration type: 'blood','tissue', or default 
'plasma'. In the case that the concentration is for plasma, selecting "blood"
will use the blood:plasma ratio to estimate blood concentration. In the case
that the argument 'tissue' specifies a particular tissue of the body, 
concentration defaults to 'tissue' -- that is, the concentration in the 
If cocentration is set to 'blood' or 'plasma' and 'tissue' specifies a
specific tissue then the value returned is for the plasma or blood in that
specific tissue.}

\item{suppress.messages}{Whether or not the output message is suppressed.}

\item{tissue}{Desired steady state tissue concentration. Default is of NULL
typically gives whole body plasma concentration.}

\item{bioactive.free.invivo}{If FALSE (default), then the total concentration is treated
as bioactive in vivo. If TRUE, the the unbound (free) plasma concentration is treated as 
bioactive in vivo. Only works with tissue = NULL in current implementation.}

\item{IVIVE}{Honda et al. (2019) identified four plausible sets of 
assumptions for \emph{in vitro-in vivo} extrapolation (IVIVE) assumptions. 
Argument may be set to "Honda1" through "Honda4". If used, this function 
overwrites the tissue, restrictive.clearance, and bioactive.free.invivo arguments. 
See Details below for more information.}

\item{parameterize.args.list}{List of arguments passed to model's associated
parameterization function, including default.to.human, 
adjusted.Funbound.plasma, regression, and minimum.Funbound.plasma. The 
default.to.human argument substitutes missing animal values with human values
if true, adjusted.Funbound.plasma returns adjusted Funbound.plasma when set 
to TRUE along with parition coefficients calculated with this value, 
regression indicates whether or not to use the regressions in calculating
partition coefficients, and minimum.Funbound.plasma is the value to which
Monte Carlo draws less than this value are set (default is 0.0001 -- half
the lowest measured Fup in our dataset).}

\item{...}{Additional parameters passed to parameterize function if 
parameters is NULL.}

\item{parameterize.args}{Additional parameters passed to parameterize function if 
parameters is NULL.}
}
\value{
Steady state plasma concentration in specified units
}
\description{
This function calculates the analytic steady state plasma or venous blood 
concentrations as a result of infusion dosing for the three compartment and 
multiple compartment PBTK models.
}
\details{
Concentrations are calculated for the specifed model with constant 
oral infusion dosing.  All tissues other than gut, liver, and lung are the 
product of the steady state plasma concentration and the tissue to plasma 
partition coefficient. 

Only four sets of IVIVE assumptions that performed well in Honda et al. 
(2019) are currently included in \code{\link{honda.ivive}}:
"Honda1" through "Honda4". The use of max (peak) 
concentration can not be currently be calculated with \code{\link{calc_analytic_css}}. 
The httk default settings correspond to "Honda3":

\tabular{lrrrrr}{
\tab \emph{In Vivo} Conc. \tab Metabolic Clearance \tab Bioactive Chemical Conc. \emph{In Vivo} \tab TK Statistic Used* \tab Bioactive Chemical Conc. \emph{In Vitro} \cr
Honda1 \tab Veinous (Plasma) \tab Restrictive \tab Free \tab Mean Conc. In Vivo \tab Free Conc. In Vitro \cr
Honda2 \tab Veinous \tab Restrictive \tab Free \tab Mean Conc. In Vivo \tab Nominal Conc. In Vitro \cr
Honda3 \tab Veinous \tab Restrictive \tab Total \tab Mean Conc. In Vivo \tab Nominal Conc. In Vitro \cr
Honda4 \tab Target Tissue \tab Non-restrictive \tab Total \tab Mean Conc. In Vivo \tab Nominal Conc. In Vitro \cr
}

"Honda1" uses plasma concentration, restrictive clearance, and treats the 
unbound invivo concentration as bioactive. For IVIVE, any input nominal 
concentration in vitro should be converted to cfree.invitro using 
\code{\link{armitage_eval}}, otherwise performance will be the same as 
"Honda2".
}
\examples{
calc_analytic_css(chem.name='Bisphenol-A',output.units='mg/L',
                 model='3compartment',concentration='blood')

\donttest{
# Test that the underlying PK models give the same answers:
calc_analytic_css(chem.cas="15972-60-8")
calc_analytic_css(chem.cas="15972-60-8",model="1compartment")
calc_analytic_css(chem.cas="15972-60-8",model="pbtk")
calc_analytic_css(chem.cas="15972-60-8",model="3compartment")

calc_analytic_css(chem.name='Bisphenol-A',tissue='liver',species='rabbit',
                 parameterize.args.list = list(
                                default.to.human=TRUE,
                                adjusted.Funbound.plasma=TRUE,
                                regression=TRUE,
                                minimum.Funbound.plasma=1e-4),daily.dose=2)

calc_analytic_css(chem.name="bisphenol a",model="1compartment")

calc_analytic_css(chem.cas="80-05-7",model="3compartmentss")

params <- parameterize_pbtk(chem.cas="80-05-7") 

calc_analytic_css(parameters=params,model="pbtk")

# Try various chemicals with differing parameter sources/issues:
calc_analytic_css(chem.name="Betaxolol")
calc_analytic_css(chem.name="Tacrine",model="pbtk")
calc_analytic_css(chem.name="Dicofol",model="1compartment")
calc_analytic_css(chem.name="Diflubenzuron",model="3compartment")
calc_analytic_css(chem.name="Theobromine",model="3compartmentss")

# permutations on steady-state for the 1compartment model
calc_analytic_css(chem.name="bisphenol a",
                  model="1compartment")
calc_analytic_css(chem.cas="80-05-7",
                  model="1compartment")
calc_analytic_css(parameters=parameterize_1comp(chem.cas="80-05-7"),
                  model="1compartment")
calc_analytic_css(chem.cas="80-05-7",
                  model="1compartment",
                  tissue="liver")
calc_analytic_css(chem.cas="80-05-7",
                  model="1compartment",
                  tissue="brain")

# permutations on steady-state for the 3compartment model
calc_analytic_css(chem.cas="80-05-7",
                  model="3compartment")
calc_analytic_css(parameters=parameterize_3comp(chem.cas="80-05-7"),
                  model="3compartment")
calc_analytic_css(chem.name="bisphenol a",
                  model="3compartment",
                  tissue="liver")
calc_analytic_css(chem.name="bisphenol a",
                  model="3compartment",
                  tissue="brain")

# permurtations on steady-state for the pbtk model:
calc_analytic_css(chem.cas="80-05-7",
                  model="pbtk")
calc_analytic_css(parameters=parameterize_pbtk(chem.cas="80-05-7"),
                  model="pbtk")
calc_analytic_css(chem.name="bisphenol a",
                  model="pbtk",
                  tissue="liver")
calc_analytic_css(chem.name="bisphenol a",
                  model="pbtk",
                  tissue="brain")

# Test oral absorption functionality:
# By default we now include calculation of Fabs and Fgut (always had Fhep):
calc_analytic_css(chem.name="bisphenol a",
                  model="pbtk")
# Therefore if we set Fabs = Fgut = 1 with keetit100=TRUE, we should get a
# higher predicted plasma steady-state concentration:
calc_analytic_css(chem.name="bisphenol a",
                  model="pbtk",
                  Caco2.options=list(keepit100=TRUE))
}

}
\references{
\insertRef{honda2019using}{httk}
}
\seealso{
\code{\link{calc_css}}
}
\author{
Robert Pearce, John Wambaugh, Greg Honda, Miyuki Breen
}
\keyword{Solve}
\keyword{steady-state}
