#' Global envelope tests for random labelling. 
#'
#' Generic function to perform global envelope test for random labelling using mark correlation functions. 
#'
#' @usage testmc(X,
#' fun,
#' nsim = 100,
#' fun_args = list(),     
#' get_args = list(),     
#' rlabel_args = list()
#' )
#'
#' @param X An object of class ppp or lpp.
#' @param fun Function that computes the desired mark correlation function.
#' @param nsim Number of permutations.
#' @param fun_args Arguments passed to the chosen mark correlation function.
#' @param get_args Arguments passed to the \code{\link[GET]{global_envelope_test}}.
#' @param rlabel_args  Arguments passed to the \code{\link[spatstat.random]{rlabel}}.
#' 
#' @details
#' 
#' Generic function to perform global envelope test for random labelling using mark correlation functions. 
#' 
#' If the point pattern has several real-valued marks, then the random labelling test will be performed for each mark individually, and the output will be a list of global envelop tests one per each mark.
#' 
#' If the mark correlation function is any of the LIMA family, a random labelling test will be performed for each individual point.
#' 
#' @examples
#' \donttest{
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  library(spatstat.explore)
#'  library(spatstat.linnet)
#'
#' #######################################
#' ## 1. Planar point patterns (ppp)
#' #######################################
#'
#' # ---------------------------------------------------------
#' # GLOBAL MARK CORRELATION (real-valued marks)
#' # ---------------------------------------------------------
#' X <- rpoispp(200)
#' marks(X) <- data.frame(m1 = runif(npoints(X), 1, 10))
#'
#' ts1 <- testmc(X,
#'               fun = mcorr.ppp,
#'               fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts1)
#' # Global mark correlation (real-valued marks)
#'
#' ts2 <- testmc(X,
#'               fun = mcorrinhom.ppp,
#'               fun_args = list(ftype = "stoyan",
#'                                method = "density",
#'                                method_lambda = "kernel"))
#' plot(ts2)
#' # Inhomogeneous global mark correlation (real-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # GLOBAL FUNCTIONAL MARK CORRELATION (function-valued marks)
#' # ---------------------------------------------------------
#' marks(X) <- data.frame(
#'   t1 = runif(npoints(X), 1, 10),
#'   t2 = runif(npoints(X), 1, 10),
#'   t3 = runif(npoints(X), 1, 10),
#'   t4 = runif(npoints(X), 1, 10),
#'   t5 = runif(npoints(X), 1, 10)
#' )
#'
#' ts3 <- testmc(X,
#'               fun = fmcorr,
#'               fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts3)
#' # Global functional mark correlation (function-valued marks)
#'
#' ts4 <- testmc(X,
#'               fun = fmcorrinhom,
#'               fun_args = list(ftype = "stoyan",
#'                                method = "density",
#'                                method_lambda = "kernel",
#'                                bw = bw.scott))
#' plot(ts4)
#' # Inhomogeneous functional mark correlation (function-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # LOCAL MARK CORRELATION (real-valued marks)
#' # ---------------------------------------------------------
#' X <- rpoispp(200)
#' marks(X) <- data.frame(m1 = runif(npoints(X), 1, 10))
#'
#' ts5 <- testmc(X,
#'               fun = lmcorr.ppp,
#'               fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts5[[1]])
#' # Local mark correlation (real-valued marks)
#'
#' ts6 <- testmc(X,
#'               fun = lmcorrinhom.ppp,
#'               fun_args = list(ftype = "stoyan",
#'                                method = "density",
#'                                method_lambda = "kernel"))
#' plot(ts6[[1]])
#' # Inhomogeneous local mark correlation (real-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # LOCAL FUNCTIONAL MARK CORRELATION (function-valued marks)
#' # ---------------------------------------------------------
#' marks(X) <- data.frame(
#'   t1 = runif(npoints(X), 1, 10),
#'   t2 = runif(npoints(X), 1, 10),
#'   t3 = runif(npoints(X), 1, 10),
#'   t4 = runif(npoints(X), 1, 10),
#'   t5 = runif(npoints(X), 1, 10)
#' )
#'
#' ts7 <- testmc(X,
#'               fun = lfmcorr,
#'               fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts7[[1]])
#' # Local functional mark correlation (function-valued marks)
#'
#' ts8 <- testmc(X,
#'               fun = lfmcorrinhom,
#'               fun_args = list(ftype = "stoyan",
#'                                method = "density",
#'                                method_lambda = "kernel",
#'                                bw = bw.scott))
#' plot(ts8[[1]])
#' # Inhomogeneous local functional mark correlation (function-valued marks)
#'
#'
#' #######################################
#' ## 2. Linear network point patterns (lpp)
#' #######################################
#'
#' Xl <- rpoislpp(40, simplenet)
#'
#' # ---------------------------------------------------------
#' # GLOBAL MARK CORRELATION (real-valued marks)
#' # ---------------------------------------------------------
#' marks(Xl) <- data.frame(m1 = runif(npoints(Xl), 1, 10))
#'
#' ts9 <- testmc(Xl,
#'               fun = mcorr.lpp,
#'               fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts9)
#' # Global mark correlation on linear networks (real-valued marks)
#'
#' ts10 <- testmc(Xl,
#'                fun = mcorrinhom.lpp,
#'                fun_args = list(ftype = "stoyan",
#'                                 method = "density",
#'                                 method_lambda = "kernel"))
#' plot(ts10)
#' # Inhomogeneous global mark correlation on linear networks (real-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # GLOBAL FUNCTIONAL MARK CORRELATION (function-valued marks)
#' # ---------------------------------------------------------
#' marks(Xl) <- data.frame(
#'   t1 = runif(npoints(Xl), 1, 10),
#'   t2 = runif(npoints(Xl), 1, 10),
#'   t3 = runif(npoints(Xl), 1, 10),
#'   t4 = runif(npoints(Xl), 1, 10),
#'   t5 = runif(npoints(Xl), 1, 10)
#' )
#'
#' ts11 <- testmc(Xl,
#'                fun = fmcorr,
#'                fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts11)
#' # Global functional mark correlation on linear networks (function-valued marks)
#'
#' ts12 <- testmc(Xl,
#'                fun = fmcorrinhom,
#'                fun_args = list(ftype = "stoyan",
#'                                 method = "density",
#'                                 method_lambda = "kernel",
#'                                 bw = bw.scott.iso))
#' plot(ts12)
#' # Inhomogeneous global functional mark correlation on linear networks
#' # (function-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # LOCAL MARK CORRELATION (real-valued marks)
#' # ---------------------------------------------------------
#' marks(Xl) <- data.frame(m1 = runif(npoints(Xl), 1, 10))
#'
#' ts13 <- testmc(Xl,
#'                fun = lmcorr.lpp,
#'                fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts13[[1]])
#' # Local mark correlation on linear networks (real-valued marks)
#'
#' ts14 <- testmc(Xl,
#'                fun = lmcorrinhom.lpp,
#'                fun_args = list(ftype = "stoyan",
#'                                 method = "density",
#'                                 method_lambda = "kernel"))
#' plot(ts14[[1]])
#' # Inhomogeneous local mark correlation on linear networks (real-valued marks)
#'
#'
#' # ---------------------------------------------------------
#' # LOCAL FUNCTIONAL MARK CORRELATION (function-valued marks)
#' # ---------------------------------------------------------
#' marks(Xl) <- data.frame(
#'   t1 = runif(npoints(Xl), 1, 10),
#'   t2 = runif(npoints(Xl), 1, 10),
#'   t3 = runif(npoints(Xl), 1, 10),
#'   t4 = runif(npoints(Xl), 1, 10),
#'   t5 = runif(npoints(Xl), 1, 10)
#' )
#'
#' ts15 <- testmc(Xl,
#'                fun = lfmcorr,
#'                fun_args = list(ftype = "stoyan", method = "density"))
#' plot(ts15[[1]])
#' # Local functional mark correlation on linear networks (function-valued marks)
#'
#' ts16 <- testmc(Xl,
#'                fun = lfmcorrinhom,
#'                fun_args = list(ftype = "stoyan",
#'                                 method = "density",
#'                                 method_lambda = "kernel",
#'                                 bw = bw.scott.iso))
#' plot(ts16[[1]])
#' # Inhomogeneous local functional mark correlation on linear networks
#' # (function-valued marks)
#'
#' }
#'
#' @references
#' Eckardt, M., & Moradi, M. (2024). Marked spatial point processes: current state and extensions to point processes on linear networks. Journal of Agricultural, Biological and Environmental Statistics, 29(2), 346-378.
#'
#' Moradi, M., & Eckardt, M. (2025). Inhomogeneous mark correlation functions for general marked point processes.
#'
#' Eckardt, M., & Moradi, M. (2025). Local indicators of mark association for marked spatial point processes.
#'
#' @return Either an object of class global_envelope or a list of such objects if the point pattern has multiple marks or the employed mark correlation function is of type LIMA.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com}
#' @seealso \code{\link[GET]{global_envelope_test}}, \code{\link[markstat]{mcorr.ppp}}, \code{\link[markstat]{mcorr.lpp}}, \code{\link[markstat]{mcorrinhom.ppp}}, \code{\link[markstat]{mcorrinhom.lpp}},  \code{\link[markstat]{fmcorr}}, \code{\link[markstat]{lmcorr.ppp}}, \code{\link[markstat]{lmcorr.lpp}}.
#' 
#' 
#' @import spatstat.random
#' @import GET
#' @import stats
#' @export
testmc <- function(
    X,
    fun,
    nsim = 100,
    fun_args = list(),     # named list of extra args passed to `fun`
    get_args = list(),     # named list of extra args passed to GET::global_envelope_test
    rlabel_args = list()   # named list of extra args for rlabel (besides permute=TRUE, nsim)
) {
  
  if(identical(fun, lmcorr.ppp) || identical(fun, lmcorr.lpp) || identical(fun, lmcorrinhom.ppp) || identical(fun, lmcorrinhom.lpp) || 
     identical(fun, lfmcorr) || identical(fun, lfmcorrinhom)
     ){
    
    # --- generate permuted simulations ---
    rlabel_call_args <- c(list(X = X, permute = TRUE, nsim = nsim), rlabel_args)
    # Ensure correct argument names for do.call: rlabel expects first argument as X (unnamed)
    # So pass as do.call(rlabel, list(X, permute=TRUE, nsim=nsim, ...))
    sims <- do.call(rlabel, rlabel_call_args)
    # `rlabel` often returns a list of point patterns; make sure sims is a list
    if (!is.list(sims)) {
      stop("`rlabel` did not return a list of simulated patterns.")
    }
    
    m <- marks(X)
    if(!is.numeric(m) && !(identical(fun, lfmcorr) || identical(fun, lfmcorrinhom))){
      out <- list()
      for (i in 1:dim(m)[2]) {
        marks(X) <- m[,i]
        out[[i]] <- testmc(X,
                           fun = fun,
                           nsim = nsim,
                           fun_args = fun_args,     
                           get_args = get_args,     
                           rlabel_args = rlabel_args)
      }
      names(out) <- names(m)
      class(out) <- c(class(out), "testmc")
      attr(out, "fun") <- deparse(substitute(fun))
      attr(out, "fun_args") <- fun_args
      attr(out,"get_args") <- get_args
      return(out)
    }
    
    # --- run the observed data through fun ---
    fun_obs <- do.call(fun, c(list(X), fun_args))
    fun_obs <- as.data.frame(fun_obs)
    
    r_values  <- fun_obs[, "r"]

    
    # --- apply fun to each simulation ---
    fun_sims_list <- lapply(sims, function(sim) {
      sim_fun <- do.call(fun, c(list(sim), fun_args, r = r_values))
      sim_fun <- as.data.frame(sim_fun)
      return(sim_fun)
    })
    
    out <- list()
    local_sims <- list() 
    
    for (i in 1:npoints(X)) {

      fbs <- fun_obs[,i+1]
      
      for(k in 1:nsim){
        local_sims[[k]] <- fun_sims_list[[k]][,i+1]
      }
      
      lsim <- do.call(rbind, local_sims)
      csets <- create_curve_set(list(r=r_values, obs=fbs, sim_m=t(lsim)))

      # call global_envelope_test with explicit args
      get_call <- c(list(csets), get_args)
      out[[i]] <- do.call(GET::global_envelope_test, get_call)
      csets <- NULL
      
    }
    names(out) <- NULL
    class(out) <- c(class(out), "testmc")
    attr(out, "fun") <- deparse(substitute(fun))
    attr(out, "fun_args") <- fun_args
    attr(out,"get_args") <- get_args
    return(out)
  }
  
  m <- marks(X)
  if(!is.numeric(m) && !(identical(fun, fmcorr) || identical(fun, fmcorrinhom))){
    out <- list()
    for (i in 1:dim(m)[2]) {
      marks(X) <- m[,i]
      out[[i]] <- testmc(X,
                         fun = fun,
                         nsim = nsim,
                         fun_args = fun_args,     
                         get_args = get_args,     
                         rlabel_args = rlabel_args)
    }
    names(out) <- names(m)
    class(out) <- c(class(out), "testmc")
    attr(out, "fun") <- deparse(substitute(fun))
    attr(out, "fun_args") <- fun_args
    attr(out,"get_args") <- get_args
    return(out)
  }
  
  # --- run the observed data through fun ---
  fun_obs <- do.call(fun, c(list(X), fun_args))
  fun_obs <- as.data.frame(fun_obs)
  
  # check expected structure
  if (!is.data.frame(fun_obs) && !is.matrix(fun_obs)) {
    stop("`fun` must return a data.frame or matrix with columns 'r' and 'est'.")
  }
  if (!all(c("r", "est") %in% colnames(fun_obs))) {
    stop("`fun` must return columns named 'r' and 'est'.")
  }
  
  r_values  <- fun_obs[, "r"]
  obs_curve <- fun_obs[, "est"]
  
  # --- generate permuted simulations ---
  rlabel_call_args <- c(list(X = X, permute = TRUE, nsim = nsim), rlabel_args)
  # Ensure correct argument names for do.call: rlabel expects first argument as X (unnamed)
  # So pass as do.call(rlabel, list(X, permute=TRUE, nsim=nsim, ...))
  sims <- do.call(rlabel, rlabel_call_args)
  
  # `rlabel` often returns a list of point patterns; make sure sims is a list
  if (!is.list(sims)) {
    stop("`rlabel` did not return a list of simulated patterns.")
  }
  
  # --- apply fun to each simulation ---
  fun_sims_list <- lapply(sims, function(sim) {
    sim_fun <- do.call(fun, c(list(sim), fun_args, r = r_values))
    sim_fun <- as.data.frame(sim_fun)
    if (!"est" %in% colnames(sim_fun)) {
      stop("Each simulated result from `fun` must include an 'est' column.")
    }
    return(sim_fun[, "est"])
  })
  
  # convert to matrix: columns = simulations
  sim_matrix <- do.call(cbind, fun_sims_list)
  
  # optional: confirm lengths match
  if (nrow(sim_matrix) != length(r_values)) {
    stop("Length mismatch: `fun` on simulations produced curves with different length than observed.")
  }
  
  # --- create curve set and run global envelope test ---
  csets <- create_curve_set(list(r = r_values, obs = obs_curve, sim_m = sim_matrix))
  
  # call global_envelope_test with explicit args
  get_call <- c(list(csets), get_args)
  result <- do.call(GET::global_envelope_test, get_call)
  class(result) <- c(class(result), "testmc")
  attr(result, "fun") <- deparse(substitute(fun))
  attr(result, "fun_args") <- fun_args
  attr(result,"get_args") <- get_args
  return(result)
}

#' @export
print.testmc <- function(x, ...) {
  # Store original attributes
  at <- attributes(x)
  
  # Remove attributes temporarily
  attributes(x) <- NULL
  
  # Print the object normally, without attributes
  print(x, ...)
  
  # Restore attributes invisibly (so they are not lost)
  attributes(x) <- at
  invisible(x)
}

