#' Fit an Alpha-Skewed Generalized Normal (ASGN) Distribution by MCMC
#'
#' @description
#' \code{asgn_func()} fits an alpha-skewed generalized normal (ASGN) distribution
#' to univariate numeric data using an MCMC algorithm. The ASGN family provides a
#' flexible parametric model that can accommodate skewness and, for certain
#' parameter values, bimodal density shapes.
#'
#' @details
#' The input \code{data} may be any univariate numeric sample (vector or a
#' one-column matrix). In the \code{\link{mmcmcBayes}} workflow, \code{data} is
#' typically a vector of sample-wise regional mean M-values (one value per sample).
#' However, \code{asgn_func()} is not specific to DNA methylation and can be used
#' more generally for fitting skewed or potentially bimodal continuous data.
#'
#' @param data A numeric vector or a one-column matrix. In typical use this is a
#'   vector of sample-wise regional mean M-values (one per sample).
#' @param priors Optional list of prior hyperparameters. If \code{NULL}, weakly
#'   informative priors are constructed from the data. If provided, expected
#'   components are \code{alpha}, \code{mu}, and \code{sigma2}.
#' @param mcmc A list of MCMC parameters:
#'   \itemize{
#'     \item \code{nburn}: Number of burn-in iterations (default: 5000)
#'     \item \code{niter}: Number of sampling iterations (default: 10000)
#'     \item \code{thin}: Thinning interval (default: 1)
#'   }
#' @param seed Optional integer random seed for reproducibility. If \code{NULL},
#'   no seed is set.
#'
#' @return
#' A list with the following elements:
#' \itemize{
#'   \item \code{posteriors}: Vector of posterior means for \code{alpha}, \code{mu}, and \code{sigma2}.
#'   \item \code{summary}: Data frame containing parameter estimates and 95\% credible intervals
#'         with columns \code{parameter}, \code{estimate}, \code{lower_ci}, and \code{upper_ci}.
#' }
#' If there are fewer than two non-missing observations, a default value
#' \code{posteriors = c(1, 0, 1)} is returned, and \code{summary} is omitted.
#'
#' @examples
#' \donttest{
#' # Generate sample data
#' set.seed(2021)
#' dt <- rgamma(1000, shape = 2, rate = 1)
#' dt <- as.matrix(dt, ncol=1)
#'
#' result <- asgn_func(dt)
#' print(result$summary)
#' }
#'
#' @author
#' Zhexuan Yang, Duchwan Ryu, and Feng Luan
#'
#' @seealso
#' \code{\link{mmcmcBayes}} for the main DMR detection function,
#' 
#' @importFrom stats sd var rnorm runif quantile
#' @importFrom MCMCpack rinvgamma
#' 
#' @references
#' Mahmoudi, E., Jafari, H., & Meshkat, R. (2019).
#' Alpha-skew generalized normal distribution and its applications.
#' \emph{Applications and Applied Mathematics: An International Journal (AAM)},
#' \bold{14}, 784-804.
#'
#' @export
asgn_func <- function(data, 
                      priors = NULL, 
                      mcmc = list(nburn = 5000, niter = 10000, thin = 1), 
                      seed = NULL) {
  
  if (!requireNamespace("MCMCpack", quietly = TRUE)) stop("MCMCpack required.")
  if (!is.null(seed)) set.seed(seed)
  
  # Remove any NA or NaN values before they enter the loop.
  # This prevents "missing value where TRUE/FALSE needed" errors later.
  ybar <- as.vector(data)
  ybar <- ybar[!is.na(ybar) & !is.nan(ybar)]
  
  n <- length(ybar)
  
  # Safety check: If all data was NA, return NULL or handle gracefully
  if (n < 2) {
    return(list(
      posteriors = c(alpha = 1, mu = 0, sigma2 = 1),
      note = "Insufficient Data"
    ))
  }
  
  # --- 1. Initialize Priors ---
  if (is.null(priors)) {
    mu_pri    <- mean(ybar)
    sigm      <- var(ybar)  
    if(is.na(sigm) || sigm == 0) sigm <- 0.01 # Prevent division by zero
    sigma2_pri <- var(ybar)
    if(is.na(sigma2_pri) || sigma2_pri == 0) sigma2_pri <- 0.01
    
    alpha_pri <- 1
    siga      <- 1 
    
    As <- (sigma2_pri^2) / (sigma2_pri^2 / 2) + 2
    Bs <- sigma2_pri * ((sigma2_pri^2) / (sigma2_pri^2 / 2) + 1)
  } else {
    alpha_pri <- as.numeric(priors$alpha); siga <- 1
    mu_pri    <- as.numeric(priors$mu); sigm <- as.numeric(priors$sigma2)
    sigma2_pri <- as.numeric(priors$sigma2)
    
    As <- (sigma2_pri^2) / (sigma2_pri^2 / 2) + 2
    Bs <- sigma2_pri * ((sigma2_pri^2) / (sigma2_pri^2 / 2) + 1)
  }
  
  # --- 2. MCMC Setup ---
  nburn <- if (is.null(mcmc$nburn)) 5000 else mcmc$nburn
  niter <- if (is.null(mcmc$niter)) 10000 else mcmc$niter
  thin  <- if (is.null(mcmc$thin)) 1 else mcmc$thin
  total_iter <- nburn + niter * thin
  
  alpha_t <- 1
  mu_t <- mu_pri
  sig_t <- sigma2_pri
  
  alpha_samples <- numeric(niter)
  mu_samples    <- numeric(niter)
  sig_samples   <- numeric(niter)
  
  # Pre-calculate gamma constants
  curr_gamma_term <- -n * log(4 * 0.886227 * (alpha_t^2) + 4 * 1.77245)
  
  # Initial Likelihood Components
  term1_t <- sum(log((1 - alpha_t * ybar)^2 + 1))
  sq_diff_t <- sum((ybar - mu_t)^2)
  term2_t <- sq_diff_t / (2 * sig_t)
  log_lik_t <- term1_t - term2_t
  
  # --- 3. MCMC Loop ---
  save_index <- 1
  
  for (k in 1:total_iter) {
    
    # === UPDATE ALPHA ===
    alpha_c <- rnorm(1, alpha_t, sd = 1) 
    
    term1_c <- sum(log((1 - alpha_c * ybar)^2 + 1))
    cand_gamma_term <- -n * log(4 * 0.886227 * (alpha_c^2) + 4 * 1.77245)
    log_lik_c <- term1_c - term2_t
    
    log_prior_c <- - (alpha_c - alpha_pri)^2 / (2 * siga)
    log_prior_t <- - (alpha_t - alpha_pri)^2 / (2 * siga)
    
    ratio <- (log_prior_c + log_lik_c + cand_gamma_term) - 
      (log_prior_t + log_lik_t + curr_gamma_term)
    
    if (!is.na(ratio) && log(runif(1)) < ratio) {
      alpha_t <- alpha_c
      term1_t <- term1_c
      curr_gamma_term <- cand_gamma_term
      log_lik_t <- log_lik_c
    }
    
    # === UPDATE MU ===
    mu_c <- rnorm(1, mu_t, sd = 1) 
    
    sq_diff_c <- sum((ybar - mu_c)^2)
    term2_c <- sq_diff_c / (2 * sig_t)
    log_lik_c <- term1_t - term2_c
    
    log_prior_c <- - (mu_c - mu_pri)^2 / (2 * sigm)
    log_prior_t <- - (mu_t - mu_pri)^2 / (2 * sigm)
    
    ratio <- (log_prior_c + log_lik_c) - (log_prior_t + log_lik_t)
    
    if (!is.na(ratio) && log(runif(1)) < ratio) {
      mu_t <- mu_c
      sq_diff_t <- sq_diff_c
      term2_t <- term2_c
      log_lik_t <- log_lik_c
    }
    
    # === UPDATE SIGMA2 ===
    sig_c <- MCMCpack::rinvgamma(1, shape = (sig_t + 2), scale = ((sig_t + 1) * sig_t))
    if (sig_c < 1e-10) sig_c <- 1e-10
    
    term2_c <- sq_diff_t / (2 * sig_c)
    log_lik_c <- term1_t - term2_c
    
    log_prior_c <- -As * log(sig_c) - (Bs / sig_c)
    log_prior_t <- -As * log(sig_t) - (Bs / sig_t)
    
    ratio <- (log_prior_c + log_lik_c) - (log_prior_t + log_lik_t)
    
    if (!is.na(ratio) && log(runif(1)) < ratio) {
      sig_t <- sig_c
      log_lik_t <- log_lik_c
    }
    
    if (k > nburn && (k - nburn) %% thin == 0) {
      alpha_samples[save_index] <- alpha_t
      mu_samples[save_index]    <- mu_t
      sig_samples[save_index]   <- sig_t
      save_index <- save_index + 1
    }
  }
  
  # --- 4. Prepare Output ---
  result <- list(
    posteriors = c(
      alpha  = mean(alpha_samples),
      mu     = mean(mu_samples),
      sigma2 = mean(sig_samples)
    )
  )
  
  result$summary <- data.frame(
    parameter = c("alpha", "mu", "sigma2"),
    estimate  = result$posteriors,
    lower_ci  = c(
      stats::quantile(alpha_samples, 0.025),
      stats::quantile(mu_samples, 0.025),
      stats::quantile(sig_samples, 0.025)
    ),
    upper_ci  = c(
      stats::quantile(alpha_samples, 0.975),
      stats::quantile(mu_samples, 0.975),
      stats::quantile(sig_samples, 0.975)
    ),
    stringsAsFactors = FALSE
  )
  row.names(result$summary) <- NULL
  
  return(result)
}