% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compare_dmrs.R
\name{compare_dmrs}
\alias{compare_dmrs}
\title{Compare Differentially Methylated Regions (DMRs) from Two Methods}
\usage{
compare_dmrs(rst1, rst2)
}
\arguments{
\item{rst1}{A data frame containing the first set of DMR results. Must contain
at least the columns \code{Chromosome}, \code{Start_CpG}, and \code{End_CpG}.}

\item{rst2}{A data frame containing the second set of DMR results, in the same
format as \code{rst1}. The \code{Chromosome} values must be comparable to
those in \code{rst1} (e.g., both using \code{"chr6"} or both using \code{"6"}).}
}
\value{
A data frame with one row per overlapping pair of regions and the columns:
\itemize{
  \item \code{Chromosome}: chromosome name of the overlapping regions.
  \item \code{Start_CpG_Method1}: start CpG ID from \code{rst1}.
  \item \code{End_CpG_Method1}: end CpG ID from \code{rst1}.
  \item \code{Start_CpG_Method2}: start CpG ID from \code{rst2}.
  \item \code{End_CpG_Method2}: end CpG ID from \code{rst2}.
  \item \code{Overlap_Percentage}: overlap percentage relative to the larger
        of the two regions (in CpG index units), rounded to two decimals.
}

Returns \code{NULL} if no overlaps are found or if, after cleaning, one of
the inputs has no usable rows.
}
\description{
\code{compare_dmrs()} identifies overlapping regions between two sets of
differentially methylated regions (DMRs), typically obtained from two
different detection methods. It reports pairwise overlaps and a simple
overlap percentage that can be used to assess consistency between methods.
}
\details{
This function compares genomic regions between two DMR result objects. For each
region in \code{rst1}, it searches for regions in \code{rst2} on the same
chromosome that have any overlap in CpG index range (partial or complete).

CpG identifiers in \code{Start_CpG} and \code{End_CpG} are assumed to contain
an embedded numeric component that reflects their ordering along the genome
(e.g., \code{"cg00017002"}). Internally, these IDs are converted to numeric
values by stripping non-digit characters; rows for which this conversion fails
are removed before comparison.

For each overlapping pair of regions, the function computes
\itemize{
  \item the size of the overlapping segment (in CpG index units), and
  \item an overlap percentage, defined as
    \deqn{\text{overlap\_size} / \max(\text{length\_method1}, \text{length\_method2}) \times 100,}
    where \code{length_method1} and \code{length_method2} are the lengths of the
    two regions in CpG index units.
}

The overlap percentage is therefore symmetric in the two methods and can be
interpreted as “how much of the larger region is covered by the overlap.”
}
\examples{
\donttest{
# Create sample DMR results
dmr_method1 <- data.frame(
  Chromosome = c("chr1", "chr1", "chr2"),
  Start_CpG = c("cg0001", "cg0050", "cg0100"),
  End_CpG = c("cg0020", "cg0070", "cg0150")
)

dmr_method2 <- data.frame(
  Chromosome = c("chr1", "chr2", "chr2"),
  Start_CpG = c("cg0005", "cg0120", "cg0090"),
  End_CpG = c("cg0025", "cg0160", "cg0110")
)

# Compare overlapping regions
overlaps <- compare_dmrs(dmr_method1, dmr_method2)
}

}
\seealso{
Related functions in this package:
\code{\link{mmcmcBayes}} for DMR detection using multi-stage MCMC,
\code{\link{asgn_func}} for parameter estimation with ASGN distribution
}
\author{
Zhexuan Yang, Duchwan Ryu, and Feng Luan
}
