% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shapley.R
\name{shapley}
\alias{shapley}
\title{Weighted Mean SHAP (WMSHAP) and Confidence Interval for Multiple Models
       (tuning grid, stacked ensemble, etc.)}
\usage{
shapley(
  models,
  newdata,
  plot = TRUE,
  performance_metric = "r2",
  standardize_performance_metric = FALSE,
  performance_type = "xval",
  minimum_performance = 0,
  method = "mean",
  cutoff = 0.01,
  top_n_features = NULL,
  n_models = 10,
  sample_size = NULL
)
}
\arguments{
\item{models}{An H2O AutoML object, H2O grid object, \code{autoEnsemble} object,
or a character vector of H2O model IDs.}

\item{newdata}{An \code{H2OFrame} (i.e., a \code{data.frame}) already uploaded to the
\code{h2o} server. SHAP contributions are computed on this data.}

\item{plot}{Logical. If \code{TRUE}, plots the WMSHAP summary (via \code{shapley.plot()}).}

\item{performance_metric}{Character. Performance metric used to weight models.
Options are \code{"r2"} (regression), \code{"aucpr"}, \code{"auc"}, and \code{"f2"}
(classification metrics).}

\item{standardize_performance_metric}{Logical. If \code{TRUE}, rescales model weights so
the weights sum to the number of included models.
The default is \code{FALSE}.}

\item{performance_type}{Character. Specify which performance metric performance estimate to use:
 \code{"train"} for training data, \code{"valid"}
for validation, or \code{"xval"} for cross-validation (default).}

\item{minimum_performance}{Numeric. Specify the minimum performance metric
for a model to be included in calculating WMSHAP.
Models below this threshold receive
zero weight and are excluded. The default is \code{0}.
Specifying a minimum performance can be used to compute
WMSHAP for a set of competitive models.}

\item{method}{Character. Specify the method for selecting important features
based on their WMSHAP. The default is
\code{"mean"}, which selects features whose WMSHAP
exceeds the \code{cutoff}. The alternative is
\code{"lowerCI"}, which selects features whose lower bound of confidence
interval exceeds the \code{cutoff}.}

\item{cutoff}{Numeric. Cutoff applied by \code{method} for selecting important features.}

\item{top_n_features}{Integer or \code{NULL}. If not \code{NULL}, restricts SHAP computation to the
top N features per model (reduces runtime). This also selects the top N features by WMSHAP
in the returned \code{selectedFeatures}.}

\item{n_models}{Integer. Minimum number of models that must meet the performance threshold
for WMSHAP and CI computation. Use \code{1} to compute summary SHAP for a single model.
The default is 10.}

\item{sample_size}{Integer. Number of rows in \code{newdata} used for SHAP assessment.
Defaults to all rows. Reducing this can speed up development runs.}
}
\value{
An object of class \code{"shapley"} (a named list) containing:
\describe{
  \item{ids}{Character vector of model IDs originally supplied or extracted.}
  \item{included_models}{Character vector of model IDs included after filtering by performance.}
  \item{ignored_models}{Data frame of excluded models and their performance.}
  \item{weights}{Numeric vector of model weights (performance metrics) for included models.}
  \item{results}{Data frame of row-level SHAP contributions merged across models.}
  \item{summaryShaps}{Data frame of feature-level WMSHAP means and confidence intervals.}
  \item{selectedFeatures}{Character vector of selected important features.}
  \item{feature_importance}{List of per-feature absolute contribution summaries by model.}
  \item{contributionPlot}{A ggplot-like object returned by \code{h2o.shap_summary_plot()}
        used for the WMSHAP (“wmshap”) style plot.}
  \item{plot}{A ggplot object (bar plot) if \code{plot = TRUE}, otherwise \code{NULL}.}
}
}
\description{
Computes Weighted Mean SHAP ratios (WMSHAP) and confidence intervals to assess feature
importance across a collection of models (e.g., an H2O grid/AutoML leaderboard or
base-learners of an ensemble). Instead of reporting SHAP contributions for a single model,
this function summarizes feature importance across multiple models and weights each model
by a chosen performance metric.
Currently, only models trained by the \code{h2o} machine learning platform,
\code{autoEnsemble}, and the \code{HMDA} R packages are supported.
}
\details{
The function works as follows:
   \enumerate{
     \item For each model, SHAP contributions are computed on \code{newdata}.
     \item For each model, feature-level absolute SHAP contributions are aggregated and
           converted to a \emph{ratio} (share of total absolute SHAP across features).
     \item Models are weighted by a performance metric (e.g., \code{"r2"} for regression or
           \code{"auc"} / \code{"aucpr"} for classification).
     \item The weighted mean SHAP ratio (WMSHAP) is computed for each feature, along with an
           confidence interval across models.
   }
}
\examples{

\dontrun{
# load the required libraries for building the base-learners and the ensemble models
library(h2o)            #shapley supports h2o models
library(shapley)

# initiate the h2o server
h2o.init(ignore_config = TRUE, nthreads = 2, bind_to_localhost = FALSE, insecure = TRUE)

# upload data to h2o cloud
prostate_path <- system.file("extdata", "prostate.csv", package = "h2o")
prostate <- h2o.importFile(path = prostate_path, header = TRUE)

set.seed(10)

### H2O provides 2 types of grid search for tuning the models, which are
### AutoML and Grid. Below, I demonstrate how weighted mean shapley values
### can be computed for both types.

#######################################################
### PREPARE AutoML Grid (takes a couple of minutes)
#######################################################
# run AutoML to tune various models (GBM) for 60 seconds
y <- "CAPSULE"
prostate[,y] <- as.factor(prostate[,y])  #convert to factor for classification
aml <- h2o.automl(y = y, training_frame = prostate, max_runtime_secs = 120,
                 include_algos=c("GBM"),

                 # this setting ensures the models are comparable for building a meta learner
                 seed = 2023, nfolds = 10,
                 keep_cross_validation_predictions = TRUE)

### call 'shapley' function to compute the weighted mean and weighted confidence intervals
### of SHAP values across all trained models.
### Note that the 'newdata' should be the testing dataset!
result <- shapley(models = aml, newdata = prostate, performance_metric = "aucpr", plot = TRUE)

#######################################################
### PREPARE H2O Grid (takes a couple of minutes)
#######################################################
# make sure equal number of "nfolds" is specified for different grids
grid <- h2o.grid(algorithm = "gbm", y = y, training_frame = prostate,
                 hyper_params = list(ntrees = seq(1,50,1)),
                 grid_id = "ensemble_grid",

                 # this setting ensures the models are comparable for building a meta learner
                 seed = 2023, fold_assignment = "Modulo", nfolds = 10,
                 keep_cross_validation_predictions = TRUE)

result2 <- shapley(models = grid, newdata = prostate, performance_metric = "aucpr", plot = TRUE)

#######################################################
### PREPARE autoEnsemble STACKED ENSEMBLE MODEL
#######################################################

### get the models' IDs from the AutoML and grid searches.
### this is all that is needed before building the ensemble,
### i.e., to specify the model IDs that should be evaluated.
library(autoEnsemble)
ids    <- c(h2o.get_ids(aml), h2o.get_ids(grid))
autoSearch <- ensemble(models = ids, training_frame = prostate, strategy = "search")
result3 <- shapley(models = autoSearch, newdata = prostate,
                   performance_metric = "aucpr", plot = TRUE)


}
}
\author{
E. F. Haghish
}
