% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pca_multithreshold.R
\name{pca_multithreshold}
\alias{pca_multithreshold}
\title{Compute Principal Component Analysis at multiple distance thresholds}
\usage{
pca_multithreshold(
  distance.matrix = NULL,
  distance.thresholds = NULL,
  max.spatial.predictors = NULL
)
}
\arguments{
\item{distance.matrix}{Numeric distance matrix between observations.}

\item{distance.thresholds}{Numeric vector of distance thresholds defining different neighborhood scales. Each threshold specifies the maximum distance for spatial neighbors at that scale. If \code{NULL}, automatically computed with \code{\link[=default_distance_thresholds]{default_distance_thresholds()}}. Default: \code{NULL}.}

\item{max.spatial.predictors}{Integer specifying the maximum number of spatial predictors to retain. If the total number of generated predictors exceeds this value, only the first \code{max.spatial.predictors} are kept (ordered by variance explained). Useful for managing memory when \code{distance.matrix} is very large. Default: \code{NULL} (keeps all predictors).}
}
\value{
Data frame where each column is a spatial predictor derived from PCA at a specific distance threshold. Columns are named with the pattern \verb{spatial_predictor_<distance>_<number>} (e.g., "spatial_predictor_1000_1", "spatial_predictor_5000_2"), where \verb{<distance>} is the distance threshold and \verb{<number>} is the principal component rank. The number of rows matches the number of observations in \code{distance.matrix}.
}
\description{
Computes principal components of a distance matrix at multiple distance thresholds to generate multi-scale spatial predictors for \code{\link[=rf_spatial]{rf_spatial()}}. Each distance threshold defines a different neighborhood scale, and PCA is applied to the weighted distance matrix at each scale.
}
\details{
This function generates multi-scale spatial predictors by applying PCA to distance matrices at different neighborhood scales. The process for each distance threshold:
\enumerate{
\item Converts the distance matrix to weights using \code{\link[=weights_from_distance_matrix]{weights_from_distance_matrix()}}, where distances above the threshold are set to zero
\item Applies \code{\link[=pca]{pca()}} to the weighted distance matrix to extract principal components
\item Names the resulting predictors with the distance threshold for identification
\item Filters out predictors with all near-zero values
}

\strong{Multi-scale spatial modeling:}

Different distance thresholds capture spatial patterns at different scales. Combining predictors from multiple thresholds allows \code{\link[=rf_spatial]{rf_spatial()}} to account for spatial autocorrelation operating at multiple spatial scales simultaneously. This is analogous to \code{\link[=mem_multithreshold]{mem_multithreshold()}} but uses PCA instead of Moran's Eigenvector Maps.

\strong{Comparison with MEMs:}

Both \code{\link[=pca_multithreshold]{pca_multithreshold()}} and \code{\link[=mem_multithreshold]{mem_multithreshold()}} generate spatial predictors from distance matrices, but differ in their approach:
\itemize{
\item \strong{PCA}: Captures the main patterns of variation in the weighted distance matrix without considering spatial autocorrelation structure
\item \strong{MEMs}: Explicitly extracts spatial patterns with specific autocorrelation scales (positive and negative eigenvalues)
}

In practice, MEMs are generally preferred for spatial modeling because they explicitly target spatial autocorrelation patterns, but PCA can serve as a simpler alternative or for comparison.
}
\examples{

data(plants_distance)

# Compute PCA spatial predictors at multiple distance thresholds
pca_predictors <- pca_multithreshold(
  distance.matrix = plants_distance,
  distance.thresholds = c(0, 1000, 5000)
)

# View structure
head(pca_predictors)
dim(pca_predictors)

# Check predictor names (show scale information)
colnames(pca_predictors)[1:6]

# Limit number of predictors to save memory
pca_limited <- pca_multithreshold(
  distance.matrix = plants_distance,
  distance.thresholds = c(0, 1000, 5000),
  max.spatial.predictors = 20
)
ncol(pca_limited)  # At most 20 predictors

}
\seealso{
\code{\link[=pca]{pca()}}, \code{\link[=mem_multithreshold]{mem_multithreshold()}}, \code{\link[=weights_from_distance_matrix]{weights_from_distance_matrix()}}, \code{\link[=default_distance_thresholds]{default_distance_thresholds()}}

Other spatial_analysis: 
\code{\link{filter_spatial_predictors}()},
\code{\link{mem}()},
\code{\link{mem_multithreshold}()},
\code{\link{moran}()},
\code{\link{moran_multithreshold}()},
\code{\link{pca}()},
\code{\link{rank_spatial_predictors}()},
\code{\link{residuals_diagnostics}()},
\code{\link{residuals_test}()},
\code{\link{select_spatial_predictors_recursive}()},
\code{\link{select_spatial_predictors_sequential}()}
}
\concept{spatial_analysis}
