\name{look_up}
\alias{look_up}
\alias{print.look_up}
\title{Look Up Sumerian Signs or Search for Translations}
\description{
Searches a Sumerian dictionary either by sign name (forward lookup) or by
translation text (reverse lookup).

The function \code{look_up} computes the search results and returns an object of class \code{"look_up"}. The \code{print} method displays formatted results with cuneiform representations, grammatical types, and translation counts.
}
\usage{
look_up(x, dic, lang = "sumer", width = 70)

\method{print}{look_up}(x, ...)
}
\arguments{
\item{x}{For \code{look_up}: A character string specifying the search term. Can be either:
\itemize{
\item A Sumerian sign name (e.g., \code{"AN"}, \code{"AN.EN.ZU"})
\item A cuneiform character string
\item A word or phrase to search in translations (e.g., \code{"Gilgamesh"}, \code{"heaven"})
}

For \code{print.look_up}: An object of class \code{"look_up"} as returned by \code{look_up}.}

\item{dic}{A dictionary data frame, typically created by
\code{\link{make_dictionary}} or loaded with \code{\link{read_dictionary}}.
Must contain columns \code{sign_name}, \code{row_type}, \code{count},
\code{type}, and \code{meaning}.}
\item{lang}{Character string specifying whether \code{x} is a Sumerian expression (\code{"sumer"}) or an English expression (\code{"en"}).}
\item{width}{Integer specifying the text width for line wrapping. Default is 70.}
\item{...}{Additional arguments passed to the print method (currently unused).}
}
\value{
\code{look_up} returns an object of class \code{"look_up"}, which is a list containing:
\item{search}{The original search term.}
\item{lang}{The language setting used for the search.}
\item{width}{The text width for formatting.}
\item{cuneiform}{The cuneiform representation (only for Sumerian searches).}
\item{sign_name}{The canonical sign name (only for Sumerian searches).}
\item{translations}{A data frame with translations for the exact sign combination (only for Sumerian searches).}
\item{substrings}{A named list of data frames with translations for individual signs and substrings (only for Sumerian searches).}
\item{matches}{A data frame with matching entries (only for non-Sumerian searches).}

\code{print.look_up} prints formatted dictionary entries to the console and returns \code{x} invisibly.
}
\details{
\subsection{Search Modes}{
The function operates in two modes depending on the input:

\strong{Forward Lookup} (Sumerian input detected):
\enumerate{
\item Converts the sign name to cuneiform
\item Retrieves all translations for the exact sign combination
\item Retrieves translations for all individual signs and substrings
}

\strong{Reverse Lookup} (non-Sumerian input):
\enumerate{
\item Searches for the term in all translation meanings
\item Retrieves matching entries with sign names and cuneiform
}
}

\subsection{Output Format}{
The print method displays results with:
\itemize{
\item Sign names with cuneiform representations
\item Occurrence counts in brackets (e.g., \code{[29]})
\item Grammatical type abbreviations (e.g., \code{S}, \code{V})
\item Translation meanings with automatic line wrapping
\item Search term highlighting in blue for reverse lookups (only for ANSI-compatible terminals)
}
}


}
\seealso{
\code{\link{read_dictionary}} for loading dictionaries,
\code{\link{make_dictionary}} for creating dictionaries,
\code{\link{as.cuneiform}} for cuneiform conversion.
}
\examples{
# Load dictionary
dic <- read_dictionary()

# Forward lookup: search by phonetic spelling
look_up("d-suen", dic)

# Forward lookup: search by Sumerian sign name
look_up("AN", dic)
look_up("AN.EN.ZU", dic)

# Forward lookup: search by cuneiform character string
AN.NA <- paste0(intToUtf8(0x1202D), intToUtf8(0x1223E))
AN.NA
look_up(AN.NA, dic)

# Reverse lookup: search in translations
look_up("Gilgamesh", dic, "en")

# Adjust output width for narrow terminals
look_up("water", dic, "en", width = 50)

# Store results for later use
result <- look_up("lugal", dic)
result$cuneiform
result$translations

# Print stored results
print(result)
}
\keyword{utilities}
\keyword{database}
