% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{trace}
\alias{trace}
\title{Main function for sample processing}
\usage{
trace(
  fragments_list,
  metadata_data.frame = NULL,
  index_override_dataframe = NULL,
  ladder_df_list = NULL,
  config_file = NULL,
  ...
)
}
\arguments{
\item{fragments_list}{A list of fragments objects containing fragment data, generated with either \code{\link[=read_fsa]{read_fsa()}}, \code{\link[=size_table_to_fragments]{size_table_to_fragments()}}, \code{\link[=genemapper_table_to_fragments]{genemapper_table_to_fragments()}}, or \code{\link[=repeat_table_to_fragments]{repeat_table_to_fragments()}}.}

\item{metadata_data.frame}{metadata passed to \code{\link[=add_metadata]{add_metadata()}} for grouping samples for metrics calculations or batch correction.}

\item{index_override_dataframe}{A data.frame to manually set index peaks. Column 1: unique sample IDs, Column 2: desired index peaks (the order of the columns is important since the information is pulled by column position rather than column name). Closest peak in each sample is selected so the number needs to just be approximate. Default: \code{NULL}. See \code{\link[=assign_index_peaks]{assign_index_peaks()}}.}

\item{ladder_df_list}{A list of dataframes, with the names being the unique id and the value being a dataframe. The dataframe has two columns, size (indicating the bp of the standard) and scan (the scan value of the ladder peak). It's critical that the element name in the list is the unique id of the sample. Either manually figure out what scan the ladder peaks should be and generate the list, or use \code{\link[=fix_ladders_interactive]{fix_ladders_interactive()}} to interactively generate the ladder_df_list.}

\item{config_file}{The file path to a YAML file containing a list of parameters. This provides a central place to adjust parameters for the pipeline. Use the following command to make a copy of the default YAML file: \code{file.copy(system.file("extdata/trace_config.yaml", package = "trace"), ".")}. The YAML file does not have to contain a complete list of parameters. Parameters directly passed to this function via ... will overwrite values in the \code{config_file}.}

\item{...}{additional parameters from any of the functions in the pipeline detailed below may be passed to this function.These parameters are grouped by functionality:

\strong{Ladder-related parameters:}
\itemize{
\item \code{ladder_channel}: string, which channel in the fsa file contains the ladder signal. Default: \code{"DATA.105"}.
\item \code{signal_channel}: string, which channel in the fsa file contains the data signal. Default: \code{"DATA.1"}.
\item \code{ladder_sizes}: numeric vector, bp sizes of ladder used in fragment analysis. Default: \code{c(50, 75, 100, 139, 150, 160, 200, 250, 300, 340, 350, 400, 450, 490, 500)}.
\item \code{ladder_start_scan}: single numeric indicating the scan number to start looking for ladder peaks (only required when ladder signal does not have large spike at start). Usually this can be automatically found (when set to NA) through the detection of the large spike at the start of the signal. Default: \code{NA}.
\item \code{minimum_ladder_signal}: single numeric for minimum signal of peak from the raw signal. If not set, the ladder will be fit to a set of the tallest peaks in the ladder channel. Default: \code{NA}.
\item \code{ladder_assign_left_to_right}: single logical for if the ladder should be assigned from the smallest base pair size to largest (TRUE), or if the order should be reversed and assigned from largest to smallest (FALSE), which can be helpful since the end often has cleaner signal than the start. Default: \code{TRUE}.
\item \code{ladder_selection_window}: single numeric for the ladder assigning algorithm. We iterate through the scans in blocks and test their linear fit (We can assume that the ladder is linear over a short distance). This value defines how large that block of peaks should be. Larger values should be better because the fit is tested in greater context, but larger numbers will make the fit increasingly slower. Default: \code{5}.
\item \code{ladder_top_n_branching}: single numeric. The ladder assigning algorithm branches as it tests the various combinations. This value defines how many branches should be created. If the correct combination is not found, you could try increasing this value, but it will make it increasingly slower. Default: \code{5}.
\item \code{ladder_branching_r_squared_threshold}: single numeric. The branches of the ladder assigning algorithm are pruned by R-squared values above this threshold to discard fits that are not promising. If the correct combination is not found, you could try decreasing this value, but it will make it increasingly slower. Default: \code{0.99}.
\item \code{min_scan}: single numeric indicating the lower scan limit to filter out scans below. Default: \code{NA}.
\item \code{max_scan}: single numeric indicating the upper scan limit to filter out scans above. Default: \code{NA}.
\item \code{max_combinations}: single numeric indicating what is the maximum number of ladder combinations that should be tested. Default: \code{2500000}.
\item \code{warning_rsq_threshold}: single numeric for the value for which this function will warn you when parts of the ladder have R-squared values below the specified threshold. Default: \code{0.998}.
\item \code{show_progress_bar}: single logical for showing progress bar. Default: \code{TRUE}.
}

\strong{Peak-finding parameters:}
\itemize{
\item \code{smoothing_window}: Single numeric value for signal smoothing window size passed to pracma::savgol(). Default: \code{21}.
\item \code{minimum_peak_signal}: Single numeric value for the minimum peak signal for a valid peak. To have no minimum signal set as "-Inf". Default: \code{20}
\item \code{min_bp_size}: Single numeric value for minimum bp size of peaks to consider. Default: \code{200}.
\item \code{max_bp_size}: Single numeric value for maximum bp size of peaks to consider. Default: \code{1000}.
\item \code{peak_scan_ramp}: Single numeric value to indicate how many scans (increasing in signal) should be either side of the peak maxima. Default: \code{5}.
}

\strong{Allele-calling parameters:}
\itemize{
\item \code{number_of_alleles}: Number of alleles to be returned for each fragment. Must either be 1 or 2. Default: \code{1}.
\item \code{peak_region_size_gap_threshold}: Gap threshold for identifying peak regions. Default: \code{6}.
\item \code{peak_region_signal_threshold_multiplier}: Multiplier for the peak signal threshold. Default: \code{1}.
}

\strong{Repeat-calling parameters:}
\itemize{
\item \code{assay_size_without_repeat}: An integer specifying the assay size without repeat for repeat calling. Default: \code{87}.
\item \code{repeat_size}: An integer specifying the repeat size for repeat calling. Default: \code{3}.
\item \code{correction}: A character vector of either "batch" to carry out a batch correction from common samples across runs (known repeat length not required), or "repeat" to use samples with validated modal repeat lengths to correct the repeat length. Default: \code{"none"}.
\item \code{force_whole_repeat_units}: A logical value specifying if the peaks should be forced to be whole repeat units apart. Default: \code{FALSE}.
\item \code{force_repeat_pattern}: A logical value specifying if the peaks should be re-called to fit the specific repeat unit pattern. Default: \code{FALSE}.
\item \code{force_repeat_pattern_size_period}: A numeric value to set the peak periodicity bp size. Default: \code{2.79}.
\item \code{force_repeat_pattern_size_window}: A numeric value for the size window when assigning the peak. Default: \code{0.5}.
}

\strong{Index peak assignment parameters:}
\itemize{
\item \code{grouped}: Logical value indicating whether samples should be grouped to share a common index peak. Default: \code{FALSE}.
}}
}
\value{
A list of fragments objects ready for calculation of instability metrics using \code{\link[=calculate_instability_metrics]{calculate_instability_metrics()}}
}
\description{
The main function for the trace package that handles processing of samples through the pipeline ready for the calculation of repeat instability metrics.
}
\details{
This function processes samples through the full pipeline, applying the library of functions within this package. Parameters can be adjusted either by passing them directly to this function or by editing a configuration file and supplying it to \code{config_file}. Below is a breakdown of the pipeline stages and their key functionalities:

\strong{Ladder-related parameters:}
The ladder is used to calibrate the base pair (bp) sizes of fragments. The ladder peaks are identified in the ladder channel using \code{\link{find_ladders}}, and a generalized additive model (GAM) with cubic regression splines is used to fit the relationship between scan numbers and bp sizes. This allows for accurate interpolation of bp sizes for all scans. Manual inspection of ladder assignments is recommended to ensure correctness. If ladders are broken, first try and adjust parameters. In a last resort, ladders may be manually set by using the \code{ladder_df_list} parameter. To generate this list, use the helper shiny app \code{\link{fix_ladders_interactive}}.

\strong{Peak-finding parameters:}
Fragment peaks are identified in the continuous trace data using \code{\link{find_fragments}}. The signal is smoothed using a Savitzky-Golay filter, and peaks are detected based on their signal intensity and bp size. Parameters such as \code{min_bp_size} and \code{max_bp_size} allow filtering peaks outside the desired range, while \code{peak_scan_ramp} controls the number of scans around the peak maxima.

\strong{Allele-calling parameters:}
The main alleles within each fragment are identified by clustering peaks into "peak regions" using \code{\link{find_alleles}}. The tallest peak in each region is selected as the main allele. If \code{number_of_alleles} is set to 2, the two tallest peaks in their respective regions are selected, with the larger repeat size assigned as the main allele.

\strong{Repeat-calling parameters:}
Repeat lengths are calculated using \code{\link{call_repeats}}, based on the assay size without the repeat and the specified repeat size. Options include batch correction to account for run-to-run variability and repeat correction using samples with known modal repeat lengths. The \code{force_whole_repeat_units} option ensures repeat lengths are whole numbers, while \code{force_repeat_pattern} re-calls peaks to fit the expected repeat pattern.

\strong{Index peak assignment parameters:}
The index peak is the reference repeat length used for instability metrics calculations. It is typically the inherited repeat length or the modal repeat length at a starting time point. Samples can be grouped to share a common index peak using \code{\link{assign_index_peaks}}, or the index peak can be manually overridden using \code{index_override_dataframe}.

\strong{Metadata:}
Metadata is used to group samples for metrics calculations, batch correction, or repeat length validation. Use \code{\link{add_metadata}} to add metadata to the fragments list. Required (even if the column is left blank) metadata fields include:
\itemize{
\item \code{batch_run_id}: Groups samples by fragment analysis run.
\item \code{batch_sample_id}: Links samples across batches for batch or repeat correction.
\item \code{batch_sample_modal_repeat}: Specifies the validated repeat length for samples used in repeat correction.
\item \code{metrics_group_id}: Groups samples for shared index peak assignment.
\item \code{metrics_baseline_control}: Identifies samples used as baseline controls for index peak assignment.
}
}
\examples{
fsa_list <- lapply(cell_line_fsa_list, function(x) x$clone())

# import data with read_fsa() to generate an equivalent list to cell_line_fsa_list
fragments_list <- trace(fsa_list, grouped = TRUE, metadata_data.frame = metadata)

metrics <- calculate_instability_metrics(
  fragments_list,
  peak_threshold = 0.05,
  window_around_index_peak = c(-40, 40),
  percentile_range = c(0.5, 0.75, 0.9, 0.95),
  repeat_range = c(2, 5, 10, 20)
)
}
