

/*
Run-time CPU feature detection on AArch64 Apple targets by using sysctlbyname.

On macOS, this module is currently only enabled on tests because AArch64 macOS
always supports FEAT_LSE and FEAT_LSE2 (see build script for more).

If macOS supporting FEAT_LSE128/FEAT_LRCPC3 becomes popular in the future, this module will
be used to support outline-atomics for FEAT_LSE128/FEAT_LRCPC3.
M4 is Armv9.2 and it doesn't support FEAT_LSE128/FEAT_LRCPC3.

Refs: https:

TODO: non-macOS targets doesn't always supports FEAT_LSE2, but sysctl on them on the App Store is...?
- https:
- https:
- https:
*/

include!("common.rs");

use core::{mem, ptr};


mod ffi {
    pub(crate) use crate::utils::ffi::{CStr, c_char, c_int, c_size_t, c_void};

    sys_fn!({
        extern "C" {
            
            
            pub(crate) fn sysctlbyname(
                name: *const c_char,
                old_p: *mut c_void,
                old_len_p: *mut c_size_t,
                new_p: *mut c_void,
                new_len: c_size_t,
            ) -> c_int;
        }
    });
}

fn sysctlbyname32(name: &ffi::CStr) -> Option<u32> {
    const OUT_LEN: ffi::c_size_t = mem::size_of::<u32>() as ffi::c_size_t;

    let mut out = 0_u32;
    let mut out_len = OUT_LEN;
    
    
    
    
    let res = unsafe {
        ffi::sysctlbyname(
            name.as_ptr(),
            (&mut out as *mut u32).cast::<ffi::c_void>(),
            &mut out_len,
            ptr::null_mut(),
            0,
        )
    };
    if res != 0 {
        return None;
    }
    debug_assert_eq!(out_len, OUT_LEN);
    Some(out)
}

#[cold]
fn _detect(info: &mut CpuInfo) {
    
    
    
    
    
    if sysctlbyname32(c!("hw.optional.arm.FEAT_LSE")).unwrap_or(0) != 0
        || sysctlbyname32(c!("hw.optional.armv8_1_atomics")).unwrap_or(0) != 0
    {
        info.set(CpuInfo::HAS_LSE);
    }
    if sysctlbyname32(c!("hw.optional.arm.FEAT_LSE2")).unwrap_or(0) != 0 {
        info.set(CpuInfo::HAS_LSE2);
    }
    if sysctlbyname32(c!("hw.optional.arm.FEAT_LSE128")).unwrap_or(0) != 0 {
        info.set(CpuInfo::HAS_LSE128);
    }
    if sysctlbyname32(c!("hw.optional.arm.FEAT_LRCPC3")).unwrap_or(0) != 0 {
        info.set(CpuInfo::HAS_RCPC3);
    }
}

#[allow(
    clippy::alloc_instead_of_core,
    clippy::std_instead_of_alloc,
    clippy::std_instead_of_core,
    clippy::undocumented_unsafe_blocks,
    clippy::wildcard_imports
)]
#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_macos() {
        assert_eq!(sysctlbyname32(c!("hw.optional.armv8_1_atomics")), Some(1));
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LSE")), Some(1));
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LSE2")), Some(1));
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LSE128")), None);
        assert_eq!(std::io::Error::last_os_error().kind(), std::io::ErrorKind::NotFound);
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LRCPC")), Some(1));
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LRCPC2")), Some(1));
        assert_eq!(sysctlbyname32(c!("hw.optional.arm.FEAT_LRCPC3")), None);
        assert_eq!(std::io::Error::last_os_error().kind(), std::io::ErrorKind::NotFound);
    }

    #[cfg(target_pointer_width = "64")]
    #[test]
    fn test_alternative() {
        use crate::utils::ffi::*;
        #[cfg(not(portable_atomic_no_asm))]
        use std::arch::asm;
        use std::mem;
        use test_helper::sys;
        
        
        
        
        
        fn sysctlbyname32_no_libc(name: &CStr) -> Result<u32, c_int> {
            // https://github.com/apple-oss-distributions/xnu/blob/8d741a5de7ff4191bf97d57b9f54c2f6d4a15585/bsd/kern/syscalls.master#L298
            #[inline]
            unsafe fn sysctl(
                name: *const c_int,
                name_len: c_uint,
                old_p: *mut c_void,
                old_len_p: *mut c_size_t,
                new_p: *const c_void,
                new_len: c_size_t,
            ) -> Result<c_int, c_int> {
                // https://github.com/apple-oss-distributions/xnu/blob/8d741a5de7ff4191bf97d57b9f54c2f6d4a15585/osfmk/mach/i386/syscall_sw.h#L158
                #[inline]
                const fn syscall_construct_unix(n: u64) -> u64 {
                    const SYSCALL_CLASS_UNIX: u64 = 2;
                    const SYSCALL_CLASS_SHIFT: u64 = 24;
                    const SYSCALL_CLASS_MASK: u64 = 0xFF << SYSCALL_CLASS_SHIFT;
                    const SYSCALL_NUMBER_MASK: u64 = !SYSCALL_CLASS_MASK;
                    (SYSCALL_CLASS_UNIX << SYSCALL_CLASS_SHIFT) | (SYSCALL_NUMBER_MASK & n)
                }
                #[allow(clippy::cast_possible_truncation)]
                
                unsafe {
                    // https://github.com/apple-oss-distributions/xnu/blob/8d741a5de7ff4191bf97d57b9f54c2f6d4a15585/bsd/kern/syscalls.master#L4
                    let mut n = syscall_construct_unix(202);
                    let r: i64;
                    asm!(
                        "svc 0",
                        "b.cc 2f",
                        "mov x16, x0",
                        "mov x0, #-1",
                        "2:",
                        inout("x16") n,
                        inout("x0") ptr_reg!(name) => r,
                        inout("x1") name_len as u64 => _,
                        in("x2") ptr_reg!(old_p),
                        in("x3") ptr_reg!(old_len_p),
                        in("x4") ptr_reg!(new_p),
                        in("x5") new_len as u64,
                        options(nostack),
                    );
                    if r as c_int == -1 { Err(n as c_int) } else { Ok(r as c_int) }
                }
            }
            
            unsafe fn sysctlbyname(
                name: &CStr,
                old_p: *mut c_void,
                old_len_p: *mut c_size_t,
                new_p: *mut c_void,
                new_len: c_size_t,
            ) -> Result<c_int, c_int> {
                let mut real_oid: [c_int; sys::CTL_MAXNAME as usize + 2] = unsafe { mem::zeroed() };

                
                
                let mut name2oid_oid: [c_int; 2] = [0, 3];

                let mut oid_len = mem::size_of_val(&real_oid);
                unsafe {
                    sysctl(
                        name2oid_oid.as_mut_ptr(),
                        2,
                        real_oid.as_mut_ptr().cast::<c_void>(),
                        &mut oid_len,
                        name.as_ptr().cast::<c_void>() as *mut c_void,
                        name.to_bytes_with_nul().len() - 1,
                    )?;
                }
                oid_len /= mem::size_of::<c_int>();
                #[allow(clippy::cast_possible_truncation)]
                unsafe {
                    sysctl(real_oid.as_mut_ptr(), oid_len as u32, old_p, old_len_p, new_p, new_len)
                }
            }

            const OUT_LEN: ffi::c_size_t = mem::size_of::<u32>() as ffi::c_size_t;

            let mut out = 0_u32;
            let mut out_len = OUT_LEN;
            
            
            
            let res = unsafe {
                sysctlbyname(
                    name,
                    (&mut out as *mut u32).cast::<ffi::c_void>(),
                    &mut out_len,
                    ptr::null_mut(),
                    0,
                )?
            };
            debug_assert_eq!(res, 0);
            debug_assert_eq!(out_len, OUT_LEN);
            Ok(out)
        }

        for name in [
            c!("hw.optional.armv8_1_atomics"),
            c!("hw.optional.arm.FEAT_LSE"),
            c!("hw.optional.arm.FEAT_LSE2"),
            c!("hw.optional.arm.FEAT_LSE128"),
            c!("hw.optional.arm.FEAT_LRCPC"),
            c!("hw.optional.arm.FEAT_LRCPC2"),
            c!("hw.optional.arm.FEAT_LRCPC3"),
        ] {
            if let Some(res) = sysctlbyname32(name) {
                assert_eq!(res, sysctlbyname32_no_libc(name).unwrap());
            } else {
                assert_eq!(sysctlbyname32_no_libc(name).unwrap_err(), libc::ENOENT);
            }
        }
    }
}
